#!/bin/bash
#
# IceWarp Server installation script
# Copyright (c) 2008-2025 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com 
#
# file: install.sh - install script
# version: 1.5
#

# variables
EMPTY_VALUE="UNKNOWN"

USE_COLORS="0"

IMAGENAME="IceWarpServer-image-14.2.0.10_RHEL8_x64.tar"

SETUP_NEWVERSION="14.2.0.10 RHEL8 x64"
SETUP_NICE_NEWVERSION="Epos Update 2"
SETUP_OLDVERSION="$EMPTY_VALUE"

SETUP_SERVER_PLATFORM="x86_64"
SETUP_SERVER_OLDPLATFORM="${EMPTY_VALUE}"

SETUP_SERVER_LIBDIR="lib64"

SETUP_INSTALL_DIR="$EMPTY_VALUE"
SETUP_INSTALL_DIR_DEFAULT="/opt/icewarp"

SETUP_CONFIG_DIR=""
SETUP_CONFIG_DIR_DEFAULT="config"

SETUP_CALENDAR_DIR=""
SETUP_CALENDAR_DIR_DEFAULT="calendar"

SETUP_SPAM_DIR=""
SETUP_SPAM_DIR_DEFAULT="spam"

SETUP_INSTALL_USER="$EMPTY_VALUE"
SETUP_INSTALL_USER_DEFAULT=$(id -un)

SETUP_INSTALL_SERVICE_NAME="icewarp"

SETUP_SERVER_CONF="$EMPTY_VALUE"
SETUP_SERVER_CONF_EXISTING="$EMPTY_VALUE"

SETUP_CONFIG_FILE_EXISTS="0"
SETUP_YODA_ENABLED="0"

MINIMAL_REQUIRED_WEBDOCUMENTS_VERSION="5.1.2.9"
WEBDOCUMENTS_UPDATE_URL="https://www.icewarp.com/downloads/icewarp-webdocuments/"
WEBDOCUMENTS_VERSION=""

SUDO_REQUIRETTY_IS_SET=0

INSTALL_LOG=~/icewarp-install.log
INSTALL_ERROR_LOG=~/icewarp-install-error.log

DASHBOARD_MINIMAL_REQUIRED_VERSION="14"
DASHBOARD_API_CHECK_URL="https://api.icewarp.com/v1/service/check_connection"

MINIMAL_VERSION_NEW_CIPHER="15"
LICENSE_PATH=""
REPLACE_TOOL_WITH_LICCHECK="0"

RUNNING_UID=$(id -u)

UPGRADE="0"
KILL_SERVICES="0"
CONVERSION_DOC_PDF="1"
CONVERSION_PDF_IMAGE="1"

RESTART_RECOMMENDED="0"

OS_DIST="$EMPTY_VALUE"
OS_PLATFORM="$EMPTY_VALUE"

POSTMASTER_ALIASES="postmaster;admin;administrator;supervisor;hostmaster;webmaster;abuse"

HELPER_WIZARD_FILE="/tmp/iwwizlog"

# cmdline flags
OPT_FAST_INSTALL_MODE="0"
OPT_ALLOW_MISSING_ARCHITECTURE="0"
OPT_AUTO="0"
OPT_INSTALL_DIR=""
OPT_USER=""
OPT_LICENSE=""
OPT_STARTONBOOT=""
OPT_DOCKER="0"
OPT_CHECK_DASHBOARD_API='1'
OPT_CHECK_FFMPEG='1'
OPT_CHECK_DASHBOARD_API_ONLY='0'

# platform specific declarations
source $(dirname $0)"/platform"

# init

SELF=$0
WORKDIR=$(dirname $0)
LIBLIST32=""
LIBLIST64=""

case "$TERM" in
xterm-256color|xterm|xterm-color)
  GOOD=$'\e[38;5;2m'
  BAD=$'\e[38;5;1m'
  NORMAL=$'\e[0m'
  WARN=$'\e[38;5;11m'
  HILITE=$'\e[40;38;5;15m'
  BRACKET=$'\e[40;38;5;8m'
  ;;
  linux)
  GOOD=$'\e[0;32m'
  BAD=$'\e[0;31m'
  NORMAL=$'\e[0m'
  WARN=$'\e[1;33m'
  HILITE=$'\e[1;37m'
  BRACKET=$'\e[1;30m'
    ;;
  *)
  # https://en.wikipedia.org/wiki/ANSI_escape_code
  GOOD=$'\e[0;32m'
  BAD=$'\e[0;31m'
  NORMAL=$'\e[0m'
  WARN=$'\e[0;33m'
  HILITE=${NORMAL}
  BRACKET=${NORMAL}
esac

good()
{
    echo -e "[$(date)] ${GOOD}**${NORMAL}\t$*"
}

bad()
{
    echo -e "[$(date)] ${BAD}**${NORMAL}\tError: $*"
}

warn()
{
    echo -e "[$(date)] ${WARN}**${NORMAL}\tWarning: $*"
}

hilite()
{
    echo -e "[$(date)] ${GOOD}**\t${HILITE}$*${NORMAL}"
}

multi_good()
{
    echo "$1" | while read LINE; do
        good "$LINE"
    done
}

clear_stdin()
{
    while read -t 0.1; do : ; done
}

getparam()
{
    echo -e -n "[$(date)] ${GOOD}**${NORMAL}\t$1 [$2]${NORMAL}: "
    clear_stdin
    read PARAM
}

getpassword()
{
    echo -e -n "[$(date)] ${GOOD}**${NORMAL}\t$1${NORMAL}: "
    clear_stdin
    read -s PARAM
    echo ""
}

copy_if_not_exists()
{
    if ! [ -f "$2" ]; then
        cp -f "$1" "$2"
    fi
}

log_install_error()
{
    echo $1 >> "$INSTALL_ERROR_LOG"
}

# calls tool and returns single value of single api variable
# IWS_INSTALL_DIR have to be set
# params: object variable
get_liccheck_variable()
{
    LICCHECK_OUT=$("${WORKDIR}/liccheck.sh" get "$1" "$2")
    CMD_RET=$?
    CMD_STDOUT=$(echo $LICCHECK_OUT | sed 's/^[^:]*: //')
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

set_liccheck_variable()
{
    CMD_STDOUT=$("${WORKDIR}/liccheck.sh" set "$1" "$2" "$3")
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

run_liccheck_command()
{
    CMD_STDOUT=$("${WORKDIR}/liccheck.sh" "$1" "$2" "$3")
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

get_api_variable()
{
    if [ "${REPLACE_TOOL_WITH_LICCHECK}" == "1" ]; then
        API_OUT=$("${WORKDIR}/liccheck.sh" get "$1" "$2")
    else
        API_OUT=$("${SETUP_INSTALL_DIR}/tool.sh" get "$1" "$2")
    fi
    
    CMD_RET=$?
    if [ $CMD_RET -eq 0 ]; then
        GET_API_VARIABLE=$(sed '1 s/^[^:]*: //' <<< "$API_OUT" | tr -d '\r')
    else
        GET_API_VARIABLE="Cannot get API variable $2 ($API_OUT)"
    fi

    return ${CMD_RET}
}

set_api_variable()
{
    if [ "${REPLACE_TOOL_WITH_LICCHECK}" == "1" ]; then
        API_OUT=$("${WORKDIR}/liccheck.sh" set "$1" "$2" "$3")
    else
        API_OUT=$("${SETUP_INSTALL_DIR}/tool.sh" set "$1" "$2" "$3")
    fi

    CMD_RET=$?
    if [ $CMD_RET -ne 0 ]; then
        SET_API_VARIABLE="$CMD_STDOUT"
    else
        SET_API_VARIABLE=""
    fi
    return ${CMD_RET}
}

# $1 - version (number only)
# $2 - respect build number flag, 0 == true
version_to_num()
{
    NUM=$(cut -f1 -d. <<< $1)
    if [ "x${NUM}" == "x" ]; then
        NUM=0
    fi
    MAJOR=$(($NUM * 1000000000000))

    NUM=$(cut -f2 -d. <<< $1)
    if [ "x${NUM}" == "x" ]; then
        NUM=0
    fi
    MIDDLE=$(($NUM * 10000000000))

    NUM=$(cut -f3 -d. <<< $1)
    NUM1=$(cut -f1 -d- <<< $NUM)
    if [ "x${NUM1}" == "x" ]; then
        NUM1=0
    fi
    MINOR1=$(($NUM1 * 100000000))

    NUM2=$(cut -f2 -d- <<< $NUM)
    if [ "x${NUM2}" == "x" ]; then
        NUM2=0
    fi
    MINOR2=$(($NUM2 * 1000000))

    if $2; then
        NUM=$(cut -f4 -d. <<< $1)
        if [ "x${NUM}" == "x" ]; then
            NUM=0
        fi
        BUILD=$NUM
    else
        BUILD=0
    fi
    
    RESULT=$(($MAJOR + $MIDDLE + $MINOR1 + $MINOR2 + $BUILD))
}

# IceWarp version string comparator
# $1, $2 version
# $3 - compareNB, 0 == true
# $4 - RespectBuildNumber, 0 == true
# returns: negative - $1 > $2, 0 - $1 == $2, positive - $1 < $2 
compare_version()
{
    VERSION_1="$1"
    VERSION_2="$2"
    COMPARE_NB=$3
    RESPECT_BUILD_NUMBER=$4

    # At first compare version numbers
    PURE_VERSION_1=$(cut -f1 -d' ' <<< $VERSION_1)
    version_to_num $PURE_VERSION_1 $RESPECT_BUILD_NUMBER
    NUM_VERSION_1=$RESULT

    PURE_VERSION_2=$(cut -f1 -d' ' <<< $VERSION_2)
    version_to_num $PURE_VERSION_2 $RESPECT_BUILD_NUMBER
    NUM_VERSION_2=$RESULT

    if $COMPARE_NB && [ $NUM_VERSION_1 = $NUM_VERSION_2 ]; then
        # Compare NB, RC
        # Get NB part
        NB_VERSION_1=$(cut -f2 -d' ' <<< $VERSION_1)
        NB_VERSION_2=$(cut -f2 -d' ' <<< $VERSION_2)
        
        # Check if RC
        RC=$(echo ${NB_VERSION_1:0:2} | tr 'a-z' 'A-Z')
        if [ "${RC}" == "RC" ]; then
            RC_1=true
        else
            RC_1=false
        fi

        RC=$(echo ${NB_VERSION_2:0:2} | tr 'a-z' 'A-Z')
        if [ "${RC}" == "RC" ]; then
            RC_2=true
        else
            RC_2=false
        fi

        # Check if NB
        if [ ${NB_VERSION_1:0:1} == "(" ]; then
            NB_1=true
        else
            NB_1=false
        fi

        if [ ${NB_VERSION_2:0:1} == "(" ]; then
            NB_2=true
        else
            NB_2=false
        fi

        if ! $RC_1 && ! $NB_1; then
            NB_VERSION_1=""
        fi

        if ! $RC_2 && ! $NB_2; then
            NB_VERSION_2=""
        fi

        # evaluate
        if [ "x${NB_VERSION_1}" == "x" ]; then
            if [ "x${NB_VERSION_2}" == "x" ]; then
                # Both wihtout RC or NB
                RESULT=0
            else
                # 1 is release, 2 NB or RC
                RESULT=-1
            fi
        elif [ "x${NB_VERSION_2}" == "x" ]; then
            # 1 is NB or RC, 2 is release
            RESULT=1
        else
            # Both are NB or RC
            if [ $RC_1 != $RC_2 ]; then
                # RC is greater than NB
                if $RC_1; then
                    RESULT=-1
                else
                    RESULT=1
                fi
            else
                # either both NB, or Both RC
                if $NB_1; then
                    # both NB, use lexicographical sorting
                    if [ "${NB_VERSION_1}" == "${NB_VERSION_2}" ]; then
                        RESULT=0
                    elif [[ "${NB_VERSION_1}" < "${NB_VERSION_2}" ]]; then
                        RESULT=1
                    else
                        RESULT=-1
                    fi
                else
                    # Both RC, we have to parse exactly the rc number
                    RC_NUM_1=${NB_VERSION_1:2}
                    RC_NUM_2=${NB_VERSION_2:2}

                    RESULT=$(($RC_NUM_2 - $RC_NUM_1))
                fi
            fi
        fi
    else
        RESULT=$(($NUM_VERSION_2 - $NUM_VERSION_1))
    fi
}

# Scans system for available shared libraries
build_dynamic_modules_list()
{
    # Refresh ld cache
    ldconfig
    LIBLIST32=""
    LIBLIST64=""
    LIBLIST32=$(ldconfig -p | grep libc6 | grep -v "x86-64" | sed 's/[[:space:]]*//' | sed 's/ (.*//' | while read LINE; do
        echo -n " ${LINE}"
    done)

    if [ "${SETUP_SERVER_PLATFORM}" == "x86_64" ]; then
        LIBLIST64=$(ldconfig -p | grep "x86-64" | sed 's/[[:space:]]*//' | sed 's/ (.*//' | while read LINE; do
           echo -n " ${LINE}"
        done)
    fi
}

# Checks, if given dynamic 32-bit library is available on system
# Params: filename
check_dynamic_module_32()
{
    grep "$1" <<< "${LIBLIST32}" &>/dev/null
}

# Checks, if given dynamic 64-bit library is available on system
# Params: filename
check_dynamic_module_64()
{
    grep "$1" <<< "${LIBLIST64}" &>/dev/null
}

# Sets variables depending on command line flags
parse_cmdline()
{
    while [ "x$1" != "x" ]; do
        case `echo $1 | tr a-z A-Z` in
            -F|--FAST)
                OPT_FAST_INSTALL_MODE="1"
                ;;
            --ALLOW-MISSING-ARCHITECTURE)
                OPT_ALLOW_MISSING_ARCHITECTURE="1"
                ;;
            -A|--AUTO)
                OPT_AUTO="1"
                ;;
            -D|--DOCKER)
                OPT_DOCKER="1"
                ;;
            --INSTALL-DIR)
                OPT_INSTALL_DIR="$2"
                shift
                ;;
            --USER)
                OPT_USER="$2"
                shift
                ;;
            --LICENSE)
                OPT_LICENSE="$2"
                shift
                ;;
            --START-ON-BOOT)
                OPT_STARTONBOOT="$2"
                shift
                ;;
            --SKIP-API-CHECK)
                OPT_CHECK_DASHBOARD_API="0"
                ;;
            --API-CHECK)
                OPT_CHECK_DASHBOARD_API_ONLY="1"
                OPT_AUTO="1"
                ;;
            --SKIP-FFMPEG-CHECK)
                OPT_CHECK_FFMPEG="0"
                ;;
        esac
        shift
    done
}

compare_generic_versions()
{
   [ "$1" == "$2" ] && return "10"

   ver1front=`echo $1 | cut -d "." -f -1`
   ver1back=`echo $1 | cut -d "." -f 2-`

   ver2front=`echo $2 | cut -d "." -f -1`
   ver2back=`echo $2 | cut -d "." -f 2-`

   if [ "$ver1front" != "$1" ] || [ "$ver2front" != "$2" ]; then
       [ "$ver1front" -gt "$ver2front" ] && return "11"
       [ "$ver1front" -lt "$ver2front" ] && return "9"

       [ "$ver1front" == "$1" ] || [ -z "$ver1back" ] && ver1back=0
       [ "$ver2front" == "$2" ] || [ -z "$ver2back" ] && ver2back=0
       compare_generic_versions "$ver1back" "$ver2back"
       return $?
   else
           [ "$1" -gt "$2" ] && return "11" || return "9"
   fi
}    

# Checks, if OSDIST was detected as expected
test_correct_osdist()
{
    MATCH=false
    for DISTRO in "${PLATFORM_OSDIST[@]}"; do
        if [ "${DISTRO}" == "${OS_DIST}" ]; then
            MATCH=true
            break
        fi
    done

    if ! $MATCH; then
        bad "Incompatible Linux distribution"
        bad "This install script have to be run on ${PLATFORM_NAME}"
        exit 1
    fi
}

test_fonts()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    FONT_PACKAGES=""

    for item in ${PLATFORM_TRUETYPE_FONTS[*]}
    do
        platform_is_package_installed $item
        
        if [ $? == "1" ]; then
          FONT_PACKAGES=$FONT_PACKAGES" "$item
        fi
    done

    if [ "$FONT_PACKAGES" != "" ]; then
        if [ "${OPT_AUTO}" != "1" ]; then
            ask_with_confirmation "TrueType fonts are required for document preview. Install them now?" "Y" "n"
            
            if [ "$?" -eq 1 ]; then
                platform_install_packages "0" $FONT_PACKAGES
            fi
        else
            good "Installing TrueType fonts and ffmpeg for document preview ..."
            platform_install_packages "1" $FONT_PACKAGES
        fi
    fi
}


# Checks, if install script runs on correct platform and version
test_correct_platform()
{
    WARN_LEVEL=0

    if platform_get_distro_id; then
        MATCH=false
        for DISTRO in "${PLATFORM_DISTRO_IDS[@]}"; do
            if [ "${DISTRO}" == "${PLATFORM_RESULT}" ]; then
                MATCH=true
                break
            fi
        done

        if ! $MATCH; then
            WARN_LEVEL=2
        fi
    else
        WARN_LEVEL=1
    fi

    if [ $WARN_LEVEL -eq 0 ]; then
        if platform_get_distro_major_version; then
            MATCH=false
            for VERSION in "${PLATFORM_VERSIONS[@]}"; do
                if [ "${VERSION}" == "${PLATFORM_RESULT}" ]; then
                    MATCH=true
                    break
                fi
            done

            if ! $MATCH; then
                WARN_LEVEL=2
            fi
        else
            WARN_LEVEL=1
        fi
    fi
    
    if [ $WARN_LEVEL -eq 0 ]; then
        if platform_get_hw_platform; then
            MATCH=false
            for PLATFORM in "${PLATFORM_HW_PLATFORMS[@]}"; do
                if [ "${PLATFORM}" == "${PLATFORM_RESULT}" ]; then
                    MATCH=true
                    break
                fi
            done

            if ! $MATCH; then
                WARN_LEVEL=3
            fi
        else
            WARN_LEVEL=1
        fi
    fi

    case $WARN_LEVEL in
        1)  warn "Cannot check, if this package is compatible with the system"
            if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                ask_with_confirmation "Are you sure you are installing on ${PLATFORM_NAME}?" "N" "y"
                if [ $? -ne 2 ]; then
                    bad "Installation aborted on user request."
                    exit 1
                fi
            else
                echo "Automatic installation mode - expecting that package is compatible"
            fi
            ;;
        2)  bad "Incompatible Linux distribution"
            bad "This install script has to be run on ${PLATFORM_NAME}"
            exit 1 
            ;;
        3)  bad "Incompatible Linux distribution"
            bad "This install script has to be run on ${PLATFORM_NAME}"
            bad "You are probably trying to install 64bit IceWarp Server on 32bit machine."
            exit 1 
    esac
}

# Checks if selinux is active, if so, warns user
test_selinux()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    SELINUX=$(getenforce 2>/dev/null)
    if grep "Enforcing" <<< ${SELINUX} &>/dev/null; then
        echo ""
        warn "SELinux is in enforcing mode on this system.
                 Please put SELinux into permissive or disabled mode
                 or follow the guide in Icewarp Installation and Control on Linux document.

                 Press ENTER to continue or Ctrl-C to exit setup"
        if [ "${OPT_AUTO}" != "1" ]; then
            read -s
            echo ""
        else
            echo "Automatic installation mode - ignoring SELinux warning"
        fi
    fi
}

# Does some platform specific checks, like RHEL6 pam module being up to date
test_platform_specifics()
{
    if ! platform_do_specific_checks; then
        exit 1
    fi
}

# Tests for dependencies, offers its installation
test_dependencies()
{
    MISSING_LIBRARIES32=""
    MISSING_LIBRARIES64=""
    MISSING_PACKAGES=""

    MISSING_PACKAGES="x"
    FIRST_PASS=0
    while [ "x${MISSING_PACKAGES}" != "x" ]; do
        MISSING_LIBRARIES32=""
        MISSING_LIBRARIES64=""
        MISSING_PACKAGES=""
        good "Checking dynamic library dependencies..."

        # Check for dependencies
        build_dynamic_modules_list

        # 32 bit libraries
        for I in $(seq 1 ${#PLATFORM_DEPENDENCIES32[*]}); do
            LIBRARY=${PLATFORM_DEPENDENCIES32[$I]}
            PACKAGE=${PLATFORM_PACKAGES32[$I]}
            # Obtain package for 32/64 bit platform if differs
            if grep "/" <<< "${PACKAGE}" &>/dev/null; then
                if [ "${OS_PLATFORM}" == "x86_64" ]; then
                    PACKAGE=$(sed 's:^.*/::' <<< "${PACKAGE}")
                else
                    PACKAGE=$(sed 's:/.*$::' <<< "${PACKAGE}")
                fi
            fi

            # Some 32 bit packages on 64 bit system are unavailable - they are distributed with IWS
            if [ "x${PACKAGE}" != "x" ]; then
                if ! check_dynamic_module_32 "${LIBRARY}"; then
                    MISSING_LIBRARIES32="${MISSING_LIBRARIES32} ${LIBRARY}"
                    MISSING_PACKAGES="${MISSING_PACKAGES}\n${PACKAGE}"
                fi
            fi
        done

        if [ "${SETUP_SERVER_PLATFORM}" == "x86_64" ]; then
            # 64 bit libraries
            for I in $(seq 1 ${#PLATFORM_DEPENDENCIES64[*]}); do
                LIBRARY=${PLATFORM_DEPENDENCIES64[$I]}
                PACKAGE=${PLATFORM_PACKAGES64[$I]}
                if ! check_dynamic_module_64 "${LIBRARY}"; then
                    MISSING_LIBRARIES64="${MISSING_LIBRARIES64} ${LIBRARY}"
                    MISSING_PACKAGES="${MISSING_PACKAGES}\n${PACKAGE}"
                fi
            done
        fi

        # Check ca-certificates bundle
        platform_ca_bundle_path
        CABUNDLE_PATH="$PLATFORM_RESULT"
        if ! [ -f "$CABUNDLE_PATH" ]; then
            platform_ca_bundle_package
            CABUNDLE_PACKAGE="$PLATFORM_RESULT"
            MISSING_PACKAGES="${MISSING_PACKAGES}\n${CABUNDLE_PACKAGE}"
        fi

        # Install missing packages
        if [ "x${MISSING_PACKAGES}" != "x" ]; then
            # Warn if this is not first pass - something went wrong with installation
            if [ ${FIRST_PASS} -ne 0 ]; then
                echo ""
                warn "All required dependencies are still not satisfied,
             probably the installation of them failed or was interrupted.
             
             Updating your system may help.
             
             Press ENTER to continue or Ctrl-C to exit setup"
                if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                    read -s
                    echo ""
                else
                    bad "Automatic installation mode - dependencies not installed, exiting"
                    exit 2
                fi
            fi

            FIRST_PASS=1
            # Strip duplicates, convert to space separated list
            MISSING_PACKAGES=$(echo -e ${MISSING_PACKAGES} | sort | uniq | tr '\n' ' ' | sed 's/[[:space:]]\+/ /g')

            warn "Some of the libraries required by IceWarp server were not found on this system."
            echo ""
            if [ "x${MISSING_LIBRARIES32}" != "x" ]; then
                echo "        Missing 32-bit libraries:${MISSING_LIBRARIES32}"
                echo ""
            fi
            if [ "x${MISSING_LIBRARIES64}" != "x" ]; then
                echo "        Missing 64-bit libraries:${MISSING_LIBRARIES64}"
                echo ""
            fi
            echo "        These packages should be installed to satisfy the dependencies:
       ${MISSING_PACKAGES}"
            echo ""
            if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                ask_with_confirmation "Do you want to install these packages into the system?" "Y" "n"
                if [ $? -eq 1 ]; then
                    platform_install_packages "0" $MISSING_PACKAGES 
                else
                    echo ""
                    warn "IceWarp server won't work without these libraries
                 Press ENTER to continue or Ctrl-C to exit setup"
                    read -s
                    echo ""
                    break
                fi
            else
                good "Automatic installation mode - installing missing dependencies"
                platform_install_packages "1" $MISSING_PACKAGES 
            fi
        fi
        echo ""
    done
}

# Tests if programs needed to install
test_program_servicemanagement()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    while ! platform_check_service_management_tool; do
        bad "Program ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME} seems not to be installed."
        if platform_is_install_service_management_tool_supported; then
            if [ "${OPT_AUTO}" != "1" ]; then
                ask_with_confirmation "Do you want to install it now into the system?" "Y" "n"
                if [ $? -eq 1 ]; then
                    platform_install_service_management_tool "0"
                else
                    bad "IceWarp server cannot be installed without ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME}"
                    exit 1
                fi
            else
                good "Automatic installation mode - installing ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME}"
                platform_install_service_management_tool "1"

                if ! platform_check_service_management_tool; then
                    bad "Automatic installation mode - cannot install ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME}"
                    exit 2
                fi
            fi
        else
            bad "Please install ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME} manually, it cannot be done by this script"
            exit 1
        fi
    done
}

test_program_sed()
{
    sed --version &> /dev/null
    if [ $? -ne 0 ]; then
        bad "Error testing sed. Program sed seems not to be installed."
        exit 1
    fi
}

test_program_unzip()
{
    # check unzip
    unzip -h &> /dev/null
    while [ $? -ne 0 ]; do
        bad "Program unzip seems not to be installed."
        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
            ask_with_confirmation "Do you want to install it into the system?" "Y" "n"
            if [ $? -eq 1 ]; then
                platform_install_unzip "0"
            else
                bad "IceWarp server cannot be installed without unzip"
                exit 1
            fi
        else
            good "Automatic installation mode - installing unzip utility"
            platform_install_unzip "1"

            unzip -h &> /dev/null
            if [ $? -ne 0 ]; then
                bad "Automatic installation mode - cannot install unzip"
                exit 2
            fi
        fi
        unzip -h &> /dev/null
    done
}

test_program_service()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    # check service
    while ! platform_check_service; do
        bad "Program service seems not to be installed."
        if [ "${OPT_AUTO}" != "1" ]; then
            ask_with_confirmation "Do you want to install it into the system?" "Y" "n"
            if [ $? -eq 1 ]; then
                platform_install_service "0"
            else
                bad "IceWarp server cannot be installed without service"
                exit 1
            fi
        else
            good "Automatic installation mode - installing service utility"
            platform_install_service "1"

            if ! platform_check_service; then
                bad "Automatic installation mode - cannot install service"
                exit 2
            fi
        fi
    done
}

test_program_sudo()
{
    # check sudo
    sudo --version &> /dev/null
    while [ $? -ne 0 ]; do
        bad "sudo seems not to be installed."
        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
            ask_with_confirmation "Do you want to install it into the system?" "Y" "n"
            if [ $? -eq 1 ]; then
                platform_install_packages "0" "sudo"
            else
                bad "IceWarp server cannot be installed without sudo"
                exit 1
            fi
        else
            good "Automatic installation mode - installing sudo"
            platform_install_packages "1" "sudo"

            sudo --version &> /dev/null
            if [ $? -ne 0 ]; then
                bad "Automatic installation mode - cannot install sudo"
                exit 2
            fi
        fi
        sudo --version &> /dev/null
    done
    
    sudo -l | grep "\ requiretty" &> /dev/null
    if [ $? -eq 0 ]; then
        SUDO_REQUIRETTY_IS_SET=1

        if [ "$PLATFORM_OSDIST" == "RHEL" ] && [ "$PLATFORM_VERSIONS" == "7" ]; then 
            bad "Your OS has enabled 'sudo' option 'requiretty' which prevents IceWarp"
            bad "to manage its suplementary system services and there is no possiblity"
            bad "for installer to override settings due OS release specifics. As result,"
            bad "Sophos AntiVirus or fulltext search will be unavailable."
            bad "For further information see https://support.icewarp.com/hc/en-us/articles/9149911294225-Manual-override-of-requiretty-sudo-option"
            
            exit 1
        fi
    fi
}

check_documentpreview_programs_presence()
{
    # Check Poppler - if present
    (cd / && pdftoppm -v &> /dev/null)
    POPPLER=$?

    if [ "$PLATFORM_FFMPEG_PACKAGE" != "" ]; then
        (ffmpeg -version &> /dev/null)
        FFMPEG=$?
    else
        FFMPEG=0
    fi
    
    if [ "${OPT_CHECK_FFMPEG}" != "1" ]; then
        warn "Skipping FFMPEG check ..."
        FFMPEG=0
    fi
}

test_program_documentpreview()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi

    MISSING_PACKAGES=""
    MISSING_TEXT=""

    check_documentpreview_programs_presence

    while [ ${POPPLER} -ne 0 ] || [ $FFMPEG -ne 0 ]; do
        if [ ${POPPLER} -ne 0 ]; then
            if [ "x${MISSING_TEXT}" != "x" ]; then
                MISSING_TEXT="${MISSING_TEXT} and "
                MISSING_PACKAGES="${MISSING_PACKAGES} "
            fi

            for I in $(seq 0 ${#PLATFORM_POPPLER_PACKAGE[*]}); do
                MISSING_TEXT="${MISSING_TEXT} ${PLATFORM_POPPLER_PACKAGE[$I]}"
                MISSING_PACKAGES="${MISSING_PACKAGES} ${PLATFORM_POPPLER_PACKAGE[$I]}"
            done
        fi

        if [ ${FFMPEG} -ne 0 ]; then
            if [ "x${MISSING_TEXT}" != "x" ]; then
                MISSING_TEXT="${MISSING_TEXT} and "
                MISSING_PACKAGES="${MISSING_PACKAGES} "
            fi

            MISSING_TEXT="${MISSING_TEXT}${PLATFORM_FFMPEG_PACKAGE}"
            MISSING_PACKAGES="${MISSING_PACKAGES}${PLATFORM_FFMPEG_PACKAGE}"
        fi

        warn "Tools needed for document preview are not installed."
        echo ""
        if [ "${OPT_AUTO}" != "1" ]; then
            ask_with_confirmation "Do you want to install ${MISSING_TEXT} into the system?" "Y" "n"
            if [ $? -eq 1 ]; then
                platform_install_packages "0" ${MISSING_PACKAGES} 
            else
                echo ""
                warn "Press ENTER to continue without document preview, CTRL+C to quit"
                read -s
                echo ""
                break
            fi
        else
            good "Automatic installation mode - installing ${MISSING_TEXT}."
            platform_install_packages "1" ${MISSING_PACKAGES} 

            check_documentpreview_programs_presence
            if [ ${POPPLER} -ne 0 ] || [ ${FFMPEG} -ne 0 ]; then
                if [ "${OPT_AUTO}" != "1" ]; then
                    bad "Automatic installation mode - cannot install document preview tools."
                    exit 2
                else
                    warn "Automatic installation mode - cannot install document preview tools."
                    break
                fi
            fi
        fi
       
        check_documentpreview_programs_presence
    done

    if [ ${POPPLER} -ne 0 ]; then
        CONVERSION_PDF_IMAGE="0"
    fi
}

check_lame_presence()
{
    # check lame - if present
    (cd / && lame --help &> /dev/null)
    LAME=$?
    
    # We distribute lame on platforms where lame is not in default repo
    if [ "x${PLATFORM_LAME_PACKAGE}" == "x" ]; then
        LAME=0
    fi
}

configure_lame()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi

    check_lame_presence

    while [ ${LAME} -ne 0 ]; do
        MISSING_TEXT=""
        MISSING_PACKAGES=""

        if [ ${LAME} -ne 0 ]; then
            if [ "x${MISSING_TEXT}" != "x" ]; then
                MISSING_TEXT="${MISSING_TEXT} and "
                MISSING_PACKAGES="${MISSING_PACKAGES} "
            fi

            MISSING_TEXT="${MISSING_TEXT}${PLATFORM_LAME_PACKAGE}"
            MISSING_PACKAGES="${MISSING_PACKAGES}${PLATFORM_LAME_PACKAGE}"
        fi

        warn "Tools needed for audio encoding/decoding are not installed."
        echo ""
        if [ "${OPT_AUTO}" != "1" ]; then
            ask_with_confirmation "Do you want to install ${MISSING_TEXT} into the system?" "Y" "n"
            if [ $? -eq 1 ]; then
                platform_install_packages "0" ${MISSING_PACKAGES} 
            else
                echo ""
                warn "Press ENTER to continue without audio encoding/decoding tools, CTRL+C to quit"
                read -s
                echo ""
                break
            fi
        else
            good "Automatic installation mode - installing ${MISSING_TEXT}."
            platform_install_packages "1" ${MISSING_PACKAGES} 

            check_lame_presence
            if [ ${LAME} -ne 0 ]; then
                bad "Automatic installation mode - cannot install audio encoding/decoding tools."
                exit 2
            fi
        fi

        check_lame_presence
    done
}

check_legacy_documentpreview()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    #remove previous files
    if [ -d "${SETUP_INSTALL_DIR}/onlyoffice/converter" ]; then
        rm -rf "${SETUP_INSTALL_DIR}/onlyoffice/converter"
    fi
    
    if [ -d "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder" ]; then
        rm -rf "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder"
    fi
    
    good "WebDocuments preparation tasks ... done."
}


get_webdocuments_version()
{
    WEBDOCUMENTS_VERSION=$(curl --connect-timeout 10 -s --header "Content-Type: application/json" \
    --request POST \
    --data '{"c":"version"}' \
    "$1coauthoring/CommandService.ashx" | sed -e 's/^.*"version":"\([^"]*\)".*$/\1/')
}

check_upgrade_webdocuments()
{
    WEBDOCUMENTS_ENABLED=$(get_liccheck_variable "system" "C_WebDocuments_Enabled")
    
    if [ "${WEBDOCUMENTS_ENABLED}" == "1" ]; then
        PACKAGES=""

        for item in ${PLATFORM_WEBDOCUMENTS_VERSION_CHECK_PACKAGES[*]}
        do
            platform_is_package_installed $item
            
            if [ $? == "1" ]; then
              PACKAGES=$PACKAGES" "$item
            fi
        done

        if [ "$PACKAGES" != "" ]; then
            if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ];  then
                ask_with_confirmation "For Webdocuments server version check is needed to install packages. Continue (or skip)?" "Y" "n"
                
                if [ "$?" -eq 1 ]; then
                    platform_install_packages "0" $PACKAGES
                fi
            else
                good "Installing packages required for Webdocuments version check ..."
                platform_install_packages "1" $PACKAGES
            fi
        fi
    
        get_webdocuments_version "$(get_liccheck_variable "system" "C_WebDocuments_Connection")"

        if [ "${WEBDOCUMENTS_VERSION}" != "" ]; then
            compare_generic_versions "$WEBDOCUMENTS_VERSION" "$MINIMAL_REQUIRED_WEBDOCUMENTS_VERSION"
            RESULT="$?"
            
            if [ $RESULT -eq "9" ]; then
                warn "Curret Webdocuments server version $WEBDOCUMENTS_VERSION in use is unsupported. If you want to"
                warn "continue using online document editing, please upgrade to version $MINIMAL_REQUIRED_WEBDOCUMENTS_VERSION or higher."
                warn "For more information please visit $WEBDOCUMENTS_UPDATE_URL."
               
            fi
        else
            warn "Webdocuments server not responding or wrongly configured."
            log_install_error "Webdocuments server not responding or wrongly configured."
        fi
    fi
}

configure_webdocuments()
{
    if [ -d "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/glibc-2.17" ]; then
        GLIBC_VERSION_STRING=$(ldd --version | cut -d ")" -f 2 | cut -d " " -f 2) 
        read -a GLIBC_CURRENT_VERSION <<< $GLIBC_VERSION_STRING

        GLIBC_MIN_VERSION="2.17"

        compare_generic_versions "$GLIBC_CURRENT_VERSION" "$GLIBC_MIN_VERSION"
        RESULT="$?"
        
        if [ $RESULT -eq "10" ] || [ $RESULT -eq "11" ]; then
            #Custom glibc not needed, delete
            good "Custom GLIBC for WebDocuments not needed, removing ..."
            rm -rf "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/glibc-2.17"
        else
            #make temporary symlink
            if [ $SETUP_SERVER_PLATFORM == "x86_64" ]; then
                ln -s "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/DoctRenderer.config" "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/glibc-2.17/lib64/DoctRenderer.config"
            else
                ln -s "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/DoctRenderer.config" "${SETUP_INSTALL_DIR}/onlyoffice/docbuilder/glibc-2.17/lib/DoctRenderer.config"
            fi
        fi
    fi
}

# Check and install Mundi dependencies
test_mundi_dependencies()
{
    MISSING_PROGRAMS=""
    MISSING_PACKAGES=""

    MISSING_PACKAGES="x"
    FIRST_PASS=0
    while [ "x${MISSING_PACKAGES}" != "x" ]; do
        MISSING_PROGRAMS=""
        MISSING_PACKAGES=""
        good "Checking Document conversion tool (Mundi) dependencies..."

        platform_check_mundi_extra_repo_enabled "${OPT_AUTO}"
        EXTRA_REPO_ENABLED=$?
        for I in $(seq 1 ${#PLATFORM_MUNDI_PROGRAMS[*]}); do
            # Run check
            eval "${PLATFORM_MUNDI_PROGRAMCHECKS[$I]}" >/dev/null 2>&1

            # Check result
            if [ $? -ne ${PLATFORM_MUNDI_CHECKRESULTS[$I]} ]; then
                # Missing
                if ${PLATFORM_MUNDI_EXTRAREPO[$I]} && [ $EXTRA_REPO_ENABLED -ne 0 ]; then
                    continue
                fi

                MISSING_PROGRAMS="${MISSING_PROGRAMS} ${PLATFORM_MUNDI_PROGRAMS[$I]}"
                MISSING_PACKAGES="${MISSING_PACKAGES}\n${PLATFORM_MUNDI_PACKAGES[$I]}"
            fi
        done
        
        if [ "x${MISSING_PACKAGES}" != "x" ]; then
            # Warn if this is not first pass - something went wrong with installation
            if [ ${FIRST_PASS} -ne 0 ]; then
                echo ""
                warn "All required dependencies are still not satisfied,
             probably the installation of them failed or was interrupted.
             
             Press ENTER to continue or Ctrl-C to exit setup"
                if [ "${OPT_AUTO}" != "1" ]; then
                    read -s
                    echo ""
                else
                    bad "Automatic installation mode - dependencies not installed, exiting"
                    exit 2
                fi
            fi

            FIRST_PASS=1
            # Strip duplicates, convert to space separated list
            MISSING_PACKAGES=$(echo -e ${MISSING_PACKAGES} | sort | uniq | tr '\n' ' ' | sed 's/[[:space:]]\+/ /g')

            warn "Some of the programs required by Document conversion tool (Mundi) were not found on this system."
            echo ""
            echo "        Missing programs: ${MISSING_PROGRAMS}"
            echo ""
            echo "        These packages should be installed to satisfy the dependencies:
       ${MISSING_PACKAGES}"
            echo ""
            if [ "${OPT_AUTO}" != "1" ]; then
                ask_with_confirmation "Do you want to install these packages into the system?" "Y" "n"
                if [ $? -eq 1 ]; then
                    platform_install_packages "0" $MISSING_PACKAGES 
                else
                    echo ""
                    warn "Document conversion tool (Mundi) will not fully operate without these packages
                 Press ENTER to continue or Ctrl-C to exit setup"
                    read -s
                    echo ""
                    break
                fi
            else
                good "Automatic installation mode - installing missing dependencies"
                platform_install_packages "1" $MISSING_PACKAGES 
            fi
        fi
        echo ""
    done
}

test_required_programs()
{
    test_program_servicemanagement
    test_program_sed
    test_program_service
    test_program_sudo
}

# Detection of host OS distribution
# Function tryes to detect OS in simple way
# Result of detection is:
# - "UNKNOWN" if distributionplatform is not known
# - "RHEL" if OS is Red Hat Enterprise Linux
detect_os_distribution()
{
    OS_DIST="UNKNOWN"
    OS_PLATFORM="UNKNOWN"

    if [ -f "/etc/redhat-release" ]; then
        OS_DIST="RHEL"
    fi
    if [ -f "/etc/debian_version" ]; then
        OS_DIST="DEB"
    fi
    if [ -f "/etc/arch-release" ]; then
        OS_DIST="ARCH"
    fi
    if [ "$OS_DIST" == "UNKNOWN" ]; then
        bad "Unknown Linux distribution"
        exit 1
    fi

    OS_PLATFORM=$(uname -m)
}

# Ask question with confirmation
# Parameters:
# $1 - question
# $2 - first choice, it is choosed if user enters only empty string
# $3 - second choice, $4 third choice...
# Return:
# 0 on error - unknown string entered
# 1 if first choice was selected
# 2 if second choice was selected
# 3 if third choice...
ask_with_confirmation()
{
    if [ -z "$1" ] || [ -z "$2" ] || [ -z "$3" ]; then
        return 0
    fi
    LOWER_CHOICE_1=$(echo "$2" | tr "A-Z" "a-z")
    LOWER_CHOICE_2=$(echo "$3" | tr "A-Z" "a-z")

    # make options string
    local OPTIONS="[$(date)] ${GOOD}**${NORMAL}\t${1} ${BRACKET}[${HILITE}${2}${NORMAL}"
    for ARG in "${@:3}"; do 
        OPTIONS+="/${ARG}"
    done
 
    echo -e -n "${OPTIONS}${BRACKET}]${NORMAL}: "
    clear_stdin
    read PARAM
    if [ -z "$PARAM" ]; then 
        return 1
    fi

    PARAM=$(echo "$PARAM" | tr "A-Z" "a-z")
    declare -i I=1
    for ARG in "${@:2}"; do  # arguments 2 through n (i.e. 3 args starting at number 2)
        local LOWER_CHOICE=$(echo "$ARG" | tr "A-Z" "a-z")
        if [ "$PARAM" == "$LOWER_CHOICE" ]; then
            return $I
        fi
        ((I++))
    done

    return 0
}

preconf()
{
    # reset variables for old configuration
    InstallDir=""
    User=""
    Version=""

    if [ -f "$1" ]; then
        source "$1"

        # check for new variables first

        # check for install dir
        if [ "x${IWS_INSTALL_DIR}" != "x" ]; then
            SETUP_INSTALL_DIR="$IWS_INSTALL_DIR"
        fi

        # check for install user
        if [ "x${IWS_PROCESS_USER}" != "x" ]; then
            SETUP_INSTALL_USER="$IWS_PROCESS_USER"
        fi

        # check for installed version
        if [ "x${IWS_VERSION}" != "x" ]; then
            SETUP_OLDVERSION="$IWS_VERSION"
        fi

        # check for installed platform
        if [ "x${IWS_PLATFORM}" != "x" ]; then
            SETUP_SERVER_OLDPLATFORM="${IWS_PLATFORM}"
        fi

        # check for old variables

        # check for install dir
        if [ "x${InstallDir}" != "x" ]; then
            SETUP_INSTALL_DIR="$InstallDir"
        fi

        if [ "x${User}" != "x" ]; then
            SETUP_INSTALL_USER="$User"
        fi

        if [ "x${Version}" != "x" ]; then
            SETUP_OLDVERSION="$Version"
        fi
    fi
}

# Function displays license file and waits until user press [ENTER]
accept_license()
{
    cat "${WORKDIR}/LICENSE"
    echo ""
    good "You must accept this license agreement if you want to continue."
    good "Press ENTER to accept license or CTRL+C to quit"
    if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
        read -s
        echo ""
    else
        good "Automatic installation mode - license accepted."
    fi
    echo ""
}

# Displayes information about stdout log and stderr log placement
display_log_info()
{
    good "Installer log is available in ${INSTALL_LOG}"
    good "Installer error log is available in ${INSTALL_ERROR_LOG}"
    good ""
}

# Set some variables in docker mode
configure_docker_install()
{
    if [ "${OPT_DOCKER}" != "1" ]; then
        return
    fi

    export NON_INTERACTIVE=1
    OPT_CHECK_FFMPEG="0"
}


# Function checks if configuration file exists and sets variables
# Parameters
#  $1 - full path to assumed configuration file
# Returns:
#  0 - if all variables SETUP_INSTALL_DIR, SETUP_INSTALL_USER are detected
# != 0 - if some of variables or non of variables SETUP_INSTALL_DIR, SETUP_INSTALL_USER are detected
detect_install_options_sub()
{
    CONFIG_FILE="$1"

    # for safety initialize variables
    SETUP_INSTALL_DIR="$EMPTY_VALUE"
    SETUP_INSTALL_USER="$EMPTY_VALUE"
    SETUP_OLDVERSION="$EMPTY_VALUE"
    SETUP_SERVER_OLDPLATFORM="${EMPTY_VALUE}"

    if [ -f "$CONFIG_FILE" ]; then
        preconf "$CONFIG_FILE"
        # check for set values
        if [ "x${SETUP_INSTALL_DIR}" != "x${EMPTY_VALUE}" ] &&
           [ "x${SETUP_INSTALL_USER}" != "x${EMPTY_VALUE}" ] &&
           [ "x${SETUP_OLDVERSION}" != "x${EMPTY_VALUE}" ]; then
           # all values was set
           return 0
        fi
    fi

    # configuration file not found, or some values was not set
    # reset variables
    SETUP_INSTALL_DIR="$EMPTY_VALUE"
    SETUP_INSTALL_USER="$EMPTY_VALUE"
    SETUP_OLDVERSION="$EMPTY_VALUE"
    SETUP_SERVER_OLDPLATFORM="${EMPTY_VALUE}"
    return 1
}

# Function detects install options entered by user from already installed server
# If variables are not detected, then default values are set
# Load balanced setup is not detected here, because for detecting load balanced setup tool needs to be called,
# so server configuration file must be created first.
# If configuration file is found, flag SETUP_CONFIG_FILE_EXISTS is set to value other than 0
# Variables detected:
# 1. install dir default - SETUP_INSTALL_DIR_DEFAULT
# 2. install user - SETUP_INSTALL_USER
# returns:
# always 0, if install options are not detected they are set to default values
detect_install_options()
{
    # detect configuration files from newest to oldest,
    # Merak 9.4 upgraded to IceWarp Server 10 can be already installed or
    # clean IceWarp Server 10 can be already installed

    # assume configuration file exists
    SETUP_CONFIG_FILE_EXISTS="1"

    detect_install_options_sub "/etc/icewarp/icewarp.conf"
    if [ $? -eq 0 ]; then
        return 0
    fi

    # directory /opt/icewarp

    # IceWarp Server 10
    detect_install_options_sub "/opt/icewarp/config/icewarp.conf"
    if [ $? -eq 0 ]; then
        return 0
    fi

    # Merak 9.4
    detect_install_options_sub "/opt/icewarp/config/merak.conf"
    if [ $? -eq 0 ]; then
        return 0
    fi

    # directory /opt/merak
    
    # IceWarp Server 10
    detect_install_options_sub "/opt/merak/config/icewarp.conf"
    if [ $? -eq 0 ]; then
        return 0
    fi

    # Merak 9.4
    detect_install_options_sub "/opt/merak/config/merak.conf"
    if [ $? -eq 0 ]; then
        return 0
    fi

    # set default values
    if [ "x${OPT_INSTALL_DIR}" == "x" ]; then
        SETUP_INSTALL_DIR="$SETUP_INSTALL_DIR_DEFAULT"
    else
        SETUP_INSTALL_DIR="${OPT_INSTALL_DIR}"
    fi

    if [ "x${OPT_USER}" == "x" ]; then
        SETUP_INSTALL_USER="$SETUP_INSTALL_USER_DEFAULT"
    else
        SETUP_INSTALL_USER="${OPT_USER}"
    fi

    # configuration file not found
    SETUP_CONFIG_FILE_EXISTS="0"

    return 1
}

# If previous installation is detected, function checks if it is not newer or if 32bit is not replacing 64bit
test_unwanted_downgrade()
{
    if [ -d "$SETUP_INSTALL_DIR" -a "$SETUP_CONFIG_FILE_EXISTS" != "0" ]; then
        # check downgrade
        compare_version "${SETUP_OLDVERSION}" "${SETUP_NEWVERSION}" true true
        if [ $RESULT -lt 0 ]; then
            echo ""
            warn "You are trying to downgrade IceWarp Server from version ${SETUP_OLDVERSION} to ${SETUP_NEWVERSION}."
            echo "        Note that downgrade is unsupported and can lead to irreversible corruption of the server."
            echo "        Do you want to continue?"
            echo ""
            echo "        Press ENTER to continue, CTRL+C to quit"
            if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                read -s
                echo ""
            else
                good "Automatic installation mode - downgrade confirmed."
            fi
        fi

        # check bitness conflict
        if [ "${SETUP_SERVER_OLDPLATFORM}" == "x86_64" ] && [ "${SETUP_SERVER_PLATFORM}" != "x86_64" ]; then
            echo ""
            warn "You are trying to install 32bit IceWarp Server over existing 64bit installation."
            echo "        Do you want to continue?"
            echo ""
            echo "        Press ENTER to continue, CTRL+C to quit"
            if [ "${OPT_AUTO}" != "1" ]; then
                read -s
                echo ""
            else
                good "Automatic installation mode - upgrading 64bit to 32bit confirmed."
            fi
        fi

    fi
}

# Function displays questions about SETUP_INSTALL_DIR, SETUP_INSTALL_USER
# Function detect_install_options() must be called first
# returns:
#  no return
ask_questions()
{
    # check for "new installation" or "upgrade"
    # check if detected installation directory exists
    if [ -d "$SETUP_INSTALL_DIR" -a "$SETUP_CONFIG_FILE_EXISTS" != "0" ]; then
        good "Previous IceWarp server installation detected."
        good "Installation directory was set to ${SETUP_INSTALL_DIR}."
        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ];  then
            ask_with_confirmation "Do you want to upgrade?" "Y" "n"
            if [ $? -eq 1 ]; then
                UPGRADE="1"
            else
                UPGRADE="0"
            fi
        else
            good "Automatic installation mode - choosing upgrade"
            UPGRADE="1"
        fi
    else
        good "Performing new install"
        UPGRADE="0"
    fi

    if [ "${UPGRADE}" == "0" ]; then
        echo ""
        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ];  then
            getparam "Installation prefix" "$SETUP_INSTALL_DIR"
            if [ "x${PARAM}" != "x" ]; then
                SETUP_INSTALL_DIR="$PARAM"
            fi
        else
            good "Automatic installation mode - using ${SETUP_INSTALL_DIR} as installation directory"
        fi

        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
            getparam "Run services as user" "$SETUP_INSTALL_USER"
            if [ "x$PARAM" != "x" ]; then
                SETUP_INSTALL_USER="$PARAM"
            fi
        else
            good "Automatic installation mode - using ${SETUP_INSTALL_USER} as user";
        fi

        if [ -d "${SETUP_INSTALL_DIR}" ]; then
            echo ""
            if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                warn "Directory ${SETUP_INSTALL_DIR} already exists
                     Please select full installation or upgrade.
                     Directory ${SETUP_INSTALL_DIR} will be DELETED in full installation!"
                echo ""
                ask_with_confirmation "Upgrade or Install?" "U" "i"
                if [ $? -eq 1 ]; then
                    UPGRADE="1"
                else
                    UPGRADE="0"
                fi
            else
                bad "Automatic installation mode - directory already exists, but previous installation information was not found."
                exit 2
            fi
        fi
    fi

    # check if installation file exists but it is not directory
    if [ "$UPGRADE" == "0" -a -f "$SETUP_INSTALL_DIR" ]; then
        bad "File ${SETUP_INSTALL_DIR} exists but it is not directory."
        exit 1
    fi

    # set server configuration file
    SETUP_SERVER_CONF="/etc/icewarp/icewarp.conf"
}

# Unpack liccheck dependencies to temporary path
unpack_liccheck_dependencies()
{
    tar -xf "${WORKDIR}/${IMAGENAME}" -C "$WORKDIR" lib/ lib64/ ldap/lib/
}

# Function obtains current license.key path
# Parameters:
#  none
# Returns:
#  none
check_license_path()
{
    # Get license path from path.dat if present
    if [ -f "${SETUP_INSTALL_DIR}/path.dat" ]; then
        LICENSE_PATH="$(sed -n "12p" < ${SETUP_INSTALL_DIR}/path.dat)"
    fi

    # If is path empty, use default
    if [ "${LICENSE_PATH}" == "" ]; then
        LICENSE_PATH="${SETUP_INSTALL_DIR}/config/license.key"
    fi
}

# Function checks if there is not expired license for upgrade
# Parameters:
#  none
# Returns:
#  none
check_upgrade_license()
{
    # Dear script editors - be warned that server will not work properly without valid license even if you comment this check out ;-)
    if [ "${UPGRADE}" == "1" ]; then
        SOPHOS_FORCE_REACTIVATION=0
        LICENSE_CIPHER_FORCE_REACTIVATION=0

        if ! get_api_variable "system" "C_License_Expired_For_Upgrade"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi

        if [ "${GET_API_VARIABLE}" == "1" ]; then
            bad "Cannot upgrade IceWarp Server because of expired license." 
            exit 3
        fi
        
        if ! get_api_variable "system" "c_version"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi

        SOPHOS_CHECK_VERSION=$(cut -d ' ' -f 1 <<< "$GET_API_VARIABLE")
        compare_generic_versions "$SOPHOS_CHECK_VERSION" "13.0.3.0"
        
        if [ "$?" -eq "9" ]; then
            SOPHOS_FORCE_REACTIVATION=1
        fi
        
        # Dump license to file and get orderid
        TMP_FILE=$(mktemp)

        if ! get_api_variable "system" "c_license_xml"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi

        # parse product
        echo "$GET_API_VARIABLE" > ${TMP_FILE}

        TMP_IFS=$IFS
        IFS=$'\n'

        PURCHASEID_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<purchaseid/?>" | sed "s|.*<purchaseid[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
        VERSION_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<version/?>" | sed "s|.*<version[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")

        #cleanup
        IFS=$TMP_IFS
        rm -rf $TMP_FILE
        
        if [ "${VERSION_STRING}" -lt "${MINIMAL_VERSION_NEW_CIPHER}" ]; then
            LICENSE_CIPHER_FORCE_REACTIVATION=1
        fi
        
        LICENSING_STATUS=$(get_liccheck_variable "system" "C_LicenseStatus")
            
        if [ "${LICENSING_STATUS}" == "8" ] || [ "${SOPHOS_FORCE_REACTIVATION}" == "1" ] || [ "${LICENSE_CIPHER_FORCE_REACTIVATION}" == "1" ]; then
            if [ "${SOPHOS_FORCE_REACTIVATION}" == "1" ]; then
                good "Re-activating license due to AntiVirus engine transition ..."
            fi
            
            if [ "${LICENSE_CIPHER_FORCE_REACTIVATION}" == "1" ]; then
                good "Re-activating license due to licensing transition ..."
                
                # Retrieve current license.key path
                check_license_path
                # Save old license
                cp -f "${LICENSE_PATH}" "${LICENSE_PATH}.old"
                # Set to use LICCHECK instead of TOOL
                REPLACE_TOOL_WITH_LICCHECK="1"
            fi
        
            # try to silently reactivate in some cases
            if ! set_liccheck_variable "system" "c_onlinelicense" "${PURCHASEID_STRING}"; then
                bad "Re-activating license failed."
            fi
        fi

        LICENSING_STATUS=$(get_liccheck_variable "system" "C_LicenseStatus")
        
        SKIP_LICENSING=false
        while true; do
            if [ "${LICENSING_STATUS}" == "8" ]; then
                LICENSE_VERSION_REQUIRED=$(get_liccheck_variable "system" "C_License_Minimal_Required_Version")
                
                if [ "${OPT_AUTO}" == "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                    SKIP_LICENSING=true
                fi
                
                if [ "${LICENSE_VERSION_REQUIRED}" == "Property c_license_minimal_required_version not found" ]; then
                    echo ""
                    warn "New terms of the license agreement have to be agreed before proceeding with installation." 
                    echo "        - Go to https://www.icewarp.com/upgrade-dc-gen2/?order-id=$PURCHASEID_STRING&os=lnx"
                    echo "        - Accept new terms of the license agreement and upgrade your license on the website"
                    echo "        - Confirm re-activation of the license below."
                    echo ""                

                    if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ];  then
                        ask_with_confirmation "Re-activate the license" "Y" "n"              
                        
                        if [ $? -eq 1 ]; then
                            set_liccheck_variable "system" "c_onlinelicense" "${PURCHASEID_STRING}"
                        else
                            SKIP_LICENSING=true
                        fi
                    fi
                else
                    echo ""
                    warn "Incorrect license version found. Before proceeding with installation, current license "
                    echo "        has to be upgraded and new terms of the license agreement have to be agreed."
                    echo "        - Go to https://www.icewarp.com/upgrade-dc-gen2/?order-id=$PURCHASEID_STRING&os=lnx"
                    echo "        - Accept new terms of the license agreement and upgrade  your license on the website"
                    echo "        - Confirm re-activation of the license below." 
                    echo ""                
                    
                    if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                        ask_with_confirmation "Re-activate the license" "Y" "n"              
                        
                        if [ $? -eq 1 ]; then
                            set_liccheck_variable "system" "c_onlinelicense" "${PURCHASEID_STRING}"
                        else
                            SKIP_LICENSING=true
                        fi
                    fi
                fi
            fi

            if [ $SKIP_LICENSING == true ]; then
                echo ""
                warn "Outdated server and license version found. If proceeding further with installation,"
                echo "        IceWarp Server may NOT WORK until manually updating current license. You can check"
                echo "        current license status or update the license via wizard.sh tool. Please contact"
                echo "        sales for license update or go to www.icewarp.com."
                echo ""                
                
                if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                    SKIP_LICENSING=false
                    ask_with_confirmation "Do you want to continue" "N" "y" 
                    
                    if [ $? -ne 2 ]; then
                        exit 1
                    else
                        break
                    fi
                else
                    break
                fi
                
            fi
            
            LICENSING_STATUS=$(get_liccheck_variable "system" "C_LicenseStatus")
            if [ $? -ne 0 ]; then
                # Broken licccheck, maybe corrupted installation
                # Let's proceed, services won't start anyway in case of wrong license
                echo ""
                bad "Failed to get license information ($LICENSING_STATUS).
        Press ENTER to continue or Ctrl+C to exit setup"
                read -s
                break
            fi

            if [ "${LICENSING_STATUS}" == "0" ]; then
                break
            fi
                
        done
    fi
}

# Function checks for cloud API availability necessary for IceWarp to function
check_upgrade_dashboard()
{
    if [ "${UPGRADE}" == "0" ]; then
        return
    fi

    if [ "${OPT_CHECK_DASHBOARD_API}" == "0" ]; then
        return
    fi

    if [ "${OPT_CHECK_DASHBOARD_API_ONLY}" == "0" ]; then
        good "Checking for cloud API availability ..."

        if ! get_api_variable "system" "C_version"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi

        SERVER_VERSION=$(sed 's/^[^:]*: //' <<< "$GET_API_VARIABLE" | cut -d'.' -f1)
        
        if [ "${SERVER_VERSION}" -ge "${DASHBOARD_MINIMAL_REQUIRED_VERSION}" ]; then
            good "Detected IceWarp version ${DASHBOARD_MINIMAL_REQUIRED_VERSION} or greater, skipping check."
            return
        fi
    fi
    
    # Make sure control service is running
    "${SETUP_INSTALL_DIR}/icewarpd.sh" --check control > /dev/null
    
    if [ "$?" != "0" ]; then
        good "Starting control service for API check ..."    
        "${SETUP_INSTALL_DIR}/icewarpd.sh" --start control > /dev/null
        
        sleep 5
    fi
    
    if ! get_api_variable "system" "c_webmail_url"; then
        bad "$GET_API_VARIABLE"
        exit 3
    fi
    SERVER_URL="$GET_API_VARIABLE"
    SERVER_HOSTNAME=$(echo $SERVER_URL | cut -d'/' -f3 | cut -d':' -f1)
    
    RESULT_CODE=$(curl -w "%{http_code}" -X 'POST' "${DASHBOARD_API_CHECK_URL}" -H 'accept: application/json' -H 'Content-Type: application/json' -d "{\"hostname\": \"${SERVER_HOSTNAME}\"}" -s -o /dev/null)
    
    if [ $RESULT_CODE != '200' ]; then
        bad "Installation cannot proceed because your IceWarp Server running on ${SERVER_HOSTNAME} cannot be reached by cloud API. Please follow instructions at below link before running the installation again:
        https://www.icewarp.com/api-unavailable-help/"
        
        exit 1
    fi
    
    good "The API check was successful, continuing."
    
    if [ "${OPT_CHECK_DASHBOARD_API_ONLY}" == "1" ]; then
        exit 0
    fi
}

# Function checks for entered install options. Failed check causes script exit.
# Parameters:
#  none
# Returns:
#  none
check_install_options()
{
    # test if user exists
    id "$SETUP_INSTALL_USER" &> /dev/null
    if [ $? -ne 0 ]; then
        bad "User ${SETUP_INSTALL_USER} doesn't exist."
        exit 1
    fi

    # test if user is in group
    id -g "$SETUP_INSTALL_USER" &> /dev/null
    if [ $? -ne 0 ]; then
        bad "User ${SETUP_INSTALL_USER} is not in group."
        exit 1
    fi
}

# Function asks user to confirm entered installation options
# Parameters:
#  none
# Returns:
#  none
confirm_install_options()
{
    SETUP_INSTALL_GROUP=$(id -gn "$SETUP_INSTALL_USER")

    good ""
    good "Please check entered informations before continuing:"
    good ""
    if [ "$UPGRADE" == "1" ]; then
        good "Installation prefix:\t\t${SETUP_INSTALL_DIR} (upgrading)"
    else
        good "Installation prefix:\t\t${SETUP_INSTALL_DIR} (directory will be created)"
    fi

    good "IceWarp Server will run as user:\t${SETUP_INSTALL_USER}"
    good "IceWarp Server will run as group:\t${SETUP_INSTALL_GROUP}"

    good ""
    good "Press ENTER to continue, CTRL+C to quit"
    if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
        read -s
        echo ""
    else
        good "Automatic installation mode - starting installation."
    fi
    echo ""
}

# Functions check for services and 3rd party processes executed by IceWarp server
# When detected, user has a choice to kill them or exit
check_server_is_running()
{
    # This procedure is performed only when upgrading
    if [ "$UPGRADE" == "0" ]; then
        return
    fi

    SERVICES=( "${SETUP_INSTALL_DIR}/icewarpd" "${SETUP_INSTALL_DIR}/control" "${SETUP_INSTALL_DIR}/cal" "${SETUP_INSTALL_DIR}/im" "${SETUP_INSTALL_DIR}/pop3" "${SETUP_INSTALL_DIR}/smtp" "${SETUP_INSTALL_DIR}/purple/purpleserv" "ctasd.bin" "ctipd.bin" "${SETUP_INSTALL_DIR}/ldap/libexec/slapd" "${SETUP_INSTALL_DIR}/kasperskyupdater" "${SETUP_INSTALL_DIR}/kaspersky/kavehost" "kavscanner" "${SETUP_INSTALL_DIR}/voip/echo-voicemail-service.jar" )

    # At first check, if any services are running
    SERVER_RUNS=1
    for SERVICE in "${SERVICES[@]}"; do
        ps ax | grep "${SERVICE}" | grep -v "grep">/dev/null
        if [ $? -eq 0 ]; then
            SERVER_RUNS=0
        fi
    done

    if [ $SERVER_RUNS -eq 0 ]; then
        good "Stopping running services"
        echo ""
        
        "${SETUP_INSTALL_DIR}/icewarpd.sh" --stop > /dev/null        
        
        KILL_SERVICES="1"
        for SERVICE in "${SERVICES[@]}"; do
            ps ax | grep "${SERVICE}" | grep -v "grep" | awk '{ print ($1); }' | while read pid; do
                kill -9 $pid
            done
        done
        
        good "Detected IceWarp services have been stopped"
        echo ""
    fi
}

# Function checks for running php processes
# If detected, user is ask to kill them or exit
# Parameters:
#  none
# Returns:
#  none
check_running_php()
{
    # This procedure is performed only when upgrading
    if [ "$UPGRADE" == "1" ]; then
        # Detect, if previous PHP runs
        PHP_DETECTED="0"
        if ps ax | grep "${SETUP_INSTALL_DIR}/php/php" | grep -v "grep">/dev/null; then
            PHP_DETECTED="1"
        fi
        if ps ax | grep "${SETUP_INSTALL_DIR}/scripts/phpd.sh" | grep -v "grep">/dev/null; then
            PHP_DETECTED="1"
        fi
        if ps ax | grep "php-fpm" | grep -v "grep">/dev/null; then
            PHP_DETECTED="1"
        fi
        
        # Warn user and let him decide
        if [ "${PHP_DETECTED}" == "1" ]; then
            if [ "${KILL_SERVICES}" == "0" ]; then
                warn "Running PHP processes of previous version detected
        Are you sure you stopped IceWarp server before upgrading?
            
        Press ENTER to kill detected PHP processes or Ctrl+C to exit setup"
                if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
                    read -s
                    echo ""
                else
                    good "Automatic installation mode - killing PHPs."
                fi
                echo ""
            fi
            
            # Kill detected phps
            ps ax | grep "${SETUP_INSTALL_DIR}/php/php" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
                kill -9 $pid
            done
            
            # Kill phpd.sh script
            ps ax | grep "${SETUP_INSTALL_DIR}/scripts/phpd.sh" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
                kill -9 $pid
            done

            # Kill php-fpm
            ps ax | grep "php-fpm" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
                kill -9 $pid
            done
            
            if [ "${KILL_SERVICES}" == "0" ]; then
                good "Detected PHP processes have been killed"
                echo ""
            fi
        fi
    fi
}

# Output service sudoers config to start/stop/restart
# $1 - full service name
# $2 - short service name
# $3 - if equals "reload", add service reload to sudo too
configure_service_sudo()
{
    # Sudoers not needed when running under root
    if [ "$SETUP_INSTALL_USER" == "root" ]; then
        return
    fi
    
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /bin/systemctl start $1" > "/etc/sudoers.d/icewarp_$2"
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /bin/systemctl restart $1" >> "/etc/sudoers.d/icewarp_$2"
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /bin/systemctl stop $1" >> "/etc/sudoers.d/icewarp_$2"
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /bin/systemctl is-active --quiet $1" >> "/etc/sudoers.d/icewarp_$2"
    
    # Allow service to be restarted on schedule
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /usr/bin/systemd-run --on-active=5 systemctl restart $1" >> "/etc/sudoers.d/icewarp_$2"    
    
    if [ "$3" == "reload" ]; then
        echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /bin/systemctl reload $1" >> "/etc/sudoers.d/icewarp_$2"    
    fi
}

# Configures sudo for single service handled by icewarpd.sh, used for scheduling commands
# $1 - service name (control, smtp, gw, pop3, im, all
configure_icewarpd_sudo_single_service()
{
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /usr/bin/systemd-run --on-active=5 ${SETUP_INSTALL_DIR}/icewarpd.sh --start $1" >> "/etc/sudoers.d/icewarp_icewarpd"
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /usr/bin/systemd-run --on-active=5 ${SETUP_INSTALL_DIR}/icewarpd.sh --stop $1" >> "/etc/sudoers.d/icewarp_icewarpd"
    echo "%${SETUP_INSTALL_USER} ALL=(ALL) NOPASSWD: /usr/bin/systemd-run --on-active=5 ${SETUP_INSTALL_DIR}/icewarpd.sh --restart $1" >> "/etc/sudoers.d/icewarp_icewarpd"
}

# Configures sudo for all possible icewarpd.sh start/stop/restart service calls, used for scheduling commands
configure_icewarpd_sudo_all_services()
{
    # Sudoers not needed when running under root
    if [ "$SETUP_INSTALL_USER" == "root" ]; then
        return
    fi
    
    rm -f "/etc/sudoers.d/icewarp_icewarpd"
    touch "/etc/sudoers.d/icewarp_icewarpd"
    
    configure_icewarpd_sudo_single_service "control"
    configure_icewarpd_sudo_single_service "smtp"
    configure_icewarpd_sudo_single_service "pop3"
    configure_icewarpd_sudo_single_service "gw"
    configure_icewarpd_sudo_single_service "im"
    configure_icewarpd_sudo_single_service "all"
}

# Installs YODA service
# $1 - service executable name
# $2 - config file name
# $3 - service name
install_yoda_service()
{
    good "Installing YODA $3 service."
    
    if [ "$UPGRADE" == "1" ]; then
        if [ -f "${SETUP_INSTALL_DIR}/yoda/$2" ]; then
            # Merge config
            chown root "${SETUP_INSTALL_DIR}/yoda/$2"
            LD_LIBRARY_PATH="${SETUP_INSTALL_DIR}/lib64:${SETUP_INSTALL_DIR}/lib:${SETUP_INSTALL_DIR}/ldap/lib:$LD_LIBRARY_PATH" "${SETUP_INSTALL_DIR}/yoda/distconfig/xmlmerge" "${SETUP_INSTALL_DIR}/yoda/distconfig/$2" "${SETUP_INSTALL_DIR}/yoda/$2"
            if [ $? -ne 0 ]; then
                warn "Failed to synchronize Yoda $3 service config file ($2)"
            fi
            chown $SETUP_INSTALL_USER:$SETUP_INSTALL_GROUP "${SETUP_INSTALL_DIR}/yoda/$2"
        else
            cp -af "${SETUP_INSTALL_DIR}/yoda/distconfig/$2" "${SETUP_INSTALL_DIR}/yoda"
        fi
    else
        cp -af "${SETUP_INSTALL_DIR}/yoda/distconfig/$2" "${SETUP_INSTALL_DIR}/yoda"
    fi
    
    # Always put current distributed config file as example
    mv -f "${SETUP_INSTALL_DIR}/yoda/distconfig/$2" "${SETUP_INSTALL_DIR}/yoda/$2.example"

    # Install system service
    "${SETUP_INSTALL_DIR}/yoda/$1" --service install --user ${SETUP_INSTALL_USER} --config "${SETUP_INSTALL_DIR}/yoda/$2" &>/dev/null
    
    if [ $? -ne 0 ]; then
        bad "Failed to install YODA $3 service."
    fi

    # Allow to start/stop/restart the service via systemctl
    configure_service_sudo $1 $1 ""
}

# Installs YODA services
install_yoda()
{
    install_yoda_service "yoda" "yoda.xml" "Server"
    install_yoda_service "mundi" "mundi.xml" "Document conversion"
    install_yoda_service "yoda-scan" "yoda-scan.xml" "Indexer"

    rm -rf "${SETUP_INSTALL_DIR}/yoda/distconfig"
}

# Checks if YODA is running, stops and uninstalls existing service
# $1 - service executable name
# $2 - config file name
# $3 - service name
check_yoda_service()
{
    PID_FILE="${SETUP_INSTALL_DIR}/yoda/run/$1.pid"
    EXECUTABLE_FILE="${SETUP_INSTALL_DIR}/yoda/$1"

    if [ ! -f "$EXECUTABLE_FILE" ]; then
        #file does not exist
        return
    fi

    if [ -f "$PID_FILE" ]; then
        YODA_PID=$(<"$PID_FILE")

        kill -0 "$YODA_PID" &> /dev/null
        
        #stop yoda service before upgrade
        if [ $? -eq 0 ]; then
            good "Stopping existing YODA $3 service."
            
            "${EXECUTABLE_FILE}" --service stop --config "${SETUP_INSTALL_DIR}/yoda/$2" &>/dev/null
            
            if [ $? -ne 0 ]; then
                bad "Failed to stop YODA service."
            fi
        fi    
    fi
    
    # uninstall yoda service
    "${EXECUTABLE_FILE}" --service uninstall --config "${SETUP_INSTALL_DIR}/yoda/$2" &>/dev/null
    
    if [ $? -ne 0 ]; then
        bad "Failed to uninstall YODA $3 service."
    fi
}

# Checks for running YODA services, stops and uninstalls them
check_yoda()
{
    # This procedure is performed only when upgrading
    if [ "$UPGRADE" == "1" ]; then
        if [ -f "${SETUP_INSTALL_DIR}/yoda/config.xml" ]; then
            check_yoda_service "yoda" "config.xml" "Server"
            mv "${SETUP_INSTALL_DIR}/yoda/config.xml" "${SETUP_INSTALL_DIR}/yoda/config.xml.obsolete"
            if ! [ -f "${SETUP_INSTALL_DIR}/yoda/yoda.xml" ]; then
                cp -af "${SETUP_INSTALL_DIR}/yoda/config.xml.obsolete" "${SETUP_INSTALL_DIR}/yoda/yoda.xml"
            fi

            warn "Yoda now uses yoda.xml instead of config.xml
        Press ENTER to continue"
            if [ "${OPT_AUTO}" != "1" ]; then
                read -s
                echo ""
            fi
            echo ""
        else
            check_yoda_service "yoda" "yoda.xml" "Server"
        fi
        check_yoda_service "mundi" "mundi.xml" "Document conversion"
        check_yoda_service "yoda-scan" "yoda-scan.xml" "Indexer"

        SETUP_YODA_ENABLED=$(get_liccheck_variable "system" "C_System_Services_Fulltext_Enabled")
    fi
}

# Checks, if uname -n is resolvable
# This is needed for ctasd to work
check_resolvable_hostname()
{
    if [ "$OPT_DOCKER" == "1" ]; then
        return
    fi

    HOSTNAME=$(uname -n)
    if ! ping -c 1 -W 5 "${HOSTNAME}" &>/dev/null; then
        warn "Your system hostname \"${HOSTNAME}\" is not resolvable
        This will cause, that Anti-Spam Live will not work

        Press ENTER to continue or Ctrl+C to exit setup"
        if [ "${OPT_AUTO}" != "1" ]; then
            read -s
            echo ""
        else
            bad "Automatic installation mode - wrong hostname configuration."
        fi
        echo ""
    fi
}

# Function loads some paths from path.dat, if the file exists
# The paths are needed for customization extraction in case of build_profile etc.
load_special_paths()
{
    if [ "${UPGRADE}" != "0" ] && [ -f "${SETUP_INSTALL_DIR}/path.dat" ]; then
        SETUP_CONFIG_DIR=$(sed '1p;d' "${SETUP_INSTALL_DIR}/path.dat" | tr -d '\r')
        SETUP_SPAM_DIR=$(sed '6p;d' "${SETUP_INSTALL_DIR}/path.dat" | tr -d '\r')
        SETUP_CALENDAR_DIR=$(sed '7p;d' "${SETUP_INSTALL_DIR}/path.dat" | tr -d '\r')
    fi

    # set to default if not defined in path.dat
    if [ "x${SETUP_CONFIG_DIR}" == "x" ]; then
        SETUP_CONFIG_DIR="${SETUP_INSTALL_DIR}/${SETUP_CONFIG_DIR_DEFAULT}"
    fi
    if [ "x${SETUP_SPAM_DIR}" == "x" ]; then
        SETUP_SPAM_DIR="${SETUP_INSTALL_DIR}/${SETUP_SPAM_DIR_DEFAULT}"
    fi
    if [ "x${SETUP_CALENDAR_DIR}" == "x" ]; then
        SETUP_CALENDAR_DIR="${SETUP_INSTALL_DIR}/${SETUP_CALENDAR_DIR_DEFAULT}"
    fi

    # remove possible trailing slashes
    SETUP_CONFIG_DIR=$(sed 's;/*$;;' <<< "${SETUP_CONFIG_DIR}")
    SETUP_CALENDAR_DIR=$(sed 's;/*$;;' <<< "${SETUP_CALENDAR_DIR}")
    SETUP_SPAM_DIR=$(sed 's;/*$;;' <<< "${SETUP_SPAM_DIR}")

    # if not full path, prepend installation directory
    SETUP_CONFIG_DIR=$(sed "s;^\([^/]\);${SETUP_INSTALL_DIR}\1;" <<< "${SETUP_CONFIG_DIR}")
    SETUP_CALENDAR_DIR=$(sed "s;^\([^/]\);${SETUP_INSTALL_DIR}\1;" <<< "${SETUP_CALENDAR_DIR}")
    SETUP_SPAM_DIR=$(sed "s;^\([^/]\);${SETUP_INSTALL_DIR}\1;" <<< "${SETUP_SPAM_DIR}")
}

# Function checks/creates installation directory
check_install_directory()
{
    if [ "$UPGRADE" == "0" ]; then
        # Install

        good "Creating ${SETUP_INSTALL_DIR} directory ..."

        if [ "x${SETUP_INSTALL_DIR}" == "x/" ]; then
            bad "Rejecting directory removal: ${SETUP_INSTALL_DIR}"
            exit 1
        fi
        rm -rf "$SETUP_INSTALL_DIR"
        mkdir -p "$SETUP_INSTALL_DIR"
        if [ $? -ne 0 ]; then
            bad "Failed! Check permissions."
            exit 1
        fi
    else
        # Upgrade

        # create directory for sure, user can delete installation directory
        # and left configuration file /etc/icewarp.conf on disk
        mkdir -p "$SETUP_INSTALL_DIR"
        if [ $? -ne 0 ]; then
            bad "Cannot create directory '$SETUP_INSTALL_DIR'. Check permissions."
            exit 1
        fi

        # remove docs from commtouch
        if [ -d "${SETUP_INSTALL_DIR}/spam/commtouch/docs" ]; then
            rm -rf "${SETUP_INSTALL_DIR}/spam/commtouch/docs"
        fi

        # move old commtouch folder to new "cyren" path if present
        if [ -d "${SETUP_INSTALL_DIR}/spam/commtouch" ]; then
            mv "${SETUP_INSTALL_DIR}/spam/commtouch" "${SETUP_INSTALL_DIR}/cyren"
        fi
        
        # backup old commtouch config files
        if [ -f "${SETUP_INSTALL_DIR}/cyren/ctasd.conf" ]; then
            mv "${SETUP_INSTALL_DIR}/cyren/ctasd.conf" "${SETUP_INSTALL_DIR}/cyren/ctasd.conf.bak"
        fi
        if [ -f "${SETUP_INSTALL_DIR}/cyren/ctasd-out.conf" ]; then
            mv "${SETUP_INSTALL_DIR}/cyren/ctasd-out.conf" "${SETUP_INSTALL_DIR}/cyren/ctasd-out.conf.bak"
        fi
        if [ -f "${SETUP_INSTALL_DIR}/cyren/ctipd.conf" ]; then
            mv "${SETUP_INSTALL_DIR}/cyren/ctipd.conf" "${SETUP_INSTALL_DIR}/cyren/ctipd.conf.bak"
        fi

        # remove voip services folder legacy contents
        if [ -d "${SETUP_INSTALL_DIR}/voip" ]; then
            rm -rf "${SETUP_INSTALL_DIR}/voip"
        fi

        # files are removed in final_cleanup()
        true
    fi
}

# Function extract setupfirst* archives
extract_first_customizations()
{
    if ! ls "${WORKDIR}"/setupfirst*.dat &> /dev/null; then
        return
    fi

    test_program_unzip
    
    good "Extracting pre-installation customization data ..."
    if [ -f "${WORKDIR}/setupfirst.dat" ]; then
        unzip -n -qq -^ -d "${SETUP_INSTALL_DIR}" "${WORKDIR}/setupfirst.dat"
    fi
    if [ -f "${WORKDIR}/setupfirstconfig.dat" ]; then
        unzip -n -qq -^ -d "${SETUP_CONFIG_DIR}" "${WORKDIR}/setupfirstconfig.dat"
    fi
    if [ -f "${WORKDIR}/setupfirstcalendar.dat" ]; then
        unzip -n -qq -^ -d "${SETUP_CALENDAR_DIR}" "${WORKDIR}/setupfirstcalendar.dat"
    fi
    if [ -f "${WORKDIR}/setupfirstspam.dat" ]; then
        unzip -n -qq -^ -d "${SETUP_SPAM_DIR}" "${WORKDIR}/setupfirstspam.dat"
    fi
}

# Function creates destination directory, extracts tar image
# and in case of installing, it copies default configuration
# files, so all not-generated files are on the place
# Parameters:
#   None
# Returns:
#   None
extract_package()
{
    OLD_SAVDID_DEFAULT_CONF_HASH=""
    NEW_SAVDID_DEFAULT_CONF_HASH=""
    
    good "Extracting data ..."
    if [ "$UPGRADE" == "1" ]; then
        # AS live doesn't like old files of itself
        if compgen -G "${SETUP_INSTALL_DIR}/cyren/*.bak" >/dev/null; then
            # There are config backup files, preserve them
            CONFIG_BACKUP=$(mktemp -d)
            mv "${SETUP_INSTALL_DIR}/cyren/"*.bak "$CONFIG_BACKUP"
            rm -rf "${SETUP_INSTALL_DIR}/cyren"
            mkdir -p "${SETUP_INSTALL_DIR}/cyren"
            mv "$CONFIG_BACKUP"/* "${SETUP_INSTALL_DIR}/cyren"
            rm -rf "$CONFIG_BACKUP"
        else
            rm -rf "${SETUP_INSTALL_DIR}/cyren"
        fi

        # Get md5sum of previous savdid.conf.default to see if needs to be regenerated
        if [ -f "${SETUP_INSTALL_DIR}/sophos/savdid.conf.default" ]; then
          OLD_SAVDID_DEFAULT_CONF_HASH=$(md5sum ${SETUP_INSTALL_DIR}/sophos/savdid.conf.default | cut -d ' ' -f 1)
        fi

        # Remove existing update.json file to force re-checking bases
        if [ -f "${SETUP_INSTALL_DIR}/sophos/update.json" ]; then
          rm -f "${SETUP_INSTALL_DIR}/sophos/update.json"
        fi
        
        # Backup legacy smartattach.dat as smartattach.dat.bkp
        if [ -f "${SETUP_CONFIG_DIR}/smartattach.dat" ]; then
          mv "${SETUP_CONFIG_DIR}/smartattach.dat" "${SETUP_CONFIG_DIR}/smartattach.dat.bkp"
        fi
    else
        # Create mail dir in advance
        MAIL_PATH="${SETUP_INSTALL_DIR}/mail"
        
        mkdir "${MAIL_PATH}"    
        chown "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${MAIL_PATH}"
        chmod 770 "${MAIL_PATH}"
    fi
    
    if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
            FREE_SPACE=$(df -Pm "$SETUP_INSTALL_DIR" | awk 'NR==2 {print $4}')
            
            if [ $FREE_SPACE -lt 3000 ]; then
                ask_with_confirmation "Not enough disk space in destination path, at least 3 GB needed. Continue?" "Y" "n"
                ANSWER=$?
                
                if [ $ANSWER -eq 2 ]; then
                    exit 1
                fi
            fi
    else
        good "Automatic installation mode - skipping free disk space check."      
    fi

    tar -xf "${WORKDIR}/${IMAGENAME}" -C "$SETUP_INSTALL_DIR"
    if [ $? -ne 0 ]; then
        bad "Failed, check user permissions and available disk space."
        exit 1
    fi
    
    if [ "${UPGRADE}" == "1" ]; then
        rm -rf "${SETUP_INSTALL_DIR}/lib/ssl"
        rm -rf "${SETUP_INSTALL_DIR}/lib64/ssl"
        rm -f "${IWS_INSTALL_DIR}/scripts/customssl.sh"
        rm -f "${IWS_INSTALL_DIR}/lib/libcrypto.so.10"
        rm -f "${IWS_INSTALL_DIR}/lib64/libcrypto.so.10"
        rm -f "${IWS_INSTALL_DIR}/lib/libssl.so.10"
        rm -f "${IWS_INSTALL_DIR}/lib64/libssl.so.10"

        rm -f "${IWS_INSTALL_DIR}/lib/libcrypto.so.1.0.0"
        rm -f "${IWS_INSTALL_DIR}/lib64/libcrypto.so.1.0.0"
        rm -f "${IWS_INSTALL_DIR}/lib/libssl.so.1.0.0"
        rm -f "${IWS_INSTALL_DIR}/lib64/libssl.so.1.0.0"

        rm -f "${IWS_INSTALL_DIR}/lib64/libcrypto.so.1.1"
        rm -f "${IWS_INSTALL_DIR}/lib64/libssl.so.1.1"

        rm -f "${IWS_INSTALL_DIR}/lib64/libcrypto.so.3"
        rm -f "${IWS_INSTALL_DIR}/lib64/libcrypto.so"
        rm -f "${IWS_INSTALL_DIR}/lib64/libssl.so.3"
        rm -f "${IWS_INSTALL_DIR}/lib64/libssl.so"
    fi

    # In case of installing, copy default configuration files
    if [ "$UPGRADE" == "0" ]; then
        # copy all configuration files

        # calendar
        if ! [ -f "${SETUP_INSTALL_DIR}/calendar/groupware.db" ]; then
            if [ -f "${SETUP_INSTALL_DIR}/calendar/default/db/groupware.db" ]; then
                cp -f "${SETUP_INSTALL_DIR}/calendar/default/db/groupware.db" "${SETUP_INSTALL_DIR}/calendar/groupware.db"
            fi
        fi

        # config
        [ -f "${SETUP_INSTALL_DIR}/config/content.xml" ] || cp -f "${SETUP_INSTALL_DIR}/config/default/content.xml" "${SETUP_INSTALL_DIR}/config/content.xml"
        [ -f "${SETUP_INSTALL_DIR}/config/imservices.dat" ] || cp -f "${SETUP_INSTALL_DIR}/config/default/imservices.dat" "${SETUP_INSTALL_DIR}/config/imservices.dat"
        [ -f "${SETUP_INSTALL_DIR}/config/servicebind.dat" ] || cp -f "${SETUP_INSTALL_DIR}/config/default/servicebind.dat" "${SETUP_INSTALL_DIR}/config/servicebind.dat"
        [ -f "${SETUP_INSTALL_DIR}/config/webserver.dat" ] || cp -f "${SETUP_INSTALL_DIR}/config/default/webserver.dat" "${SETUP_INSTALL_DIR}/config/webserver.dat"
        [ -f "${SETUP_INSTALL_DIR}/config/siprules.dat" ] || cp -f "${SETUP_INSTALL_DIR}/config/default/siprules.dat" "${SETUP_INSTALL_DIR}/config/siprules.dat"
        if ! [ -f "${SETUP_INSTALL_DIR}/config/accounts.db" ]; then
            if [ -f "${SETUP_INSTALL_DIR}/config/default/accounts.db" ]; then
                cp -f "${SETUP_INSTALL_DIR}/config/default/accounts.db" "${SETUP_INSTALL_DIR}/config/accounts.db"
            fi
        fi

        # antispam
        [ -f "${SETUP_INSTALL_DIR}/spam/antispam.db" ] || cp -f "${SETUP_INSTALL_DIR}/spam/default/db/antispam.db" "${SETUP_INSTALL_DIR}/spam/antispam.db"
        # local.cf needn't be present
        if [ -f "${SETUP_INSTALL_DIR}/spam/default/rules/local.cf" ] && [ ! -f "${SETUP_INSTALL_DIR}/spam/rules/local.cf" ]; then
            cp -f "${SETUP_INSTALL_DIR}/spam/default/rules/local.cf" "${SETUP_INSTALL_DIR}/spam/rules/local.cf"
        fi
        
        # ldap
        mkdir "${SETUP_INSTALL_DIR}/config/ldap"
        
        mv "${SETUP_INSTALL_DIR}/ldap/etc" "${SETUP_INSTALL_DIR}/config/ldap/etc"
        mv "${SETUP_INSTALL_DIR}/ldap/data" "${SETUP_INSTALL_DIR}/config/ldap/data"
        
        [ -f "${SETUP_INSTALL_DIR}/config/ldap/etc/ldap.conf" ] || cp -f "${SETUP_INSTALL_DIR}/config/ldap/etc/ldap.conf.default" "${SETUP_INSTALL_DIR}/config/ldap/etc/ldap.conf"
        [ -f "${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf" ] || cp -f "${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf.default" "${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf"
        
        #Sophos
        if ! [ -d "${SETUP_INSTALL_DIR}/config/sophos" ]; then
            mkdir "${SETUP_INSTALL_DIR}/config/sophos"
        fi  
        
        [ -f "${SETUP_INSTALL_DIR}/config/sophos/savdid.conf" ] || cp -f "${SETUP_INSTALL_DIR}/sophos/savdid.conf.default" "${SETUP_INSTALL_DIR}/config/sophos/savdid.conf"
    else
        # Re-set permissions for config directory (from path.dat and also hardcoded config (used by license for example))
        chown -R "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${SETUP_CONFIG_DIR}"
        chown -R "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${SETUP_INSTALL_DIR}/config"

        rm -f "${SETUP_INSTALL_DIR}/calendar/tz.ics"
        rm -f "${SETUP_INSTALL_DIR}/calendar/calendar.xml"
        rm -rf "${SETUP_INSTALL_DIR}/calendar/holidays"
        rm -rf "${SETUP_INSTALL_DIR}/calendar/weather"
        rm -rf "${SETUP_INSTALL_DIR}/calendar/doc"
        
        if ! [ -d "${SETUP_INSTALL_DIR}/config/ldap" ]; then
            mkdir "${SETUP_INSTALL_DIR}/config/ldap"
        fi
        
        if ! [ -d "${SETUP_INSTALL_DIR}/config/ldap/etc" ]; then
            mv "${SETUP_INSTALL_DIR}/ldap/etc" "${SETUP_INSTALL_DIR}/config/ldap/etc"

            # change original paths to new ones
            SED_SETUP_OLD_LDAP_DIR=$(echo "${SETUP_INSTALL_DIR}/ldap" | sed -e 's/\(\/\|\\\|&\)/\\&/g')
            SED_SETUP_NEW_LDAP_DIR=$(echo "${SETUP_CONFIG_DIR}/ldap" | sed -e 's/\(\/\|\\\|&\)/\\&/g')

            sed -i "s|${SED_SETUP_OLD_LDAP_DIR}|${SED_SETUP_NEW_LDAP_DIR}|g" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf"
        fi
        
        if ! [ -d "${SETUP_INSTALL_DIR}/config/ldap/data" ]; then
            mv "${SETUP_INSTALL_DIR}/ldap/data" "${SETUP_CONFIG_DIR}/ldap/data"
        fi
        
        # move purple/custom if exists to config/purple
        if [ -d "${SETUP_INSTALL_DIR}/purple/custom" ] && (! [ -d "${SETUP_CONFIG_DIR}/purple" ]); then
            cp -fr "${SETUP_INSTALL_DIR}/purple/custom" "${SETUP_INSTALL_DIR}/purple/custom.old"
            mv "${SETUP_INSTALL_DIR}/purple/custom" "${SETUP_CONFIG_DIR}/purple"
        fi
        
        # Get md5sum of new savdid.conf.default 
        NEW_SAVDID_DEFAULT_CONF_HASH=$(md5sum ${SETUP_INSTALL_DIR}/sophos/savdid.conf.default | cut -d ' ' -f 1)
        
        # If md5sums of new and old savdid.conf.default dont match, move live savdid.conf to savdid.conf.old to force configuration file refresh
        if [ "${OLD_SAVDID_DEFAULT_CONF_HASH}" != "" ] && [ "${OLD_SAVDID_DEFAULT_CONF_HASH}" != "${NEW_SAVDID_DEFAULT_CONF_HASH}" ] && [ -f "${SETUP_CONFIG_DIR}/sophos/savdid.conf" ]; then
            mv "${SETUP_CONFIG_DIR}/sophos/savdid.conf" "${SETUP_CONFIG_DIR}/sophos/savdid.conf.old"
        fi
    fi

    # In case of upgrading, check for missing configuration files
    # this is done in function check_settings_files()
}

get_system_timezone()
{
    if [ -f /etc/timezone ]; then
        TZ=`cat /etc/timezone`
    elif [ -h /etc/localtime ]; then
        TZ=`readlink /etc/localtime | sed "s/.*\/usr\/share\/zoneinfo\///"`
    else
        checksum=`md5sum /etc/localtime | cut -d' ' -f1`
        TZ=`find /usr/share/zoneinfo/ -type f -exec md5sum {} \; | grep "^$checksum" | sed "s/.*\/usr\/share\/zoneinfo\///" | grep -v ^posix | head -n 1`
    fi
    
    echo $TZ
}

upgrade_yoda_index()
{
    # Index was moved from indexpath to serverid/indexpath
    # Detect such situation and move it
    if [ "$UPGRADE" == "0" ]; then
        # Do not upgrade index on fresh installs
        return
    fi

    if [ "$SETUP_YODA_ENABLED" != "1" ]; then
        # Do not upgrade index when Yoda is disabled
        return
    fi
    
    if ! get_api_variable "system" "C_System_Services_Fulltext_Database_URL"; then
        bad "$GET_API_VARIABLE"
        exit 3
    fi
    YODA_IP="$GET_API_VARIABLE"
    if ! grep -q "127\.0\.0\.1" <<< "$YODA_IP" && ! grep -q "localhost" <<< "$YODA_IP"; then
        # Do not upgrade index when Yoda is not local
        return
    fi
    
    if ! get_api_variable "system" "C_System_Services_Fulltext_Database_Path"; then
        bad "$GET_API_VARIABLE"
        exit 3
    fi
    INDEX_PATH="$GET_API_VARIABLE"
    if [ "$INDEX_PATH" == "" ]; then
        # Beware of empty index path
        return
    fi
    
    if ! [ -d "$INDEX_PATH" ]; then
        # Do not upgrade index when it does not exist locally
        return
    fi
    
    if ! get_api_variable "system" "C_System_Services_Fulltext_ServerID"; then
        bad "$GET_API_VARIABLE"
        exit 3
    fi
    SERVER_ID="$GET_API_VARIABLE"
    if [ -d "${INDEX_PATH}/${SERVER_ID}" ]; then
        # Server ID directory already exists in the index directory
        return
    fi
    
    # Move the index
    mv "$INDEX_PATH" "$(dirname $INDEX_PATH)/${SERVER_ID}"
    mkdir "$INDEX_PATH"
    mv "$(dirname $INDEX_PATH)/${SERVER_ID}" "$INDEX_PATH"
}

fix_indexpause_example()
{
    if [ -f "${SETUP_INSTALL_DIR}/yoda/.indexpause_fixed" ]; then
        return
    fi

    sed -i 's/<indexpause enabled=\"true\">/<indexpause enabled=\"false\">/' "${SETUP_INSTALL_DIR}/yoda/yoda-scan.xml"
    touch "${SETUP_INSTALL_DIR}/yoda/.indexpause_fixed"
}

configure_yoda()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    test_mundi_dependencies
    install_yoda
    upgrade_yoda_index
    fix_indexpause_example
}

configure_reporting()
{
    if [ "${OPT_DOCKER}" == "1" ]; then
        return 0
    fi
    
    if [ -f "${SETUP_CONFIG_DIR}/setup_dont_ask_sentry_reporting" ]; then
        return 0
    fi
    
    if [ "${OPT_AUTO}" != "1" ]; then
        ask_with_confirmation "Enable online error reporting?" "Y" "n"
        ANSWER=$?
    else
        good "Automatic installation mode - enabling online error reporting."
        ANSWER=1
    fi    
    
    touch "${SETUP_CONFIG_DIR}/setup_dont_ask_sentry_reporting"
    
    if [ ${ANSWER} -eq 2 ]; then
        if ! set_api_variable "system" "C_System_SentryErrorReporting" "0" ; then
            bad "Failed to set C_System_SentryErrorReporting."
        fi
    fi
}

info_yoda()
{
    if [ "$OPT_DOCKER" == "1" ]; then
        return
    fi

    if ! get_api_variable "system" "C_Mail_SMTP_General_HostName"; then
        bad "$GET_API_VARIABLE"
        exit 3
    fi
    SERVER_HOSTNAME="$GET_API_VARIABLE"
    
    good "To configure YODA fulltext search service and related functionality please visit"
    good "following URL - https://${SERVER_HOSTNAME}/admin/#menu=server_settings&tab_server_settings=fulltext_search"
}

create_etc_icewarp_conf()
{
    DATE=`date`

    # print IceWarp Server configuration file
    DIR_CONFIG=`dirname ${SETUP_SERVER_CONF}`
    mkdir -p "${DIR_CONFIG}"
    if [ $? -ne 0 ]; then
        bad "Can not create directory: ${DIR_CONFIG}"
        exit 1
    fi
    
    #change permissions on /etc/icewarp/ to 755
    chmod 755 "${DIR_CONFIG}"

    touch "$SETUP_SERVER_CONF"
    if [ $? -ne 0 ]; then
        bad "Can not create file: ${SETUP_SERVER_CONF}"
        exit 1
    fi

    #change permissions on /etc/icewarp/icewarp.conf to 644
    chmod 644 "${SETUP_SERVER_CONF}"
    
    # Determine locale for boot start, prefer en_US.utf8, otherwise use current session locale
    if locale -a | grep -q "en_US.utf8"; then
        DEFAULT_LOCALE="en_US.utf8"
    else
        DEFAULT_LOCALE="${LANG}"
    fi

    echo "# IceWarp Server configuration file
# this file was generated by IceWarp installer on ${DATE}

# Server installation directory
# IWS_INSTALL_DIR=/path/to/server
IWS_INSTALL_DIR=\"${SETUP_INSTALL_DIR}\"

# Directory with native libraries for server
IWS_LIB_DIR=\"${SETUP_SERVER_LIBDIR}\"

# Platform this server is built for
IWS_PLATFORM=\"${SETUP_SERVER_PLATFORM}\"

# User to run server as
# IWS_PROCESS_USER=username
IWS_PROCESS_USER=\"${SETUP_INSTALL_USER}\"

# IceWarp Server version, used for upgrade process
# IWS_VERSION=number
IWS_VERSION=\"${SETUP_NEWVERSION}\"

# Previous version number, for information
IWS_OLDVERSION=\"${SETUP_OLDVERSION}\"

# Locale used when started at boot.
IWS_DEFAULT_LOCALE=\"${DEFAULT_LOCALE}\"

" > "$SETUP_SERVER_CONF"
}

create_default_settings()
{
    # In docker, settings are generated by entrypoint on first run
    if [ $OPT_DOCKER == "1" ]; then
        return
    fi

    # Generate default settings
    if ! [ -f "${SETUP_INSTALL_DIR}/config/settings.cfg" ]; then
        RESULT=$("${SETUP_INSTALL_DIR}/tool.sh" create settings)
        SUCCESS=$?
        if ! [ -z "$RESULT" ]; then
            if [ $? -eq 0 ]; then
                good $RESULT
            else
                bad $RESULT
            fi
        fi
    fi
}

configure_php_ini()
{
    # print settings to php.ini
    echo "
zend_extension = \"${SETUP_INSTALL_DIR}/php/ext/ioncube_loader_lin_8.2.so\"
" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    echo "
zend_extension = \"${SETUP_INSTALL_DIR}/php/ext/opcache.so\"
" >> "${SETUP_INSTALL_DIR}/php/php.ini"

    platform_ca_bundle_path
    CABUNDLE_PATH="$PLATFORM_RESULT"
    echo "
openssl.cafile=\"$CABUNDLE_PATH\"
" >> "${SETUP_INSTALL_DIR}/php/php.ini"

    echo "[IceWarp]
icewarp_sharedlib_path = \"${SETUP_INSTALL_DIR}/html/_shared/\"
extension_dir = \"${SETUP_INSTALL_DIR}/php/ext/\"
error_log = \"${SETUP_INSTALL_DIR}/logs/phperror.log\"
open_basedir_ignore = \"${SETUP_INSTALL_DIR}/\"
session.save_path = \"${SETUP_INSTALL_DIR}/php/tmp/\"
sendmail_path = \"${SETUP_INSTALL_DIR}/sendmail -i -t\"
upload_tmp_dir = \"${SETUP_INSTALL_DIR}/php/tmp/\"

; Set timezone to required value
date.timezone = \"$(get_system_timezone)\"
" >> "${SETUP_INSTALL_DIR}/php/php.ini"

    # write timeout settings if they are not in php.user.ini
    if ! [ -f "${SETUP_INSTALL_DIR}/php/php.user.ini" ] \
    || ! grep "icewarpphp.calendarfunctioncall_timeout" "${SETUP_INSTALL_DIR}/php/php.user.ini">/dev/null
    then
        echo "; Timeout of function calendarfunctioncall in milliseconds
icewarpphp.calendarfunctioncall_timeout = 0
" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    fi

    if ! [ -f "${SETUP_INSTALL_DIR}/php/php.user.ini" ] \
    || ! grep "icewarpphp.challengeresponsefunctioncall_timeout" "${SETUP_INSTALL_DIR}/php/php.user.ini">/dev/null
    then
        echo "; Timeout of function challengeresponsefunctioncall in milliseconds
icewarpphp.challengeresponsefunctioncall_timeout = 0
" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    fi

    if ! [ -f "${SETUP_INSTALL_DIR}/php/php.user.ini" ] \
    || ! grep "icewarpphp.apiobjectcall_persistent_timeout" "${SETUP_INSTALL_DIR}/php/php.user.ini">/dev/null
    then
        echo "; Timeout of function apiobjectcall_persistent in milliseconds
icewarpphp.apiobjectcall_persistent_timeout = 0
" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    fi
    
    if ! [ -f "${SETUP_INSTALL_DIR}/php/php.user.ini" ] \
    || ! grep "disable_functions.*=" "${SETUP_INSTALL_DIR}/php/php.user.ini">/dev/null
    then    
        echo "disable_functions = \"escapeshellarg,escapeshellcmd,exec,fp,fput,passthru,popen,posix_kill,posix_mkfifo,posix_setpgid,posix_setsid,posix_setuid,posix_setuid,proc_get_status,proc_nice,proc_open,proc_terminate,shell_exec,system,eval,chmod\"" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    fi

    # append php.user.ini
    if [ -f "${SETUP_INSTALL_DIR}/php/php.user.ini" ]; then
        echo -e "; Content of php.user.ini" >> "${SETUP_INSTALL_DIR}/php/php.ini"
        cat "${SETUP_INSTALL_DIR}/php/php.user.ini" >> "${SETUP_INSTALL_DIR}/php/php.ini"
        echo "" >> "${SETUP_INSTALL_DIR}/php/php.ini"
    fi
}

# Helper function for check_settings_files()
# Parameters:
#  $1 - source file name, copy from
#  $2 - destination file name to check and copy to
# Returns:
#  None
check_settings_file()
{
    FILE_NAME_SRC=$1
    FILE_NAME_DST=$2
    if [ ! -f "$FILE_NAME_DST" ]; then
        good "Configuration file ${FILE_NAME_DST} does not exist, copying from default."
        cp -f "$FILE_NAME_SRC" "$FILE_NAME_DST"
        if [ $? -ne 0 ]; then
            bad "Can not copy ${FILE_NAME_SRC} to ${FILE_NAME_DST}"
            exit 1
        fi
    fi
}

# Function to create accounts.db for the first install
setup_accounts_db()
{
    if [ "$UPGRADE" == "0" ] && [ $OPT_DOCKER != "1" ]; then
        if ! set_api_variable 'system' 'C_System_Storage_Accounts_ODBCConnString' 'config/accounts.db;;;;7;3'; then
            bad "$SET_API_VARIABLE"
            exit 3
        fi
    fi
}

# Function upgrades settings files from older versions to newest one
# load_special_paths() must be called before this function
# Paramters:
#  none
# Returns:
#  none
check_settings_files()
{
    # this function is needed only for upgrading
    if [ "$UPGRADE" == "0" ]; then
        return 0
    fi

    # extract calendar, config and spam directories into temporary folder
    DIR_TEMP=$(mktemp -d)

    tar -xf "${WORKDIR}/${IMAGENAME}" -C "$DIR_TEMP" calendar
    if [ $? -ne 0 ]; then
        bad "Can not extract calendar files from image"
        exit 1
    fi
    tar -xf "${WORKDIR}/${IMAGENAME}" -C "$DIR_TEMP" config
    if [ $? -ne 0 ]; then
        bad "Can not extract config files from image"
        exit 1
    fi
    tar -xf "${WORKDIR}/${IMAGENAME}" -C "$DIR_TEMP" spam
    if [ $? -ne 0 ]; then
        warn "Can not extract spam files from image"
        SPAM_EXCLUDED=true
    else
        SPAM_EXCLUDED=false
    fi

    # copy files from temp to destination
    cp -fr "${DIR_TEMP}/calendar/"* "$SETUP_CALENDAR_DIR"
    if [ $? -ne 0 ]; then
        bad "Can not copy calendar files"
        exit 1
    fi
    cp -fr "${DIR_TEMP}/config/"* "$SETUP_CONFIG_DIR"
    if [ $? -ne 0 ]; then
        bad "Can not copy config files"
        exit 1
    fi
    if ! $SPAM_EXCLUDED; then
        cp -fr "${DIR_TEMP}/spam/"* "$SETUP_SPAM_DIR"
        if [ $? -ne 0 ]; then
            bad "Can not copy spam files"
            exit 1
        fi
    fi

    # remove temporary directory
    rm -rf "$DIR_TEMP"

    # test if settings files exists, copy one from default

    # Files in calendar directory

    # calendar/groupware.db
    FNAME_SRC="${SETUP_CALENDAR_DIR}/default/db/groupware.db"
    FNAME_DST="${SETUP_CALENDAR_DIR}/groupware.db"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"

    # Files in config directory

    # config/content.xml
    FNAME_SRC="${SETUP_CONFIG_DIR}/default/content.xml"
    FNAME_DST="${SETUP_CONFIG_DIR}/content.xml"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    # config/imservices.dat
    FNAME_SRC="${SETUP_CONFIG_DIR}/default/imservices.dat"
    FNAME_DST="${SETUP_CONFIG_DIR}/imservices.dat"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    # config/servicebind.dat
    FNAME_SRC="${SETUP_CONFIG_DIR}/default/servicebind.dat"
    FNAME_DST="${SETUP_CONFIG_DIR}/servicebind.dat"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    # config/webserver.dat
    FNAME_SRC="${SETUP_CONFIG_DIR}/default/webserver.dat"
    FNAME_DST="${SETUP_CONFIG_DIR}/webserver.dat"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    # config/siprules.dat
    FNAME_SRC="${SETUP_CONFIG_DIR}/default/siprules.dat"
    FNAME_DST="${SETUP_CONFIG_DIR}/siprules.dat"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"

    if ! $SPAM_EXCLUDED; then
        # Files in spam directory

        # spam/antispam.db
        FNAME_SRC="${SETUP_SPAM_DIR}/default/db/antispam.db"
        FNAME_DST="${SETUP_SPAM_DIR}/antispam.db"
        check_settings_file "$FNAME_SRC" "$FNAME_DST"
        # spam/rules/local.cf
        FNAME_SRC="${SETUP_SPAM_DIR}/default/rules/local.cf"
        FNAME_DST="${SETUP_SPAM_DIR}/rules/local.cf"
        check_settings_file "$FNAME_SRC" "$FNAME_DST"
    fi

    # Files in ldap directory
    
    # config/ldap/etc/ldap.conf
    FNAME_SRC="${SETUP_INSTALL_DIR}/config/ldap/etc/ldap.conf.default"
    FNAME_DST="${SETUP_INSTALL_DIR}/config/ldap/etc/ldap.conf"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    # config/ldap/etc/slapd.conf
    FNAME_SRC="${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf.default"
    FNAME_DST="${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"
    
    # Sophos
    if ! [ -d "${SETUP_CONFIG_DIR}/sophos" ]; then
        mkdir "${SETUP_CONFIG_DIR}/sophos"
    fi  
    
    FNAME_SRC="${SETUP_INSTALL_DIR}/sophos/savdid.conf.default"
    FNAME_DST="${SETUP_CONFIG_DIR}/sophos/savdid.conf"
    check_settings_file "$FNAME_SRC" "$FNAME_DST"

    return 0
}

# Function extracts setupcustom* archives
extract_custom_customizations()
{
    if [ -f "${WORKDIR}/setupscript.dat" ]; then
        test_program_unzip
        unzip -o -qq -^ -d "${SETUP_INSTALL_DIR}/scripts" "${WORKDIR}/setupscript.dat"
    fi

    if ! ls "${WORKDIR}"/setupcustom*.dat &> /dev/null; then
        return
    fi

    test_program_unzip
    
    good "Extracting post-installation customization data ..."
    if [ -f "${WORKDIR}/setupcustom.dat" ]; then
        unzip -o -qq -^ -d "${SETUP_INSTALL_DIR}" "${WORKDIR}/setupcustom.dat"
    fi
    if [ -f "${WORKDIR}/setupcustomconfig.dat" ]; then
        unzip -o -qq -^ -d "${SETUP_CONFIG_DIR}" "${WORKDIR}/setupcustomconfig.dat"
    fi
    if [ -f "${WORKDIR}/setupcustomcalendar.dat" ]; then
        unzip -o -qq -^ -d "${SETUP_CALENDAR_DIR}" "${WORKDIR}/setupcustomcalendar.dat"
    fi
    if [ -f "${WORKDIR}/setupcustomspam.dat" ]; then
        unzip -o -qq -^ -d "${SETUP_SPAM_DIR}" "${WORKDIR}/setupcustomspam.dat"
    fi
}

# Function deals with kaspersky_dist directory - either replaces current sdk or removes it
update_kaspersky()
{
    if [ "$UPGRADE" == "0" ]; then
        rm -rf "${SETUP_INSTALL_DIR}/kaspersky"     # Just to be sure
        if [ -d "${SETUP_INSTALL_DIR}/kaspersky_dist" ]; then
            # Fresh install, place it there
            mv "${SETUP_INSTALL_DIR}/kaspersky_dist" "${SETUP_INSTALL_DIR}/kaspersky"
        fi
    elif [ -d "${SETUP_INSTALL_DIR}/kaspersky_dist" ]; then
        # Upgrade, compare sdk version
        PREVIOUS_KAVSDK_VERSION=$(<"${SETUP_INSTALL_DIR}/kaspersky/sdkversion.txt") 2>/dev/null
        CURRENT_KAVSDK_VERSION=$(<"${SETUP_INSTALL_DIR}/kaspersky_dist/sdkversion.txt")

        if [ "${PREVIOUS_KAVSDK_VERSION}" == "${CURRENT_KAVSDK_VERSION}" ]; then
            # always update libkavi.so and license files
            cp -f "${SETUP_INSTALL_DIR}/kaspersky_dist/${SETUP_SERVER_LIBDIR}/libkavi.so" "${SETUP_INSTALL_DIR}/kaspersky/${SETUP_SERVER_LIBDIR}"/
            cp -f "${SETUP_INSTALL_DIR}/kaspersky_dist/appinfo.kli" "${SETUP_INSTALL_DIR}/kaspersky/"
            cp -f "${SETUP_INSTALL_DIR}/kaspersky_dist/klicense.key" "${SETUP_INSTALL_DIR}/kaspersky/"
            # don't update the sdk itself
            rm -rf "${SETUP_INSTALL_DIR}/kaspersky_dist"
        else
            # update whole sdk directory
            rm -rf "${SETUP_INSTALL_DIR}/kaspersky"
            mv "${SETUP_INSTALL_DIR}/kaspersky_dist" "${SETUP_INSTALL_DIR}/kaspersky"
        fi
    fi
}

# Function replaces some setup variables and does edits in config files
# Paramters:
#  none
# Returns:
#  none
patch_settings_files()
{
    SED_SETUP_INSTALL_DIR=$(echo ${SETUP_INSTALL_DIR} | sed -e 's/\(\/\|\\\|&\)/\\&/g')
    SED_SETUP_CONFIG_DIR=$(echo ${SETUP_CONFIG_DIR} | sed -e 's/\(\/\|\\\|&\)/\\&/g')

    if [ -f "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml" ]; then
        # place installation path to kavehost.xml (by replacing env vars)
        rm -f "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml.new"
        sed "s/\${SETUP_INSTALL_DIR}/${SED_SETUP_INSTALL_DIR}/g" < "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml" > "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml.new"
        mv "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml.new" "${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml"
    fi

    # place installation path to ldap configuration files
    cp "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.new"
    sed -i "s/\${SETUP_CONFIG_DIR}/${SED_SETUP_CONFIG_DIR}/g" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.new" 
    sed -i "s/\${SETUP_INSTALL_DIR}/${SED_SETUP_INSTALL_DIR}/g" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.new"
    mv "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.new" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf"

    cp "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default.new"
    sed -i "s/\${SETUP_CONFIG_DIR}/${SED_SETUP_CONFIG_DIR}/g" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default.new" 
    sed -i "s/\${SETUP_INSTALL_DIR}/${SED_SETUP_INSTALL_DIR}/g" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default.new"
    mv "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default.new" "${SETUP_CONFIG_DIR}/ldap/etc/slapd.conf.default"
    
    # place icewarp process user to cyren config files
    if [ -f "${SETUP_INSTALL_DIR}/cyren/ctasd.conf" ]; then
        sed -i "s/\${SETUP_INSTALL_USER}/${SETUP_INSTALL_USER}/g" "${SETUP_INSTALL_DIR}/cyren/ctasd.conf"
    fi
    if [ -f "${SETUP_INSTALL_DIR}/cyren/ctasd-out.conf" ]; then
        sed -i "s/\${SETUP_INSTALL_USER}/${SETUP_INSTALL_USER}/g" "${SETUP_INSTALL_DIR}/cyren/ctasd-out.conf"
    fi
    if [ -f "${SETUP_INSTALL_DIR}/cyren/ctipd.conf" ]; then
        sed -i "s/\${SETUP_INSTALL_USER}/${SETUP_INSTALL_USER}/g" "${SETUP_INSTALL_DIR}/cyren/ctipd.conf"
    fi
    if [ -f "${SETUP_INSTALL_DIR}/cyren/ctipd.conf" ]; then
        sed -i "s:\${SETUP_INSTALL_DIR}:${SETUP_INSTALL_DIR}:g" "${SETUP_INSTALL_DIR}/cyren/ctipd.conf"
    fi
    
    # Sophos - control updates logdir and tempdir at start
    sed -i "s:\${SETUP_INSTALL_DIR}:${SETUP_INSTALL_DIR}:g" "${SETUP_CONFIG_DIR}/sophos/savdid.conf"
    sed -i "s:\${SETUP_LOG_DIR}:${SETUP_INSTALL_DIR}/logs/:g" "${SETUP_CONFIG_DIR}/sophos/savdid.conf"
    sed -i "s:\${SETUP_TEMP_DIR}:${SETUP_INSTALL_DIR}/temp/:g" "${SETUP_CONFIG_DIR}/sophos/savdid.conf"
}

# Function creates ldap root node
# Parameters:
#  none
# Returns:
#  none
create_slapd_root()
{
    if [ "$UPGRADE" == "1" ]; then
        return 0
    fi
    
    TMP_LDIF=$(mktemp)

    echo "dn: dc=root
objectClass: dcObject
objectClass: organization
dc: root
description: root
o: Organization
" > "${TMP_LDIF}"

    LD_LIBRARY_PATH="${SETUP_INSTALL_DIR}/lib64:${SETUP_INSTALL_DIR}/ldap/lib" "${SETUP_INSTALL_DIR}/ldap/sbin/slapadd" -f "${SETUP_INSTALL_DIR}/config/ldap/etc/slapd.conf" -l "${TMP_LDIF}" -b ""
    rm -f "${TMP_LDIF}"
}

# Creates SystemD IceWarp service
# Parameters:
#  $1 - Descriptive service title
#  $2 - short service name used for handling in system
create_icewarp_service()
{
    SERVICE_TITLE="$1"
    SERVICE_EXECUTABLE="$2"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SETUP_INSTALL_DIR}/${SERVICE_EXECUTABLE}
After=network-online.target
Wants=network-online.target

[Service]
Type=forking
StartLimitInterval=5
StartLimitBurst=10
EnvironmentFile=/etc/icewarp/icewarp.conf
EnvironmentFile=-${SETUP_INSTALL_DIR}/var/env-${SERVICE_EXECUTABLE}
ExecStart=/bin/bash -c \"source ${SETUP_INSTALL_DIR}/scripts/env && ${SETUP_INSTALL_DIR}/${SERVICE_EXECUTABLE}\"
ExecStopPost=${SETUP_INSTALL_DIR}/icewarpd.sh --finalize
AmbientCapabilities=CAP_NET_BIND_SERVICE
TimeoutSec=10

User=${SETUP_INSTALL_USER}

Restart=on-failure
RestartSec=30
LimitNOFILE=65535

[Install]
WantedBy=multi-user.target" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" ""
}

# Creates SystemD LDAP service
# Parameters:
#  None
# Returns:
#  None
create_slapd_service()
{
    SERVICE_TITLE="LDAP"
    SERVICE_EXECUTABLE="ldap"
    SERVICE_EXECUTABLE_PATH="${SETUP_INSTALL_DIR}/ldap/libexec/slapd"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SERVICE_EXECUTABLE_PATH}
PartOf=icewarp-control.service
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
StartLimitInterval=5
StartLimitBurst=10
EnvironmentFile=-${SETUP_CONFIG_DIR}/ldap/etc/env
ExecStart=/bin/bash -c \"source ${SETUP_INSTALL_DIR}/scripts/env && ${SERVICE_EXECUTABLE_PATH} -d \$IWS_LDAP_DEBUG_LEVEL -f \$IWS_LDAP_CONFIG_FILE -h \$IWS_LDAP_LISTEN_PORTS\"
AmbientCapabilities=CAP_NET_BIND_SERVICE
PIDFile=${SETUP_INSTALL_DIR}/var/slapd.pid
User=root
TimeoutSec=10
" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" ""
}

# Creates SystemD Kaspersky service
# Parameters:
#  None
# Returns:
#  None
create_kaspersky_service()
{
    SERVICE_TITLE="Kaspersky antivirus"
    SERVICE_EXECUTABLE="kaspersky"
    SERVICE_EXECUTABLE_PATH="${SETUP_INSTALL_DIR}/kaspersky/kavehost"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SERVICE_EXECUTABLE_PATH}
After=network-online.target
Wants=network-online.target

[Service]
Type=forking
StartLimitInterval=5
StartLimitBurst=10
ExecStart=/bin/bash -c \"source ${SETUP_INSTALL_DIR}/scripts/env && ${SETUP_INSTALL_DIR}/kaspersky/kavehost -c ${SETUP_INSTALL_DIR}/kaspersky/kavehost.xml\"
ExecReload=/bin/bash -c \"kill -USR1 \$MAINPID\"
PIDFile=${SETUP_INSTALL_DIR}/var/kavehost.pid
TimeoutSec=10

User=root" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" "reload"
}

# Creates SystemD Kaspersky service
# Parameters:
#  None
# Returns:
#  None
create_kaspersky_update_service()
{
    SERVICE_TITLE="Kaspersky antivirus update"
    SERVICE_EXECUTABLE="kaspersky-update"
    SERVICE_EXECUTABLE_PATH="${SETUP_INSTALL_DIR}/kasperskyupdater"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SERVICE_EXECUTABLE_PATH}
After=network-online.target
Wants=network-online.target

[Service]
Type=oneshot
ExecStart=/bin/bash -c \"source ${SETUP_INSTALL_DIR}/scripts/env && ${SERVICE_EXECUTABLE_PATH}\"
User=root" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" ""
}


# Creates SystemD Sophos service
# Parameters:
#  None
# Returns:
#  None
create_sophos_service()
{
    SERVICE_TITLE="Sophos antivirus"
    SERVICE_EXECUTABLE="sophos"
    SERVICE_EXECUTABLE_PATH="${SETUP_INSTALL_DIR}/sophos/savdid"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SERVICE_EXECUTABLE_PATH}
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
StartLimitInterval=5
StartLimitBurst=10
ExecStart=/bin/bash -c \"LD_LIBRARY_PATH=${SETUP_INSTALL_DIR}/sophos/ ${SETUP_INSTALL_DIR}/sophos/savdid -c ${SETUP_CONFIG_DIR}/sophos/savdid.conf\"
PIDFile=${SETUP_INSTALL_DIR}/var/sophos.pid
TimeoutSec=10

User=${SETUP_INSTALL_USER}" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" ""
}

# Function creates /etc/sudoers.d/icewarp_requiretty_override if needed
# Parameters:
#  none
# Returns:
#  none
create_requiretty_override()
{
    # Sudoers not needed when running under root
    if [ "$SETUP_INSTALL_USER" == "root" ]; then
        return
    fi
    
    if [ "$SUDO_REQUIRETTY_IS_SET" -eq 0 ]; then
        return
    fi
    
    echo "Defaults:%${SETUP_INSTALL_USER} !requiretty" > "/etc/sudoers.d/icewarp_requiretty_override"
}

# Creates SystemD Bitdefender service
# Parameters:
#  None
# Returns:
#  None
create_bitdefender_service()
{
    SERVICE_TITLE="Bitdefender"
    SERVICE_EXECUTABLE="bitdefender"
    SERVICE_EXECUTABLE_PATH="${SETUP_INSTALL_DIR}/bitdefender/bdamserver"
    SERVICE_FILE="/etc/systemd/system/icewarp-${SERVICE_EXECUTABLE}.service"
    
    echo "[Unit]
Description=IceWarp ${SERVICE_TITLE} service
ConditionFileIsExecutable=${SERVICE_EXECUTABLE_PATH}
PartOf=icewarp-control.service
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
StartLimitInterval=5
StartLimitBurst=10
ExecStart=${SETUP_INSTALL_DIR}/bitdefender/bdamserver -c ${SETUP_CONFIG_DIR}/bitdefender/bdamserver.conf
WorkingDirectory=${SETUP_INSTALL_DIR}/bitdefender
AmbientCapabilities=CAP_NET_BIND_SERVICE
PIDFile=${SETUP_INSTALL_DIR}/var/bdamserver.pid
User=${SETUP_INSTALL_USER}
TimeoutSec=10
" > "${SERVICE_FILE}"

    chmod 644 "${SERVICE_FILE}"
    chown root:root "${SERVICE_FILE}"
    
    # Allow to start/stop/restart the service via systemctl 
    configure_service_sudo "icewarp-${SERVICE_EXECUTABLE}" "${SERVICE_EXECUTABLE}" ""
}

# Function installs service to system
# Parameters:
#  none
# Returns:
#  none
install_service()
{
    if [ "$OPT_DOCKER" == "1" ]; then
        return
    fi

    # check if merak service is installed
    service merak status &> /dev/null
    if [ $? -eq 0 ]; then
        good "Service 'merak' was found installed in system"
        good "which is now obsolete and renamed to 'icewarp' service."
        if [ "${OPT_AUTO}" != "1" ]; then
            ask_with_confirmation "Delete service 'merak'?" "Y" "n"
            ANSWER=$?
        else
            good "Automatic installation mode - deleting service merak."
            ANSWER=1
        fi
        if [ ${ANSWER} -eq 1 ]; then
            good "Removing 'merak' system service ..."
            platform_remove_service merak
            rm -f "/etc/init.d/merak"
        fi
    fi

    if [ "$UPGRADE" == "1" ]; then
        # Set legacy icewarpd.sh logs owners
        if ! get_api_variable "system" "C_System_Storage_Dir_LogPath"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi
        chown -h "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "$GET_API_VARIABLE/icewarpd" 2>/dev/null
    fi

    # check if system service is installed
    good "Checking if IceWarp Server is added as system service ..."
    good ""
    service "$SETUP_INSTALL_SERVICE_NAME" status > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        good "Removing legacy IceWarp Server system service ..."
        platform_remove_service "$SETUP_INSTALL_SERVICE_NAME"
        rm -f "/etc/init.d/${SETUP_INSTALL_SERVICE_NAME}"
    fi

    # Install systemd services
    create_icewarp_service "Control" "control"
    create_icewarp_service "POP3/IMAP" "pop3"
    create_icewarp_service "Groupware" "cal"
    create_icewarp_service "SMTP" "smtp"
    create_icewarp_service "IM" "im"
    create_requiretty_override
    configure_icewarpd_sudo_all_services
    
    create_slapd_service
    create_kaspersky_service
    create_kaspersky_update_service
    create_sophos_service
    if [ -d "${SETUP_INSTALL_DIR}/bitdefender" ]; then
        create_bitdefender_service
    fi
    
    systemctl daemon-reload  
       
    case "$OPT_STARTONBOOT" in
        "1") ANSWER=1
        ;;
        "0") ANSWER=0
        ;;
        *)
            if [ "${OPT_AUTO}" != "1" ]; then
                ask_with_confirmation "Do you want to start IceWarp Server on system startup?" "Y" "n"
                ANSWER=$?
            else
                good "Automatic installation mode - making IceWarp Server to start on system startup."
                ANSWER=1
            fi
        ;;
    esac
    
    if [ ${ANSWER} -eq 1 ]; then
        good "Setting service to start on system startup..."
        
        SERVICES=( "icewarp-control" "icewarp-pop3" "icewarp-cal" "icewarp-smtp" "icewarp-im")
        for SERVICE in "${SERVICES[@]}"; do
            systemctl enable "${SERVICE}"

            if [ $? -ne 0 ]; then
                bad "Failed to enable service ${SERVICE}."
            fi
        done            
    else
        good "IceWarp Server will not start automatically on system startup."
        
        SERVICES=( "icewarp-control" "icewarp-pop3" "icewarp-cal" "icewarp-smtp" "icewarp-im")
        for SERVICE in "${SERVICES[@]}"; do
            systemctl disable "${SERVICE}"

            if [ $? -ne 0 ]; then
                bad "Failed to disable service ${SERVICE}."
            fi
        done     
    fi
    
    
    return 0
}

configure_missing_antivirus()
{
    if [ "$UPGRADE" == "0" ] && [ ! -d "${SETUP_INSTALL_DIR}/kaspersky" ]; then
        # Fresh install without AV, configure AV module settings
        "${SETUP_INSTALL_DIR}/tool.sh" set system C_AV_General_IntegratedAV 0 >/dev/null
        "${SETUP_INSTALL_DIR}/tool.sh" set system C_AV_Misc_DisableInternal 1 >/dev/null
    fi
}

# Function upgrades server data and tables from previous version to latest
# Parameters:
#  None
# Returns:
#  None
upgrade_server_data()
{
    if [ "$OPT_DOCKER" == "1" ]; then
        return 0
    fi

    if [ "$UPGRADE" != "1" ] && ! [ -f "${SETUP_INSTALL_DIR}/scripts/script.php" ]; then
        schedule_antivirus_update
        return 0
    fi
    
    # run upgrade tool if upgrading
    if [ "$UPGRADE" == "1" ]; then
        good ""
        good "It is possible now to upgrade data to the current version."
        good "This process includes possible customization script execution"
        good "To do this, IceWarp Server must be started."
        good "Note: You can run upgrade process later by command"
        good "      in ${SETUP_INSTALL_DIR}: ./upgrade.sh _old_version_"
        good ""
    else
        good ""
        good "It is needed to execute installation customization script."
        good "To do this, IceWarp Server must be started."
        good "Note: You can execute the script manually later by command"
        good "      in ${SETUP_INSTALL_DIR}: ./upgrade.sh"
        good ""
    fi
    
    if [ "${OPT_AUTO}" != "1" ];  then
        ask_with_confirmation "Start services and perform the operation now?" "Y" "n"
        ANSWER=$?
    else
        good "Automatic installation mode - performing upgrade."
        ANSWER=1
    fi
    if [ ${ANSWER} -eq 1 ]; then
        good "Refreshing additional licenses ..."
        "${SETUP_INSTALL_DIR}/tool.sh"  set system c_refreshextralicenses true > /dev/null
    
        good "Starting IceWarp Server ..."
        "${SETUP_INSTALL_DIR}/icewarpd.sh" --start > /dev/null
        if [ $? -ne 0 ]; then
            bad "Cannot start IceWarp Server"
            exit 1
        fi

        good "Performing upgrade tasks ..."
        sleep 5
        wait_groupware 300
        
        good "Performing customizations tasks ..."
        if [ "$UPGRADE" == "1" ]; then
            "${SETUP_INSTALL_DIR}/upgrade.sh" "$SETUP_OLDVERSION" > /dev/null
        else
            "${SETUP_INSTALL_DIR}/upgrade.sh" > /dev/null
        fi
        
        if [ $? -eq 0 ]; then
            sleep 10
            good "Data upgrade and customization completed successfully"
        else
            bad "Data upgrade or customization problem!"
        fi

        good "Restarting IceWarp Server ..."
        "${SETUP_INSTALL_DIR}/icewarpd.sh" --stop > /dev/null
        if [ $? -ne 0 ]; then
            bad "Cannot restart IceWarp Server, please do the restart manually"
        fi

        schedule_antivirus_update
        
        "${SETUP_INSTALL_DIR}/icewarpd.sh" --start > /dev/null
        if [ $? -ne 0 ]; then
            bad "Cannot restart IceWarp Server, please do the restart manually"
        fi
    else
        schedule_antivirus_update
    fi
}

# Install dependencies, disable document preview components, if crucial tools are missing
configure_docpreview()
{
    test_program_documentpreview
    test_fonts

    if [ "${CONVERSION_DOC_PDF}" == "0" ]; then
        "${SETUP_INSTALL_DIR}/tool.sh" set system C_GW_DocumentPDFConversion 0 >/dev/null
    fi

    if [ "${CONVERSION_PDF_IMAGE}" == "0" ]; then
        "${SETUP_INSTALL_DIR}/tool.sh" set system C_GW_PDFImageConversion 0 >/dev/null
    fi
}

# Schedule antivirus update
schedule_antivirus_update()
{
    touch "${SETUP_INSTALL_DIR}/var/updateav"
    chown -h "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${SETUP_INSTALL_DIR}/var/updateav"
}

# delete dot files
# Parameters:
#  $1 - folder to search in
# Returns:
#  none
delete_dot_files()
{
    if [ -f "$1/.gitignore" ]; then
        rm -f "$1/.gitignore"
    fi

    if [ -f "$1/.gitattributes" ]; then
        rm -f "$1/.gitattributes"
    fi

    if [ -f "$1/.npmignore" ]; then
        rm -f "$1/.npmignore"
    fi
}

# Delete old things before extracting package
delete_before_install()
{
    if [ "${UPGRADE}" == "1" ]; then
        rm -rf "${SETUP_INSTALL_DIR}/html/admin/client/languages"
        
        WEBCLIENT_SKINS=("banner" "bigger" "biggerrtl" "default")
        
        for ITEM in ${WEBCLIENT_SKINS[*]}
        do
            FILES="${SETUP_INSTALL_DIR}/html/webmail/client/skins/${ITEM}/"
    
            rm -rf "${FILES}css/style.css" &>/dev/null
            rm -rf "${FILES}templates" &>/dev/null
            rm -rf "${FILES}objects/objects.xml" &>/dev/null
        done
        
        if [ -d "${SETUP_INSTALL_DIR}/html/webmail/tools/" ]; then
            rm -rf "${SETUP_INSTALL_DIR}/html/webmail/tools/"
        fi
        
        delete_dot_files "${SETUP_INSTALL_DIR}/html/_shared"
        delete_dot_files "${SETUP_INSTALL_DIR}/html/activesync"
        delete_dot_files "${SETUP_INSTALL_DIR}/html/admin"
        delete_dot_files "${SETUP_INSTALL_DIR}/html/webmail"
        
        run_liccheck_command "clean"
        if [ $? -eq 0 ]; then
            RESTART_RECOMMENDED="1"
        fi

        # libzip is only for php, was moved to php/lib
        rm -f "${SETUP_INSTALL_DIR}/lib64/libzip"*
        rm -f "${SETUP_INSTALL_DIR}/php/ext/libicewarpphp.so"
        rm -f "${SETUP_INSTALL_DIR}/php/ext/ioncube_loader_lin_7.3.so"
        rm -f "${SETUP_INSTALL_DIR}/php/ext/ioncube_loader_lin_8.1.so"

        rm -f "${SETUP_INSTALL_DIR}/install/notifier-setup.exe"
        rm -f "${SETUP_INSTALL_DIR}/install/icewarp-configurator.dmg"
    fi
}

change_user_owner()
{
    # All files are installed OK, change permissions
    good "Changing permissions ..."
    if [ "${UPGRADE}" == "1" ]; then
        if ! get_api_variable "system" "C_System_Storage_Dir_MailPath"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi
        MAIL_PATH=$(sed 's/\/$//' <<< "$GET_API_VARIABLE")

        if ! get_api_variable "system" "C_System_Tools_AutoArchive_Path"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi
        ARCHIVE_PATH=$(sed 's/\/$//' <<< "$GET_API_VARIABLE")

        if ! get_api_variable "system" "C_System_Services_Fulltext_Database_Path"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi
        YODAINDEX_PATH=$(sed 's/\/$//' <<< "$GET_API_VARIABLE")

        if [ "${ARCHIVE_PATH:0:1}" != "/" ]; then
            ARCHIVE_PATH="${SETUP_INSTALL_DIR}/${ARCHIVE_PATH}"
        fi
        find "${SETUP_INSTALL_DIR}" -path "${MAIL_PATH}" -prune -o -path "${ARCHIVE_PATH}" -prune -o -print0 | xargs -0 chown -h "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}"
        if [ -d "${YODAINDEX_PATH}" ]; then
            find "${YODAINDEX_PATH}" -print0 | xargs -0 chown -h "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}"
        fi
    else
        # Fresh installation
        chown -h -R "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${SETUP_INSTALL_DIR}"
        if [ $? -ne 0 ]; then
            bad "Failed! Check user permissions on ${SETUP_INSTALL_DIR}"
            "Setup continues ..."
        fi
    fi
}

final_cleanup()
{
    # binaries
    rm -f "${SETUP_INSTALL_DIR}/merakd"
    rm -f "${SETUP_INSTALL_DIR}/startd"
    rm -f "${SETUP_INSTALL_DIR}/stopd"
    rm -f "${SETUP_INSTALL_DIR}/restartd"
    # scripts
    rm -f "${SETUP_INSTALL_DIR}/merakd.sh"
    # sockets
    rm -f "${SETUP_INSTALL_DIR}/var/icewarpd.sock"
    rm -f "${SETUP_INSTALL_DIR}/var/phpsocket"
    # previous configuration
    rm -f "${SETUP_INSTALL_DIR}/config/icewarp.conf"
    rm -f "${SETUP_INSTALL_DIR}/config/merak.conf"
    # libraries needed for 32bit server on 64bit platform
    if  [ "${OS_PLATFORM}" != "x86_64" ] && [ -f "${SETUP_INSTALL_DIR}/lib/liblist_x86.txt" ]; then
      while read line; do
          rm -f "${SETUP_INSTALL_DIR}/lib/${line}"
      done < "${SETUP_INSTALL_DIR}/lib/liblist_x86.txt"
    fi
    rm -f "${SETUP_INSTALL_DIR}/lib/liblist_x86.txt"
    
    #remove legacy icewarpd
    rm -f "${SETUP_INSTALL_DIR}/icewarpd"

    # files introduced by mistake in previous versions
    rm -rf "${SETUP_INSTALL_DIR}/config/_webmail/images/.git"

    # desktop applications not distributed anymore
    rm -f "${SETUP_INSTALL_DIR}/install/desktop-setup.msi"

    # moved to SETUP_INSTALL_DIR/lib64
    rm -f "${SETUP_INSTALL_DIR}/php/lib/"libgd.so*
}


# $1 - wait start time
# $2 - max wait in seconds
# returns true when wait elapsed
wait_elapsed()
{
    local current_unix=$(date -u "+%s")
    local diff=$(($current_unix-$1-$2))

    [ $diff -gt 0 ]
}

# $1 - timeout in seconds
wait_groupware()
{
    good "Waiting for groupware DB ..."
    local gw_ready=false
    local wait_start_time=$(date -u "+%s")
    local upd_status
    local max_wait=$1

    local gw_upgrade_started=false
    while ! wait_elapsed $wait_start_time $max_wait; do
        if ! get_api_variable "system" "C_GW_UpgradeStatus"; then
            bad "$GET_API_VARIABLE"
            exit 3
        fi

        upd_status=$(head -n 1 <<< "$GET_API_VARIABLE")
        if [ "$upd_status" != "0" ]; then
            gw_upgrade_started=true
            break
        fi

        sleep 1
    done

    if $gw_upgrade_started; then
        if [ "$upd_status" == "1" ]; then
            good "Upgrading groupware DB ..."

            while true; do
                local msg=$(tail -n '+2' <<< "$GET_API_VARIABLE")
                if ! [ -z $msg ]; then
                    multi_good "$msg"
                fi

                if [ "$upd_status" == "2" ]; then
                    break
                fi

                sleep 1

                if ! get_api_variable "system" "C_GW_UpgradeStatus"; then
                    bad "$GET_API_VARIABLE"
                    exit 3
                fi

                upd_status=$(head -n 1 <<< "$GET_API_VARIABLE")
            done
        fi
    fi

    if [ "$upd_status" == "2" ]; then
        good "Groupware DB is ready"

        good "Waiting for groupware ..."
        local wait_start_time=$(date -u "+%s")
        while ! wait_elapsed $wait_start_time $max_wait; do
            if ! get_api_variable "system" "C_GW_IsAvailable"; then
                bad "$GET_API_VARIABLE"
                exit 3
            fi
            if [ "$GET_API_VARIABLE" == "1" ]; then
                gw_ready=true
                break
            fi

            sleep 1
        done
    fi

    if $gw_ready; then
        good "Groupware is ready"
    else
        warn "Waiting for groupware timed out. Please check IceWarp server state."
        echo "                 It is possible the initial setup won't fully succeed."

        if [ "${OPT_AUTO}" == "1" ]; then
            exit 3
        fi

        echo "                 Press ENTER to continue"
        read -s
    fi
    echo ""
}

start_server_wait_groupware()
{
	if [ "${UPGRADE}" != "0" ] || [ "$OPT_DOCKER" != "0" ]; then
		return
	fi
	
	good "Starting IceWarp Server ..."
	"${SETUP_INSTALL_DIR}/icewarpd.sh" --start > /dev/null
	if [ $? -ne 0 ]; then
		warn "Cannot start IceWarp Server"
	else
		good "IceWarp Server started"
		echo ""
	fi
	
	wait_groupware 60
}

create_initial_data()
{
    if [ "$UPGRADE" != "0" ]; then
        return
    fi

    INIT_DOMAIN="icewarpdemo.com"
    INIT_USER="admin@icewarpdemo.com"
    INIT_PASSWORD="admin"

    good "Initial domain and user with administrator privileges can be created now."
    
    if [ "${OPT_AUTO}" != "1" ]; then
        ask_with_confirmation "Create initial domain and user?" "Y" "n"
        if [ $? -ne 1 ]; then
            return
        fi
    fi

    "${SETUP_INSTALL_DIR}/tool.sh" "create" "domain" "$INIT_DOMAIN" "d_description" "Initial domain" "d_adminemail" "$INIT_USER" &> /dev/null
    if [ $? -ne 0 ]; then
        bad "Error creating domain $INIT_DOMAIN"
    else
        "${SETUP_INSTALL_DIR}/tool.sh" "set" "system" "c_accounts_policies_pass_enable" "0" &> /dev/null
        "${SETUP_INSTALL_DIR}/tool.sh" "create" "account" "$INIT_USER" "u_name" "System and domain administrator" "u_password" "$INIT_PASSWORD" "u_admin" "1" &> /dev/null
        if [ $? -ne 0 ]; then
            bad "Error creating user $INIT_USER"
        else
            good "Initial domain and user was created:"
            good "Domain: $INIT_DOMAIN"
            good "User: $INIT_USER"
            good "Password: $INIT_PASSWORD"
        fi
        "${SETUP_INSTALL_DIR}/tool.sh" "set" "system" "c_accounts_policies_pass_enable" "1" &> /dev/null
    fi
}

# Function sets all smartdiscover items to use given hostname
# Parameters: $1 - hostname
set_hostname_to_smartdiscover()
{
    if [ "x$1" == "x" ]; then return; fi

    "${SETUP_INSTALL_DIR}/tool.sh" set system c_activesync_url "https://$1/Microsoft-Server-ActiveSync" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_syncml_url "https://$1/syncml/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_webdav_url "https://$1/webdav/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_gw_webdavurl "https://$1/webdav/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_webmail_url "https://$1/webmail/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_webadmin_url "https://$1/admin/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_freebusy_url "https://$1/freebusy/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_gw_freebusyurl "https://$1/freebusy/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_internetcalendar_url "https://$1/calendar/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_smsservice_url "https://$1/sms/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_as_spamchallengeurl "https://$1/reports/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_install_url "https://$1/install/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_teamchat_api_url "https://$1/teamchatapi/" &>/dev/null
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_collaboration_api_url "https://$1/collaboration/" &>/dev/null    
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_conference_api_url  "https://$1/conference/" &>/dev/null    
}

new_installation_wizard()
{
    if [ "$UPGRADE" != "0" ]; then
        return
    fi
 
    HOSTNAME_OK=false
    DOMAIN_OK=false
    ADMIN_OK=false
    
    while ! $DOMAIN_OK || ! $HOSTNAME_OK || ! $ADMIN_OK; do
        if ! $HOSTNAME_OK; then
            INIT_HOSTNAME=$(uname -n)
            echo ""
            good "Enter the name of your server. This is the hostname you will use to access your server"
            good "from the Internet. You should setup the DNS as explained in the documentation."
            getparam "Hostname" "$INIT_HOSTNAME"
            if [ "x$PARAM" != "x" ]; then
                if [ "${PARAM}" != "${INIT_HOSTNAME}" ]; then
                    echo ""
                    warn "You have entered another hostname, than is set in the system.
        Please note, that system hostname has to be configured properly
        and has to be resolvable by DNS, or some components won't work."
                fi
                INIT_HOSTNAME="$PARAM"
            fi
        fi

        if ! $DOMAIN_OK; then
            INIT_DOMAIN="icewarpdemo.com"
            echo ""
            getparam "Enter the name of primary domain" "$INIT_DOMAIN"
            if [ "x$PARAM" != "x" ]; then
                INIT_DOMAIN="$PARAM"
            fi
        fi
        
        if ! $ADMIN_OK; then
            INIT_USERNAME="admin"
            INIT_PASSWORD=""
            CONFIRM_PASSWORD=""

            echo ""
            good "Enter the username and password for the administrator account."
            good "Choose a strong password to avoid account hijacking."
            getparam "Username" "$INIT_USERNAME"
            if [ "x$PARAM" != "x" ]; then
                INIT_USERNAME="$PARAM"
            fi
            getpassword "Password"
            if [ "x$PARAM" != "x" ]; then
                INIT_PASSWORD="$PARAM"
            fi
            getpassword "Confirm password"
            if [ "x$PARAM" != "x" ]; then
                CONFIRM_PASSWORD="$PARAM"
            fi            
        fi

        ADMIN_EMAIL="${INIT_USERNAME}@${INIT_DOMAIN}"
        
        # Create domain
        if ! $DOMAIN_OK; then
            "${SETUP_INSTALL_DIR}/tool.sh" create domain "${INIT_DOMAIN}" D_AdminEmail "${ADMIN_EMAIL}" D_Postmaster "${POSTMASTER_ALIASES}" D_SharedRoster >"${HELPER_WIZARD_FILE}" 2>&1
            if [ $? -ne 0 ]; then
                echo ""
                bad "Error creating domain ${INIT_DOMAIN}"
                bad "$(<"${HELPER_WIZARD_FILE}")"
            else
                DOMAIN_OK=true
            fi
        fi

        # Set hostname
        if ! $HOSTNAME_OK; then
            "${SETUP_INSTALL_DIR}/tool.sh" set system C_Mail_SMTP_General_HostName "${INIT_HOSTNAME}" >"${HELPER_WIZARD_FILE}" 2>&1
            if [ $? -ne 0 ]; then
                echo ""
                bad "Error setting hostname ${INIT_HOSTNAME}"
                bad "$(<"${HELPER_WIZARD_FILE}")"
            else
                HOSTNAME_OK=true
            fi

            # Set IM services names
            sed "s/icewarpdemo.com/${INIT_HOSTNAME}/" < "${SETUP_INSTALL_DIR}/config/imservices.dat" > "${SETUP_INSTALL_DIR}/config/imservices.dat.new"
            mv "${SETUP_INSTALL_DIR}/config/imservices.dat.new" "${SETUP_INSTALL_DIR}/config/imservices.dat"
            chown -h "${SETUP_INSTALL_USER}:${SETUP_INSTALL_GROUP}" "${SETUP_INSTALL_DIR}/config/imservices.dat"

            # Set smartdiscover
            set_hostname_to_smartdiscover "${INIT_HOSTNAME}"
        fi

        # Create admin account
        if ! $ADMIN_OK; then
            if [ "${INIT_PASSWORD}" != "${CONFIRM_PASSWORD}" ]; then
                bad "Entered passwords don't match"
            else
                "${SETUP_INSTALL_DIR}/tool.sh" create account "${ADMIN_EMAIL}" U_Alias "${INIT_USERNAME}" U_Mailbox "${INIT_USERNAME}" U_Password "${INIT_PASSWORD}" U_Name "${INIT_USERNAME}" U_Admin 1 >"${HELPER_WIZARD_FILE}" 2>&1
                ADMIN_CREATE_RESULT=$?
                if [ $ADMIN_CREATE_RESULT -ne 0 ]; then
                    echo ""
                    bad "Creating administrator account failed"
                    case $ADMIN_CREATE_RESULT in
                        4)  bad "Account already exists!"
                            ;;
                        5)  bad "Password violates password policy!"
                    echo "        Password cannot contain username or alias, must be at least 6 characters long"
                    echo "        and must contain at least 1 numeric and 1 alpha character."
                            ;;
                        *)  bad "Account save problem!"
                            bad "$(<"${HELPER_WIZARD_FILE}")"
                            ;;
                    esac
                else
                    ADMIN_OK=true
                fi
            fi
        fi
        
        rm -f "${HELPER_WIZARD_FILE}"

        if ! $DOMAIN_OK || ! $HOSTNAME_OK || ! $ADMIN_OK; then
            echo ""
            ask_with_confirmation "Retry?" "Y" "n"
            if [ $? -ne 1 ]; then
                echo ""
                warn "Please use a wizard or other tool to configure the server"
                return
            fi
        fi
    done
    
    # Make default self signed domain cert and Lets Encrypt
    good "Generating default certificates ..."
    "${SETUP_INSTALL_DIR}/tool.sh" set system c_CreateDefaultCertificates "${INIT_DOMAIN}" > /dev/null 2>&1

    if [ $? -eq 0 ]; then
        good "Certificates successfully added."
    else
        bad "Certificates couldn't be created."
    fi
}

rejoice()
{
    good ""
    if [ "$UPGRADE" == "0" ]; then
        good "IceWarp Server was successfully installed."
    else
        good "IceWarp Server was successfully upgraded."
    fi
    good ""
    
    if [ "$RESTART_RECOMMENDED" == "1" ]; then
        echo ""
        warn "Please restart the machine to finish the upgrade."
        echo ""
    fi
}

license_questions()
{
    if [ "${UPGRADE}" == "0" ] && [ "$OPT_DOCKER" == "0" ]; then
        if [ "x${OPT_LICENSE}" != "x" ]; then
            good "Activating license..."
            "${SETUP_INSTALL_DIR}/tool.sh" set system c_onlinelicense "${OPT_LICENSE}"
            if [ $? -eq 0 ]; then
                good "License activation successful."
            else
                bad "License activation error. Please check server error log for further information."
                exit 1
            fi
        elif [ "${OPT_AUTO}" != "1" ]; then
            good "IceWarp Server is fully functional only with registered license."
            good "If you already have a license (trial or purchased) you can activate it now."
            good "If you don't have it yet, you can register a trial for 30 days."
            echo ""
            ask_with_confirmation "Do you want to run wizard and obtain trial license now?" "Y" "n"
            if [ $? -eq 1 ]; then
                "${SETUP_INSTALL_DIR}/scripts/wizard.sh" "action" "license_obtain_trial"
            else
                ask_with_confirmation "Do you want to run wizard and activate license you already have?" "Y" "n"
                if [ $? -eq 1 ]; then
                    "${SETUP_INSTALL_DIR}/scripts/wizard.sh" "action" "license_online"
                else
                    echo ""
                    good "Please use wizard.sh or other tool to activate your license."
                    good "Your 30 day evaluation has started now."
                    echo ""
                    warn "IceWarp Server is not fully functional in evaluation mode without a license!"
                fi
            fi
        fi
    fi
    good ""
}

copyright()
{
    good ""
    good "IceWarp Server installer"
    good "(c) 1999-2025 IceWarp Ltd. "
    good ""
}

getscriptparams()
{
    case $1 in
    "-h"|"--help")
        copyright
        good ""
        good "Usage: $0 [-f|--fast] [--allow-missing-architecture] [-a|--auto] [--install-dir <installation directory>] [--user <user>] [--license <order id>] [--start-on-boot 0|1]"
        good ""
        good "Options:"
        good "    -f or --fast"
        good "         Skip new installation wizard, create default admin."
        good "    --allow-missing-architecture"
        good "         Allow installation on multiarch distros even if i386 architecture is not present."
        good "    -a or --auto"
        good "         Skip all interactive questions, use default values"
        good "    -d or --docker"
        good "         Docker automated installation with predefined options. It is recommended to use this option only as standalone."
        good "    --install-dir"
        good "         Specify installation directory. This is /opt/icewarp by default in auto mode."
        good "    --user"
        good "         Specify an user under which the IceWarp Server will run. Default user in auto mode is root."
        good "    --license"
        good "         Specify an order ID of your license. If license is not specified in auto mode, IceWarp server is installed without any license."
        good "    --start-on-boot"
        good "         Set to 0 to not set IW to start on system boot. Set to 1 for the opposite. By default, user is asked or automatic mode sets start on boot to 1."
        good "    --skip-api-check"
        good "         Skips cloud API availability check"
        good "    --skip-ffmpeg-check"
        good "         Skips FFMPEG presence check and installation"
        good "    --api-check"
        good "         Do cloud API availability check only"
        good ""
        good "This script provides functionality for installing or upgrading IceWarp Server."
        good "Default action is to install but if the following conditions are met,"
        good "only upgrade will be performed:"
        good "  1. entered destination directory exists"
        good "  2. file INSTALL_DESTINATION_DIRECTORY/config/merak.conf or"
        good "     file INSTALL_DESTINATION_DIRECTORY/config/icewarp.conf exists"
        good ""
        good "Install mode extracts all files and creates configuration files"
        good "from default configuration file directories."
        good ""
        good "Upgrade mode:"
        good "  1. replaces all files which are not configuration files"
        good "     (i.e. files not modified by running server)"
        good "  2. recreates default configuration files that do not exists:"
        good "       ./calendar/groupware.db"
        good "       ./config/webserver.dat"
        good "       ./spam/antispam.dat"
        good "       ./spam/rules/local.cf"
        good ""
        good "[END]"
    ;;
    *)
        parse_cmdline $*
        configure_docker_install
        accept_license
        display_log_info
        #
        # part 1.
        #
        # detect needed programs and dependencies
        # allow to install them, if they are missing
        test_correct_osdist
        test_correct_platform
        test_selinux
        detect_install_options
        test_unwanted_downgrade
        test_platform_specifics
        test_dependencies
        test_required_programs
        
        #
        # part 2.
        #
        # get installation options, extract server files
        #
        ask_questions
        unpack_liccheck_dependencies
        check_upgrade_license
        check_upgrade_dashboard
        check_install_options
        confirm_install_options
        check_upgrade_webdocuments
        check_legacy_documentpreview                
        check_server_is_running
        check_running_php
        check_yoda        
        check_resolvable_hostname
        load_special_paths
        check_install_directory
        delete_before_install
        extract_first_customizations
        extract_package
        create_etc_icewarp_conf
        #
        # part 3.
        #
        # configure settings files
        #
        configure_php_ini
        check_settings_files            # checks if setting file is missing, if yes replace with default one
        extract_custom_customizations   # extract partner customizations
        update_kaspersky
        patch_settings_files            # replace variables with install path etc.
        create_slapd_root
        install_service                 # register services at systemd, allow autostart
        change_user_owner               # chmod to icewarp user (if not root)
        create_default_settings         # create initial settings.cfg (must be done after change_user_owner)
        setup_accounts_db
        configure_missing_antivirus
        configure_docpreview
        configure_lame    
        configure_webdocuments
        configure_yoda
        configure_reporting
        upgrade_server_data        
        #
        # Part 4.
        #
        # server is fully installed and configured, delete not needed things
        final_cleanup
        start_server_wait_groupware
        #
        # Part 5.
        #
        # create initial domain and user
        if [ "${OPT_AUTO}" != "1" ] && [ "${OPT_DOCKER}" != "1" ]; then
            if [ "$OPT_FAST_INSTALL_MODE" -eq "0" ]; then
                new_installation_wizard
            else
                create_initial_data
            fi
        else
            if [ "$OPT_FAST_INSTALL_MODE" -eq "1" ]; then
                create_initial_data
            fi
        fi
        
        rejoice
        license_questions
        info_yoda
    ;;
    esac
}

# this install script can be executed only by user with root privileges
if [ "x${RUNNING_UID}" != "x0" ]; then
    bad "This install script can be executed by root user only."
    exit 1
fi

DATE=$(date)

detect_os_distribution

getscriptparams $* 2>> "$INSTALL_ERROR_LOG"

exit 0
