#!/bin/bash
#
# IceWarp Server uninstallation script
# Copyright (c) 2008-2025 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com 
#
# file: uninstall.sh - uninstall script
# version: 1.0
#

# variables
EMPTY_VALUE="UNKNOWN"

USE_COLORS="0"

SETUP_OLDVERSION="$EMPTY_VALUE"
SETUP_INSTALL_DIR="$EMPTY_VALUE"
SETUP_INSTALL_USER="$EMPTY_VALUE"

SETUP_INSTALL_SERVICE_NAME="icewarp"

UNINSTALL_LOG=~/icewarp-uninstall.log
UNINSTALL_ERROR_LOG=~/icewarp-uninstall-error.log

RUNNING_UID=$(id -u)

# cmdline flags
OPT_FORCE_UNINSTALL="0"
OPT_DELETE_DATA="0"

# platform specific declarations
source $(dirname $0)"/scripts/platform"

# init

SELF=$0
WORKDIR=$(dirname $0)

if [ "$TERM" == "xterm" ] || [ "$TERM" == "xterm-color" ] || [ "$TERM" == "screen" ]; then
  GOOD=$'\e[0;40;32m'
  BAD=$'\e[0;40;31m'
  NORMAL=$'\e[0m'
  WARN=$'\e[0;40;93m'
  HILITE=$'\e[0;40;97m'
  BRACKET=$'\e[0;40;39m'
else
  GOOD=$'\e[0;40;32m'
  BAD=$'\e[0;40;31m'
  NORMAL=$'\e[0m'
  WARN=$'\e[0;40;33m'
  HILITE='\e[4;40;30m'
  BRACKET=${NORMAL}
fi

good()
{
    echo -e "${GOOD}**${NORMAL}\t$*"
}

bad()
{
    echo -e "${BAD}**${NORMAL}\tError: $*"
}

warn()
{
    echo -e "${WARN}**${NORMAL}\tWarning: $*"
}

hilite()
{
    echo -e "${GOOD}**${HILITE}\t$*${NORMAL}"
}

get_api_variable()
{
    CMD_STDOUT=$("${IWS_INSTALL_DIR}/tool.sh" get "$1" "$2" | sed 's/^[^:]*: //')
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

copy_if_not_exists()
{
    if ! [ -f "$2" ]; then
        cp -f "$1" "$2"
    fi
}

# Displayes information about stdout log and stderr log placement
display_log_info()
{
    good "Uninstaller log is available in ${UNINSTALL_LOG}"
    good "Uninstaller error log is available in ${UNINSTALL_ERROR_LOG}"
    good ""
}

# Sets variables depending on command line flags
parse_cmdline()
{
    OPT_FORCE_UNINSTALL="0"
    OPT_DELETE_DATA="0"
    for i in $*; do
        case `echo ${i} | tr a-z A-Z` in
            --FORCE)
                OPT_FORCE_UNINSTALL="1"
                ;;
            --DELETEDATA)
                OPT_DELETE_DATA="1"
                ;;
        esac
    done
}

# Tests if programs needed to uninstall
# IceWarp Server are installed
test_programs()
{
    if ! platform_check_service_management_tool; then
        bad "Program ${PLATFORM_SERVICE_MANAGEMENT_TOOL_NAME} seems not to be installed."
        exit 1
    fi
    sed --version &> /dev/null
    if [ $? -ne 0 ]; then
        bad "Error testing sed. Program sed seems not to be installed."
        exit 1
    fi
}

# Ask question with confirmation
# Parameters:
# $1 - question
# $2 - first choice, it is choosed if user enters only empty string
# $3 - second choice, $4 third choice...
# Return:
# 0 on error - unknown string entered
# 1 if first choice was selected
# 2 if second choice was selected
# 3 if third choice...
ask_with_confirmation()
{
    if [ -z "$1" ] || [ -z "$2" ] || [ -z "$3" ]; then
        return 0
    fi
    LOWER_CHOICE_1=$(echo "$2" | tr "A-Z" "a-z")
    LOWER_CHOICE_2=$(echo "$3" | tr "A-Z" "a-z")

    # make options string
    local OPTIONS="${GOOD}**${NORMAL}\t${1} ${BRACKET}[${HILITE}${2}${NORMAL}"
    for ARG in "${@:3}"; do 
        OPTIONS+="/${ARG}"
    done
 
    echo -e -n "${OPTIONS}${BRACKET}]${NORMAL}: "
    read PARAM
    if [ -z "$PARAM" ]; then 
        return 1
    fi

    PARAM=$(echo "$PARAM" | tr "A-Z" "a-z")
    declare -i I=1
    for ARG in "${@:2}"; do  # arguments 2 through n (i.e. 3 args starting at number 2)
        local LOWER_CHOICE=$(echo "$ARG" | tr "A-Z" "a-z")
        if [ "$PARAM" == "$LOWER_CHOICE" ]; then
            return $I
        fi
        ((I++))
    done

    return 0
}

# Function asks, if server should be really uninstalled
confirm_uninstallation()
{
    if [ "${OPT_FORCE_UNINSTALL}" == "0" ]; then
        echo -e -n "${GOOD}**${NORMAL}\tAre you sure to completely remove all files\n  \tand components of this program? (yes/no): "

        while true; do
            read ANSWER
            case `echo ${ANSWER} | tr a-z A-Z` in
                "YES")
                    good ""
                    good "Starting IceWarp server uninstallation!"
                    return
                    ;;
                "NO")
                    exit 1
                    ;;
                *)
                    bad "Please answer either yes or no."
                    ;;
            esac
        done 
    fi
}

# Function checks if configuration file exists and sets variables
# Parameters
# none
# Returns:
#  0 - if all variables SETUP_INSTALL_DIR, SETUP_INSTALL_USER are detected
# != 0 - if some of variables or non of variables SETUP_INSTALL_DIR, SETUP_INSTALL_USER are detected
preconf()
{
    # reset variables for old configuration
    InstallDir=""
    User=""
    Version=""

    if [ -f "$1" ]; then
        source "$1"

        # check for new variables first

        # check for install dir
        if [ "x${IWS_INSTALL_DIR}" != "x" ]; then
            SETUP_INSTALL_DIR="$IWS_INSTALL_DIR"
        fi

        # check for install user
        if [ "x${IWS_PROCESS_USER}" != "x" ]; then
            SETUP_INSTALL_USER="$IWS_PROCESS_USER"
        fi

        # check for installed version
        if [ "x${IWS_VERSION}" != "x" ]; then
            SETUP_OLDVERSION="$IWS_VERSION"
        fi

        # check for old variables

        # check for install dir
        if [ "x${InstallDir}" != "x" ]; then
            SETUP_INSTALL_DIR="$InstallDir"
        fi

        if [ "x${User}" != "x" ]; then
            SETUP_INSTALL_USER="$User"
        fi

        if [ "x${Version}" != "x" ]; then
            SETUP_OLDVERSION="$Version"
        fi
    fi
}

detect_install_options()
{
    CONFIG_FILE="/etc/icewarp/icewarp.conf"

    # for safety initialize variables
    SETUP_INSTALL_DIR="$EMPTY_VALUE"
    SETUP_INSTALL_USER="$EMPTY_VALUE"
    SETUP_OLDVERSION="$EMPTY_VALUE"

    if [ -f "$CONFIG_FILE" ]; then
        preconf "$CONFIG_FILE"
        # check for set values
        if [ "x${SETUP_INSTALL_DIR}" != "x${EMPTY_VALUE}" ] &&
           [ "x${SETUP_INSTALL_USER}" != "x${EMPTY_VALUE}" ] &&
           [ "x${SETUP_OLDVERSION}" != "x${EMPTY_VALUE}" ]; then
           # all values was set
           return 0
        fi
    fi

    # configuration file not found, or some values was not set
    # reset variables
    SETUP_INSTALL_DIR="$EMPTY_VALUE"
    SETUP_INSTALL_USER="$EMPTY_VALUE"
    SETUP_OLDVERSION="$EMPTY_VALUE"
    return 1
}

# Functions check for services and 3rd party processes executed by IceWarp server
# When detected, user has a choice to kill them or exit
check_server_is_running()
{
    SERVICES=( "${SETUP_INSTALL_DIR}/icewarpd" "${SETUP_INSTALL_DIR}/control" "${SETUP_INSTALL_DIR}/cal" "${SETUP_INSTALL_DIR}/im" "${SETUP_INSTALL_DIR}/pop3" "${SETUP_INSTALL_DIR}/smtp" "${SETUP_INSTALL_DIR}/purple/purpleserv" "ctasd.bin" "ctipd.bin" "${SETUP_INSTALL_DIR}/ldap/libexec/slapd" "${SETUP_INSTALL_DIR}/kasperskyupdater" "${SETUP_INSTALL_DIR}/kaspersky/kavehost" "kavscanner" "${SETUP_INSTALL_DIR}/voip/echo-voicemail-service.jar" "yoda" "yoda-scan")

    # At first check, if any services are running
    SERVER_RUNS=1
    for SERVICE in "${SERVICES[@]}"; do
        ps ax | grep "${SERVICE}" | grep -v "grep">/dev/null
        if [ $? -eq 0 ]; then
            SERVER_RUNS=0
        fi
    done

    # If so, kill them
    if [ $SERVER_RUNS -eq 0 ]; then
        for SERVICE in "${SERVICES[@]}"; do
            ps ax | grep "${SERVICE}" | grep -v "grep" | awk '{ print ($1); }' | while read pid; do
                kill -9 $pid
            done
        done
        
        good "The rest of running services have been killed"
        echo ""
    fi
}

# Function checks for running php processes
# If detected, they are killed
# Parameters:
#  none
# Returns:
#  none
check_running_php()
{
    # Detect, if previous PHP runs
    PHP_DETECTED="0"
    if ps ax | grep "${SETUP_INSTALL_DIR}/php/php" | grep -v "grep">/dev/null; then
        PHP_DETECTED="1"
    fi
    if ps ax | grep "${SETUP_INSTALL_DIR}/scripts/phpd.sh" | grep -v "grep">/dev/null; then
        PHP_DETECTED="1"
    fi
    if ps ax | grep "php-fpm" | grep -v "grep">/dev/null; then
        PHP_DETECTED="1"
    fi
    
    if [ "${PHP_DETECTED}" == "1" ]; then
        # Kill detected phps
        ps ax | grep "${SETUP_INSTALL_DIR}/php/php" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
            kill -9 $pid
        done
        
        # Kill phpd.sh script
        ps ax | grep "${SETUP_INSTALL_DIR}/scripts/phpd.sh" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
            kill -9 $pid
        done
        
        # Kill php-fpm
        ps ax | grep "php-fpm" | grep -v "grep" | awk '{ print($1); }' | while read pid; do
            kill -9 $pid
        done
        
        good "Detected PHP processes have been killed"
        echo ""
    fi
}

# Function stops the server
# It first tries to do it normally, then kills the rest
stop_server()
{
    good ""
    good "Press ENTER to stop the server"
    read -s
    good "Stopping server..."
    "${SETUP_INSTALL_DIR}/icewarpd.sh" --stop >/dev/null
    if [ $? -ne 0 ]; then
        warn "There were some problems during stopping icewarp server"
        echo ""
    fi

    # Kill the rest
    check_server_is_running
    check_running_php

    echo ""
    good "IceWarp server stopped"
}

# Inform license server about uninstallation
uninstall_license()
{
    # TODO - not needed now
    return
}

# Function uninstalls service from system
# Paramters:
#  none
# Returns:
#  none

uninstall_systemctl_service()
{
    systemctl disable "$1"
    rm -rf "/etc/systemd/system/${1}.service"
}

uninstall_service()
{
    # add/remove merakd as system service
    SERVICE_INSTALLED="0"
    SERVICE_STARTED="0"

    # check if system service is installed
    good "Removing IceWarp Server system service"
    good ""
    service "$SETUP_INSTALL_SERVICE_NAME" status > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        platform_remove_service "$SETUP_INSTALL_SERVICE_NAME"
        rm -f "/etc/init.d/${SETUP_INSTALL_SERVICE_NAME}"
    fi
    
    
    uninstall_systemctl_service "icewarp-control"
    uninstall_systemctl_service "icewarp-cal"
    uninstall_systemctl_service "icewarp-pop3"
    uninstall_systemctl_service "icewarp-smtp"
    uninstall_systemctl_service "icewarp-im"
    
    uninstall_systemctl_service "icewarp-ldap"
    uninstall_systemctl_service "icewarp-kaspersky"
    uninstall_systemctl_service "icewarp-kaspersky-update"
    uninstall_systemctl_service "icewarp-sophos"
    uninstall_systemctl_service "icewarp-bitdefender"

    systemctl daemon-reload
    systemctl reset-failed
}

uninstall_sudo()
{
    rm -f /etc/sudoers.d/icewarp_*
}

# Function asks and deletes all files or files except data and configuration
delete_files()
{
    if [ "${OPT_DELETE_DATA}" == "0" ]; then
        ask_with_confirmation "Do you want to remove also all data and configuration?" "N" "y"
        if [ $? -ne 1 ]; then
            OPT_DELETE_DATA="1"
        fi
    fi
  
    if [ "${OPT_DELETE_DATA}" == "1" ]; then
        rm -rf "${SETUP_INSTALL_DIR}"
    else
        DATA_FOLDERS="calendar,config,mail,logs,spam,status"

        DIR_BCK=$(pwd);
        cd "${SETUP_INSTALL_DIR}"

        # create feed for egrep
        FOLDERS_REGEX=$(echo "${DATA_FOLDERS}" | sed 's/\([a-z]\+\)/^\1$/g' | sed 's/,/|/g')

        # do deletion
        ls | while read DIR; do
            if ! echo "${DIR}" | egrep "${FOLDERS_REGEX}" >/dev/null; then
                rm -rf "${DIR}"
            fi
        done

        # restore current directory
        cd "${DIR_BCK}"
    fi

    # Delete configuration in /etc
    rm -rf "/etc/icewarp"
}

# Function just shows message about successfull instaltion
done_message()
{
    good ""
    good "Uninstall completed."

    if [ "${OPT_DELETE_DATA}" == "1" ]; then
        good ""
        good "Note, that if you set external storage directory for mails, logs or temp,"
        good "then these were not deleted."
    fi

    good ""
}

# Function displays script copyright
copyright()
{
    echo ""
    good ""
    good "IceWarp Server uninstaller"
    good "(c) 1999-2025 IceWarp Ltd. "
    good ""
}

# Checks if YODA is running, stops and uninstalls existing service
# $1 - service executable name
# $2 - config file name
# $3 - service name
check_yoda_service()
{
    PID_FILE="${SETUP_INSTALL_DIR}/yoda/run/$1.pid"
    EXECUTABLE_FILE="${SETUP_INSTALL_DIR}/yoda/$1"

    if [ ! -f "$EXECUTABLE_FILE" ]; then
        #file does not exist
        return
    fi

    if [ -f "$PID_FILE" ]; then
        YODA_PID=$(<"$PID_FILE")

        kill -0 "$YODA_PID" &> /dev/null
        
        #stop yoda service before upgrade
        if [ $? -eq 0 ]; then
            good "Stopping existing YODA $3 service."
            
            "${EXECUTABLE_FILE}" --service stop --config "${SETUP_INSTALL_DIR}/yoda/$2" &>/dev/null
            
            if [ $? -ne 0 ]; then
                bad "Failed to stop YODA service."
            fi
        fi    
    fi
    
    # uninstall yoda service
    "${EXECUTABLE_FILE}" --service uninstall --config "${SETUP_INSTALL_DIR}/yoda/$2" &>/dev/null
    
    if [ $? -ne 0 ]; then
        bad "Failed to uninstall YODA $3 service."
    fi
}
                                                      
# Checks if is YODA running, stops and uninstalls existing service
check_yoda()
{
    check_yoda_service "yoda" "config.xml" "Server"
    check_yoda_service "mundi" "mundi.xml" "Document conversion"
    check_yoda_service "yoda-scan" "yoda-scan.xml" "Indexer"
}

getscriptparams()
{
    case $1 in
    "-h"|"--help")
        copyright
        good ""
        good "Usage: $0 [--force] [--deletedata]"
        good ""
        good "Options: --force"
        good "         Skips uninstallation confirmations"
        good "         Data deleteion depends on --deletedata option. If not presen, data is not deleted"
        good ""
        good "         --deletedata"
        good "         Deletes user data, doesn't ask user"
        good ""   
        good "This script provides functionality for automatically uninstalling IceWarp Server."
        good "User data and configuration can be preserved or removed with the software."
        good ""
        good "[END]"
    ;;
    *)
        copyright
        display_log_info
        parse_cmdline $*
        test_programs

        detect_install_options
        if [ $? -ne 0 ]; then
            bad "Cannot find IceWarp server installation, exiting"
            exit 1
        fi
        # simple error prevention
        if [ "${SETUP_INSTALL_DIR}" == "/" ]; then
            bad "Wrongly detected setup installation directory"
            exit 2
        fi

        confirm_uninstallation
        stop_server
        check_yoda         
        uninstall_license
        uninstall_service
        uninstall_sudo
        delete_files
        done_message
    ;;
    esac
}

# this uninstall script can be executed only by user with root privileges
if [ "x${RUNNING_UID}" != "x0" ]; then
    bad "This uninstall script can be executed by root user only."
    exit 1
fi

DATE=$(date)

touch "$UNINSTALL_LOG"
touch "$UNINSTALL_ERROR_LOG"

echo "UNINSTALLATION STARTED ON ${DATE}" > "$UNINSTALL_LOG"
echo "UNINSTALLATION STARTED ON ${DATE}" > "$UNINSTALL_ERROR_LOG"
getscriptparams $* 2>> "$UNINSTALL_ERROR_LOG" | tee -a "$UNINSTALL_LOG"
echo "UNINSTALLATION FINISHED ON ${DATE}" >> "$UNINSTALL_LOG"
echo "UNINSTALLATION FINISHED ON ${DATE}" >> "$UNINSTALL_ERROR_LOG"

exit 0
