#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2025 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: db_sqlite.sh - wizard menu script
#

source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"

# Init variables
DB_NAME=""
ODBC_CONN_STRING=""

# Function asks user for ODBC connection parameters
# Sets: DB_NAME with user entered informations
#       ODBC_CONN_STRING to constructed ODBC connection string
# Returns: != 0 in case of error
#          0 if OK
#
# ODBC connection string
# format is:
# DB_NAME;;;;syntax;driver
# - DB_NAME is string
# - syntax is:
#     0 - default
#     1 - MS Access
#     2 - MS SQL
#     3 - MySQL
#     4 - InterBase
#     5 - Oracle
#     6 - FireBird
#     7 - SQLite
# - driver is:
#     0 - default
#     1 - ODBC
#     2 - MySQL
#     3 - SQLite
#     4 - FireBird
fnc_get_odbc_conn_string()
{
    good ""
    good "Please enter SQLite connection parameters:"
    good "Note: Database file name can be specified with full path"
    good "      or with path relative to installation directory."
    good ""
    getparam "Database file name:"
    DB_NAME=$PARAM

    if [ -z "$DB_NAME" ] || [ "$DB_NAME" == "" ]; then
        warn "Missing database name"
        return 1
    fi

    # construct ODBC string, use SQLite syntax and driver
    ODBC_CONN_STRING="${DB_NAME};;;;7;3"
    return 0
}

# Common function does:
# - asks for ODBC conn string
# - switches storage mode for selected set: accounts, groupware, antispam
# - creates tables in database
# Parameters:
# $1 - number for tool command "create tables"
#      0 - accounts
#      2 - groupware
#      3 - antispam
# $2 - string with type: "accounts", "groupware", "antispam"
#      used in informative prints
#
# Storage type:
# 0 - Professional file system
# 1 - Standard file system
# 2 - ODBC
fnc_db_sqlite_common()
{
    SERVICE_TYPE="$1"
    SERVICE_NAME="$2"

    good ""
    ask_with_confirmation "Setup ${SERVICE_NAME} storage to SQLite?" "Y" "n"
    if [ $? -ne 1 ]; then
        warn "Setup ${SERVICE_NAME} storage cancelled."
        return 0
    fi

    fnc_get_odbc_conn_string
    if [ $? -ne 0 ]; then
        return 1
    fi

    good "Setting database connection..."
    # Accounts
    if [ "$SERVICE_TYPE" -eq "0" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_System_Storage_Accounts_ODBCConnString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi
    # GroupWare
    if [ "$SERVICE_TYPE" -eq "2" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_GW_ConnectionString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi
    # Antispam
    if [ "$SERVICE_TYPE" -eq "3" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_AS_Challenge_ConnectionString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi

    good ""
    ask_with_confirmation "Create tables for ${SERVICE_NAME}?" "Y" "n"
    if [ $? -eq 1 ]; then
        good "Creating tables..."
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "create" "tables" "${SERVICE_TYPE}" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ $EXITCODE -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
        return 1
        fi
        else
            good "Tables were not created."
        # continue, because user can have tables already created
    fi

    good "Storage switched to SQLite."

    return 0
}

fnc_db_sqlite_accounts()
{
    fnc_db_sqlite_common "0" "Accounts"
    if [ $? -ne 0 ]; then
        return 1
    fi

    good ""
    ask_with_confirmation "Convert data from file system to database?" "Y" "n"
    if [ $? -ne 1 ]; then
        return 0
    fi

    good "Converting data from file system to database..."
    run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "set" "storage" "1"
    EXITCODE=$?
    if [ ${EXITCODE} -ne 0 ]; then
        bad "Error calling tool, error code ${EXITCODE}"
        return 1
    fi
    good "Data converted."

    return 0
}

fnc_db_sqlite_groupware()
{
    fnc_db_sqlite_common "2" "GroupWare"
    if [ $? -ne 0 ]; then
        return 1
    fi
    return 0
}

fnc_db_sqlite_antispam()
{
    fnc_db_sqlite_common "3" "Antispam"
    if [ $? -ne 0 ]; then
        return 1
    fi
    return 0
}

SCRIPT_NAME=`basename "$0"`
RES=0
case $SCRIPT_NAME in
    "db_sqlite_accounts")
        fnc_db_sqlite_accounts
        RES=$?
        ;;
    "db_sqlite_groupware")
        fnc_db_sqlite_groupware
        RES=$?
        ;;
    "db_sqlite_antispam")
        fnc_db_sqlite_antispam
        RES=$?
        ;;
    *)
        echo "Script called for unknown operation: ${SCRIPT_NAME}"
        RES=1
        ;;
esac

exit $RES

