#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2025 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: icewarpd.sh - icewarpd script
#

SERVER_VERSION="IceWarp Server 14.2.0.10 RHEL8 x64"

# include configuration file
source /etc/icewarp/icewarp.conf
if [ $? -ne 0 ]; then
    echo "Can not find configuration file icewarp.conf"
    exit 255
fi

# Ensure some locale is set
if [ "x${LANG}" == "x" ]; then
    export LANG="${IWS_DEFAULT_LOCALE}"
fi

IWS_TO_SYSTEMD_NAME=""
IWS_SCHEDULE="0"
IWS_SERVICE=""

# include functions
source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"
if [ $? -ne 0 ]; then
    echo "Can not find functions include file"
    exit 255
fi

usage()
{
    echo ""
    echo "$SERVER_VERSION"
    echo ""
    echo "Usage:"
    echo "    icewarpd.sh Command [Service]"
    echo ""
    echo "Command:"
    echo "    --start        Start specified Service or all services."
    echo "    --stop         Stop specified Service or all services."
    echo "    --restart      Restart specified Service or all services."
    echo "    --check        Test if specified Service is running."
    echo "                   Returned status code is 0 if Service is running,"
    echo "                   non-zero if service is not running."
    echo "    --finalize     Internal use only. Stops additional services as"
    echo "                   LDAP server or antivirus when none IceWarp service"
    echo "                   is running."
    echo "    --schedule     Requested start/stop/restart of services will be"
    echo "                   scheduled to run 5s in future and as background task."
    echo ""
    echo "Service can be one of: control, gw, im, pop3, smtp, all"
    echo "Command --check does not accept 'all' as a service name."
    echo ""
}

get_api_variable()
{
    CMD_STDOUT=$("${IWS_INSTALL_DIR}/tool.sh" get "$1" "$2" | sed 's/^[^:]*: //')
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

# Params
# 1  - Command to run
run_command_impersonated()
{
    if [ "$(whoami)" != "${IWS_PROCESS_USER}" ]; then
        su "${IWS_PROCESS_USER}" -c "$1"
    else
        bash -c "$1"
    fi
}

get_logfile_path()
{
    IW_LOGPATH="$(get_api_variable system C_System_Storage_Dir_LogPath)icewarpd/"
    LOG_FN="$(date +"%Y%m%d").log"
    run_command_impersonated "mkdir -p \"$IW_LOGPATH\""
    LOGFILE="${IW_LOGPATH}${LOG_FN}"
    echo "$LOGFILE"
}

# Params
# 1 - action (start, stop)
# 2 - name of service
log()
{
    LOGFILE="$(get_logfile_path)"
        
    LOG="$(date)\n"
    LOG+="$1 of $2\n"
    LOG+="Current user: $(whoami)\n"
    LOG+="$(pstree -sA $$ | tr '`' 'L')\n"
    
    CMD="echo -e \"${LOG}\" >> \"${LOGFILE}\""
    
    run_command_impersonated "${CMD}"
}

icewarp_to_systemd_servicename()
{
    IWS_TO_SYSTEMD_NAME=''
    
    case $1 in
        control)
            IWS_TO_SYSTEMD_NAME='icewarp-control'
            ;;
        gw)
            IWS_TO_SYSTEMD_NAME='icewarp-cal'
            ;;
        im)
            IWS_TO_SYSTEMD_NAME='icewarp-im'
            ;;
        pop3)
            IWS_TO_SYSTEMD_NAME='icewarp-pop3'
            ;;
        smtp)
            IWS_TO_SYSTEMD_NAME='icewarp-smtp'
            ;;
    esac    
}

do_systemd_service_action()
{
    sudo systemctl "$1" "$2"
}

start_systemd_service()
{
    do_systemd_service_action "start" "$1"
}

stop_systemd_service()
{
    do_systemd_service_action "stop" "$1"
}

check_systemd_service()
{
    icewarp_to_systemd_servicename "$1"
    systemctl is-active --quiet "$IWS_TO_SYSTEMD_NAME"
    
    if [ $? -eq 0 ]; then
        return 0
    else
        return 255
    fi
}

normalize_service_name()
{
    IWS_SERVICE=$1
    
    if [ "x$IWS_SERVICE" == "x" ]; then
        IWS_SERVICE=all
    fi
}

schedule_command()
{
    normalize_service_name "$2"
    
    log "scheduled $1" "$IWS_SERVICE"
    
    sudo systemd-run --on-active=5 ${IWS_INSTALL_DIR}/icewarpd.sh --"${1}" "${IWS_SERVICE}"
    
    if [ $? -eq 0 ]; then
      echo "Scheduled ${IWS_INSTALL_DIR}/icewarpd.sh --${1} ${IWS_SERVICE}"
    else
      echo "Failed to schedule command."
    fi
}

check_service()
{
    check_systemd_service "$1"
    
    if [ $? -eq 0 ]; then
      echo "Service $1 is running"
    else
      echo "Service $1 is not running"
      return 255
    fi
    
    return 0
}

start_service()
{
    normalize_service_name "$1"
    
    if [ "$IWS_SCHEDULE" != '0' ]; then
        schedule_command "start" "$IWS_SERVICE"
        
        return 0
    fi    
    
    log "start" "$IWS_SERVICE"    
    
    case $IWS_SERVICE in
        control)
            echo -n "Starting control ... "
            start_systemd_service "icewarp-control"
            ;;
        gw)
            echo -n "Starting groupware ... "
            start_systemd_service "icewarp-cal"
            ;;
        im)
            echo -n "Starting im ... "
            start_systemd_service "icewarp-im"
            ;;
        pop3)
            echo -n "Starting pop3 ... "
            start_systemd_service "icewarp-pop3"            
            ;;
        smtp)
            echo -n "Starting smtp ... "
            start_systemd_service "icewarp-smtp"                        
            ;;
        all)
            start_service control
            start_service gw
            start_service im
            start_service pop3
            start_service smtp
            return 0
            ;;
        *)
            echo "Unknown service: $IWS_SERVICE"
            return 1
            ;;
    esac

    check_systemd_service "$IWS_SERVICE"
    if [ $? -eq 0 ]; then
        echo "OK"
    else
        echo "ERROR"
    fi
}

stop_service()
{
    normalize_service_name "$1"    
    
    if [ "$IWS_SCHEDULE" != '0' ]; then
        schedule_command "stop" "$IWS_SERVICE"
        
        return 0
    fi    
    
    log "stop" "$IWS_SERVICE"    

    case $IWS_SERVICE in
        control)
            echo -n "Stopping control ... "
            stop_systemd_service "icewarp-control"
            ;;
        gw)
            echo -n "Stopping groupware ... "
            stop_systemd_service "icewarp-cal"
            ;;
        im)
            echo -n "Stopping im ... "
            stop_systemd_service "icewarp-im"
            ;;
        pop3)
            echo -n "Stopping pop3 ... "
            stop_systemd_service "icewarp-pop3"
            ;;
        smtp)
            echo -n "Stopping smtp ... "
            stop_systemd_service "icewarp-smtp"
            ;;
        all)
            stop_service control
            stop_service gw
            stop_service im
            stop_service pop3
            stop_service smtp
            
            return 0
            ;;
        *)
            echo "Unknown service: $IWS_SERVICE"
            return 1
            ;;
    esac

    check_systemd_service "$IWS_SERVICE"
    if [ $? -eq 0 ]; then
        echo "ERROR"
    else
        echo "OK"
    fi
}

restart_service()
{
    if [ "$IWS_SCHEDULE" != '0' ]; then
        schedule_command "restart" "$1"
        
        return 0
    fi    
    
    stop_service "$1"
    start_service "$1"
}

finalize_services()
{
    check_systemd_service "control"
    
    if [ $? -eq 0 ]; then
        return
    fi
    
    check_systemd_service "gw"
    
    if [ $? -eq 0 ]; then
        return
    fi

    check_systemd_service "smtp"
    
    if [ $? -eq 0 ]; then
        return
    fi

    check_systemd_service "pop3"
    
    if [ $? -eq 0 ]; then
        return
    fi

    check_systemd_service "im"
    
    if [ $? -eq 0 ]; then
        return
    fi
    
    stop_systemd_service "icewarp-ldap"
    stop_systemd_service "icewarp-kaspersky"
    stop_systemd_service "icewarp-sophos"
    stop_systemd_service "icewarp-bitdefender"
}

parse_cmdline()
{
    while [ "x$1" != "x" ]; do
        case `echo $1 | tr a-z A-Z` in
            --START)
                start_service "$2"
                shift
                ;;
            --STOP)
                stop_service "$2"
                shift
                ;;
            --RESTART)
                restart_service "$2"
                shift
                ;;
            --CHECK)
                check_service "$2"
                RET=$?
                shift
                ;;
            --FINALIZE)
                finalize_services
                ;;
            --SCHEDULE)
                IWS_SCHEDULE="1"
                ;;
            *)
                usage
                RET=1
                return
                ;;
        esac
        shift
    done
}

RET=0

if [ $# -eq 0 ]; then
    usage
else
    parse_cmdline $*
fi

exit $RET
