<?php

namespace server\model;

use GroupWareFolder;
use Sabre\VObject\Document;
use Sabre\VObject\Node;
use server\inc\gw\Search;
use tools\ParamLineParser;
use tools\VCalendar;

 
class Journal extends AGwEventItem implements IHasAttachments, IHasReminders, IHasRecurrences, IHasAttendees, IHasOccurrences
{
    use THasAttachments, THasReminders, THasRecurrences, THasAttendees, THasOccurrences, TEvnGwSearch;

    CONST TYPE = 'J';

    public $duplicity;
    protected $evntitle;
    protected $meeting_action;
    protected $evnlocation;
    protected $evntype;
    protected $evnsharetype;
    protected $evndescformat;
    protected $evnnote;
    protected $evnflags;
    protected $evntimeformat;
    protected $_tzevnstartdate;
    protected $_tzevnenddate;
    protected $_tzevnstarttime;
    protected $_tzevnendtime;
    protected $ctz;
    protected $evnOrganizer;
    protected $evnstartdate;
    protected $evnenddate;
    protected $evnstarttime;
    protected $evnendtime;
    protected $evnmeetingid;
    protected $conferenceId;
    protected $organizerName;
    protected $started;
    protected $starttime;
    protected $desktop;
    protected $password;
    protected $laststart;
    protected $autorecording;
    protected $autolobby;
    protected $informationUrl;
    protected $informationBody;
    protected $participants;
    protected $recordinginprogress;
    protected $mutestatus;
    protected $evncomplete;
    protected $evnurl;
    protected $evnMetadata;

    private static $lengthVariablesExport = [
        'evnmeetingid' => 255,
        'evnlocation' => 510,
        'evntitle' => 510,
    ];

    public static $searchFields = [
        'EvnTitle',
        'EvnNote',
    ];

     
    public function loadById(string $id) : bool
    {
        $this->setId($id);
        $data = $this->getGwApi()->FunctionCall('getEventInfo', $this->getFolderSessionId(), $this->getId(), 'use_tzid=1&revision_extended=1');
        $paramParser = new ParamLineParser($data, true);
        if(empty($paramParser->parsedParamLines[0] ?? null)) return false;
        $this->fillFromArray($paramParser->parsedParamLines[0], true);
        $this->loadMeetingInfo();
        $this->afterLoad();
        return true;
    }

    protected function loadMeetingInfo()
    {
         
        $gwApi = $this->folder->account->gwAPI;
        $result = $gwApi->FunctionCall('GetMeetingInfo', $_SESSION['EMAIL'], $this->getEvnmeetingid(), '1');
        parse_str($result, $resultArray);
        if(isset($resultArray['id'])){
            $resultArray['conferenceId'] = $resultArray['id'];
            unset($resultArray['id']);
        }
        $this->fillFromArray($resultArray);
        $this->addAttendeesFromGW();
        $this->addAttachmentsFromGW();
        $this->addRemindersFromGW();
        $this->addRecurrencesFromGW();
    }

    protected function afterLoad(): void
    {
        if($this->loaded) return;
        $this->loaded = true;
        if(!($this->getEvnflags() & \Item::FLAG_RECORDED)) return;
        $string = $this->getEvnMetadata();
        if(!preg_match('/recordings=([^&]+)/iu', $string, $match)) return;
        $string = urldecode(urldecode($match[1]));
        if(!preg_match_all('/(?P<key>[^&=]+)=(?P<value>[^&=]+)?/', $string, $matches, PREG_SET_ORDER)) return;
        $params = [];
        foreach ($matches as $match) {
            if(isset($params[$match['key']])){
                $attachment = new Attachment($this->folder, null, $this);
                $attachment->setParams(http_build_query($params));
                $attachment->setFileLocation('url');
                $this->addAttachment($attachment);
                unset($params);
            }
            $params[$match['key']] = $match['value'];
        }
        $attachment = new Attachment($this->folder, null, $this);
        $attachment->setParams(http_build_query($params));
        $attachment->setFileLocation('url');
        $this->addAttachment($attachment);
    }

    public static function getByFilter(\Folder $folder, &$filter): array
    {
         
        $gwApi = $folder->account->gwAPI;
        $sessionId = $folder->openAccess();
        if(!empty($filter['sql'])){
            $filter['sql'] .= ' AND ';
        }
        $filter['sql'] .= '(EvnMeetingID IS NOT NULL AND EvnMeetingID <> \'\')';
        \slToolsIcewarp::removeEmptyConditionsSQL($filter['sql']);
        $attributes = $gwApi->filterToAttributes($filter);

        $selectFields = explode(',', strtolower($filter['tag']));
        if(($key = array_search('osd', $selectFields)) !== false){
            unset($selectFields[$key]);
            $selectFields[] = 'EVNSTARTDATE as OSD';
        }
        if(($key = array_search('oed', $selectFields)) !== false){
            unset($selectFields[$key]);
            $selectFields[] = 'EVNENDDATE as OED';
        }
        if(($key = array_search('oet', $selectFields)) !== false){
            unset($selectFields[$key]);
            $selectFields[] = 'EVNENDTIME as OET';
        }
        if(($key = array_search('ost', $selectFields)) !== false){
            unset($selectFields[$key]);
            $selectFields[] = 'EVNSTARTTIME as OST';
        }
        $selectFields[] = 'EvnFlags';
        $selectFields[] = 'EvnComplete';
        $selectFields[] = 'EvnUrl';
        $selectFields[] = 'EvnNote';
        $result = $gwApi->FunctionCall('GetEventList', $sessionId, $filter['sql'], implode(',', $selectFields), $attributes);
        $parser = new \tools\ParamLineParser($result);
        $items = [];
        foreach ($parser->parsedParamLines as $parsedParamLine) {
            $item = new Journal($folder);
            $item->fillFromArray($parsedParamLine, false);
            $item->setFilterTag($filter);
            $items[] = $item;
        }
        return $items;
    }

    public static function getCountByFilter(\Folder $folder, $filter)
    {
         
        $gwApi = $folder->account->gwAPI;
        $sessionId = $folder->openAccess();
        return $gwApi->FunctionCall('GetEventCount', $sessionId, $filter['sql']);
    }

     
    public function getTemplateDataArray(array $allowEmpty = []) : array
    {
        $result = $this->getAsArrayFE(false, true, $allowEmpty);
        $result['id'] = \slToolsPHP::htmlspecialchars($this->getId());
        $result['sFields'] = '<values>' . $this->xml . '</values>';
        $this->addAttachmentsTemplateDataArray($result);
        $this->addAttendeesTemplateDataArray($result);
        $this->addRemindersTemplateDataArray($result);
        $this->addRecurrencesTemplateDataArray($result);
        return $result;
    }

     
    public function getICalendar(): string
    {
        $vCalendar = new VCalendar();
        $vEvent = $vCalendar->add('VEVENT');
        if(!empty($this->getId())){
            $vEvent->add('UID', $this->getId());
            $vEvent->add('X-SERVER-UID', $this->getId());
        }else{
            unset($vEvent->UID);
        }
        if(!empty($this->getEvntype())) $vEvent->add('CATEGORIES', $this->getEvntype());
        $vEvent->add('DESCRIPTION', strip_tags(html_entity_decode($this->getEvnnote())));
        $vEvent->add('X-ALT-DESC', $this->getEvnnote(), ['FMTTYPE' => 'text/html']);
        if(!empty($this->getEvnlocation())) $vEvent->add('LOCATION', $this->getEvnlocation());
        if(!empty($this->getMeetingAction())) $vEvent->add('X-ICEWARP-MEETING', $this->getMeetingAction());
        $vEvent->add('SUMMARY', $this->getEvntitle());
        $vEvent->add('CLASS', ($this->getEvnsharetype() == 'P' ? 'PRIVATE' : 'PUBLIC'));
        $vEvent->add('CREATED', $this->getEvnCreated());
        if($this->getEvnModified() instanceof \DateTime) $vEvent->add('LAST-MODIFIED', $this->getEvnModified());
        if($this->getEvndeleted() instanceof \DateTime) $vEvent->add('DELETED', $this->getEvndeleted());
        $vEvent->add('DTSTART', $this->getCalendarToDateTime($this->getTzevnstartdate(), $this->getTzevnstarttime()));
                 $vEvent->add('DTEND', $this->getCalendarToDateTime($this->getTzevnenddate(), $this->getTzevnendtime()));
        if(!empty($this->getTzid())){
            $vEvent->DTSTART['TZID'] = $this->getTzid();
            $vEvent->DTEND['TZID'] = $this->getTzid();
        }
        $vEvent->add('ORGANIZER', $_SESSION['EMAIL']);
        if($this->getEvnflags() == \GroupWareItem::FLAG_ORGANIZATOR){
            $vEvent->add('X-MICROSOFT-CDO-BUSYSTATUS', 'BUSY');
        }elseif($this->getEvnflags() == \GroupWareItem::FLAG_TRANSPARENT){
            $vEvent->add('X-MICROSOFT-CDO-BUSYSTATUS', 'FREE');
        }elseif($this->getEvnflags() == \GroupWareItem::FLAG_TENTATIVE){
            $vEvent->add('X-MICROSOFT-CDO-BUSYSTATUS', 'TENTATIVE');
        }elseif($this->getEvnflags() == \GroupWareItem::FLAG_OUTOFOFFICE){
            $vEvent->add('X-MICROSOFT-CDO-BUSYSTATUS', 'OOF');
        }else{
            $vEvent->add('X-MICROSOFT-CDO-BUSYSTATUS', 'BUSY');
        }
                 return $this->serialize($vCalendar, $vEvent);
    }

     
    public static function createFromXML(\Folder $folder, string $xml, bool $save = false, AGwEventItem $parent = null)
    {
        if($folder->type == 'I') return TeamChat::createFromXML($folder, $xml, $save, $parent);
        $item = parent::createFromXML($folder, $xml, $save, $parent);
        $item->setDuplicity($item->getSourceSimpleXML()['duplicity']);
        return $item;
    }

    public function getType(): string
    {
        return self::TYPE;
    }

     
    public function fillFromVObject(Node $vObject)
    {
        parent::fillFromVObject($vObject->VEVENT);
        $this->setEvntitle($vObject->VEVENT->SUMMARY ?? null);
        $this->setEvntitle($vObject->VEVENT->{'X-ICEWARP-MEETING'} ?? null);
        $this->setEvntype($vObject->VEVENT->CATEGORIES ?? null);
        $this->setEvnsharetype((string)$vObject->VEVENT->CLASS == 'PRIVATE' ? 'P' : 'U');
        if(isset($vObject->VEVENT->{'X-ALT-DESC'})){
            $this->setEvnnote($vObject->VEVENT->{'X-ALT-DESC'});
        }elseif(isset($vObject->VEVENT->{'DESCRIPTION'})){
            $this->setEvnnote($vObject->VEVENT->{'DESCRIPTION'});
        }
    }

    public function afterCondition(Search $search, $value) : string
    {
        return $this->placeValueToCondition('EvnStartDate > %s', $this->strToDelphiDate($value));
    }

    public function aftertimeCondition(Search $search, $value) : string
    {
        return $this->placeValueToCondition('EvnStartTime > %s', $this->strToTime($value));
    }

    public function beforetimeCondition(Search $search, $value) : string
    {
        return $this->placeValueToCondition('EvnStartTime < %s', $this->strToTimeBefore($value));
    }

    public function beforeCondition(Search $search, $value) : string
    {
        return $this->placeValueToCondition('EvnStartDate < %s', $this->strToDelphiDateBefore($value));
    }

    public function hasCondition(Search $search, $value) : string
    {
        if($value == 'conference') return "(EvnMeetingID IS NOT NULL AND EvnMeetingID <> '')";
        return '';
    }

    public function conferenceCondition(Search $search, $value) : string
    {
        $accountID = $this->folder->account->accountID;
        if($value == 'organizer') return $this->placeValueToCondition("((EvnMeetingID IS NOT NULL AND EvnMeetingID <> '') AND (EvnOrganizer LIKE '%%%s%%' {ESCAPE '\\'}))", $accountID);
        if($value == 'attendee') return $this->placeValueToCondition("((EvnMeetingID IS NOT NULL AND EvnMeetingID <> '') AND NOT (EvnOrganizer IS NOT NULL AND EvnOrganizer LIKE '%%%s%%' {ESCAPE '\\'}))", $accountID);
        return '';
    }

    public function afterendCondition(Search $search, $value) : string
    {
        $filter = & $search->filter;
        if($filter) $filter['idinterval'] = 'FUTURE';
        return '';
    }

    public function beforeendCondition(Search $search, $value) : string
    {
        $filter = & $search->filter;
        if($filter) $filter['idinterval'] = 'PAST';
        return '';
    }

    public function afteridCondition(Search $search, $value) : string
    {
        $filter = & $search->filter;
        if(preg_match('/(?:afterend|beforeend)/i', $search->searchCondition)){
            $filter['idinterval'] = ($filter['idinterval'] ?? '') . '+' . $value;
            return '';
        }
        return $this->placeValueToCondition("(Evn_ID > '%s')", $value);
    }

    public function beforeidCondition(Search $search, $value) : string
    {
        $filter = & $search->filter;
        if(preg_match('/(?:afterend|beforeend)/i', $search->searchCondition)){
            $filter['idinterval'] = ($filter['idinterval'] ?? '') . '+' . $value;
            return '';
        }else {
            return $this->placeValueToCondition("(Evn_ID < '%s')", $value);
        }
    }

    public function notifyGroupChat($email)
    {
         
        $gwApi = $this->folder->account->gwAPI;
        $sessionId = $this->folder->openAccess();
        if(($result = $gwApi->FunctionCall("NotifyAboutItemSharing", $sessionId, $this->getId(), 'email='.urlencode($email))) <= 0){
            throw new \Exc('item_notify_groupchat',$result);
        }
    }

    public function recording($targetFolder,$note = '')
    {
         
        $gwApi = $this->folder->account->gwAPI;
        $sessionId = $this->folder->openAccess();
        $targetFolder = $this->folder->account->getFolder($targetFolder);
        $targetSessionId = $targetFolder->openAccess();

        $result = $gwApi->FunctionCall("AddDocumentLink", $sessionId, $targetSessionId, $this->getId(), 'EvnNote='.urlencode($note));
        if(!$result){
            throw new \Exc('recording');
        }
    }

     
    public function getEvnOrganizer()
    {
        return $this->evnOrganizer;
    }

     
    public function setEvnOrganizer($evnOrganizer): Journal
    {
        $this->evnOrganizer = $evnOrganizer;
        return $this;
    }

     
    public function getDuplicity()
    {
        return $this->duplicity;
    }

     
    public function setDuplicity($duplicity): Journal
    {
        $this->duplicity = $duplicity;
        return $this;
    }

     
    public function getEvntitle()
    {
        return $this->evntitle;
    }

     
    public function setEvntitle($evntitle): Journal
    {
        $this->evntitle = (string)$evntitle;
        return $this;
    }

     
    public function getMeetingAction()
    {
        return $this->meeting_action;
    }

     
    public function setMeetingAction($meeting_action): Journal
    {
        $this->meeting_action = $meeting_action;
        return $this;
    }

     
    public function getEvnlocation()
    {
        return $this->evnlocation;
    }

     
    public function setEvnlocation($evnlocation): Journal
    {
        $this->evnlocation = $evnlocation;
        return $this;
    }

     
    public function getEvntype()
    {
        return $this->evntype;
    }

     
    public function setEvntype($evntype): Journal
    {
        $this->evntype = $evntype;
        return $this;
    }

     
    public function getEvnsharetype()
    {
        return $this->evnsharetype;
    }

     
    public function setEvnsharetype($evnsharetype): Journal
    {
        $this->evnsharetype = $evnsharetype;
        return $this;
    }

     
    public function getEvndescformat()
    {
        return $this->evndescformat;
    }

     
    public function setEvndescformat($evndescformat): Journal
    {
        $this->evndescformat = $evndescformat;
        return $this;
    }

     
    public function getEvnnote()
    {
        return $this->evnnote;
    }

     
    public function setEvnnote($evnnote): Journal
    {
        $this->evnnote = $evnnote;
        return $this;
    }

     
    public function getEvnflags()
    {
        return $this->evnflags;
    }

     
    public function getEvnstartdate()
    {
        return $this->evnstartdate;
    }

     
    public function setEvnstartdate($evnstartdate): Journal
    {
        $this->evnstartdate = $evnstartdate;
        return $this;
    }

     
    public function getEvnenddate()
    {
        return $this->evnenddate;
    }

     
    public function setEvnenddate($evnenddate): Journal
    {
        $this->evnenddate = $evnenddate;
        return $this;
    }

     
    public function getEvnstarttime()
    {
        return $this->evnstarttime;
    }

     
    public function setEvnstarttime($evnstarttime): Journal
    {
        $this->evnstarttime = $evnstarttime;
        return $this;
    }

     
    public function getEvnendtime()
    {
        return $this->evnendtime;
    }

     
    public function setEvnendtime($evnendtime): Journal
    {
        $this->evnendtime = $evnendtime;
        return $this;
    }

     
    public function setEvnflags($evnflags): Journal
    {
        $this->evnflags = $evnflags;
        if(!empty($evnflags) && \GroupWareItem::isOrganizator($evnflags) ){
            $groupChatOwnerEmail = '';
            $sOrganizer = $this->getFolder()->account->gwAPI->getFolderOwner($this->getFolder(), $groupChatOwnerEmail);
            $sOrganizerFullAddress = $this->getFolder()->account->gwAPI->getOwnerFullAddress($sOrganizer, $groupChatOwnerEmail);
            $this->setEvnOrganizer($sOrganizerFullAddress);
        }
        return $this;
    }

     
    public function getEvntimeformat()
    {
        return $this->evntimeformat;
    }

     
    public function setEvntimeformat($evntimeformat): Journal
    {
        $this->evntimeformat = $evntimeformat;
        return $this;
    }

     
    public function getTzevnstartdate()
    {
        return $this->_tzevnstartdate;
    }

     
    public function setTzevnstartdate($_tzevnstartdate): Journal
    {
        $this->_tzevnstartdate = $_tzevnstartdate;
        return $this;
    }

     
    public function getTzevnenddate()
    {
        return $this->_tzevnenddate;
    }

     
    public function setTzevnenddate($_tzevnenddate): Journal
    {
        $this->_tzevnenddate = $_tzevnenddate;
        return $this;
    }

     
    public function getTzevnstarttime()
    {
        return $this->_tzevnstarttime;
    }

     
    public function setTzevnstarttime($_tzevnstarttime): Journal
    {
        $this->_tzevnstarttime = $_tzevnstarttime;
        return $this;
    }

     
    public function getTzevnendtime()
    {
        return $this->_tzevnendtime;
    }

     
    public function setTzevnendtime($_tzevnendtime): Journal
    {
        $this->_tzevnendtime = $_tzevnendtime;
        return $this;
    }

     
    public function getCtz()
    {
        return $this->ctz;
    }

     
    public function setCtz($ctz): Journal
    {
        $this->ctz = $ctz;
        return $this;
    }

     
    public function getEvnmeetingid()
    {
        return $this->evnmeetingid;
    }

     
    public function setEvnmeetingid($evnmeetingid): void
    {
        $this->evnmeetingid = $evnmeetingid;
    }

     
    public function getConferenceId()
    {
        return $this->conferenceId;
    }

     
    public function setConferenceId($conferenceId): void
    {
        $this->conferenceId = $conferenceId;
    }

     
    public function getOrganizerName()
    {
        return $this->organizerName;
    }

     
    public function setOrganizerName($organizerName): void
    {
        $this->organizerName = $organizerName;
    }

     
    public function getStarted()
    {
        return $this->started;
    }

     
    public function setStarted($started): void
    {
        $this->started = $started;
    }

     
    public function getStarttime()
    {
        return $this->starttime;
    }

     
    public function setStarttime($starttime): void
    {
        $this->starttime = $starttime;
    }

     
    public function getDesktop()
    {
        return $this->desktop;
    }

     
    public function setDesktop($desktop): void
    {
        $this->desktop = $desktop;
    }

     
    public function getPassword($bool = true)
    {
        if($bool) return !empty($this->password);
        return $this->password;
    }

     
    public function setPassword(
                 $password
    ): void
    {
        $this->password = $password;
    }

     
    public function getLaststart()
    {
        return $this->laststart;
    }

     
    public function setLaststart($laststart): void
    {
        $this->laststart = $laststart;
    }

     
    public function getAutorecording()
    {
        return $this->autorecording;
    }

     
    public function setAutorecording($autorecording): void
    {
        $this->autorecording = $autorecording;
    }

     
    public function getAutolobby()
    {
        return $this->autolobby;
    }

     
    public function setAutolobby($autolobby): void
    {
        $this->autolobby = $autolobby;
    }

     
    public function getInformationUrl()
    {
        return $this->informationUrl;
    }

     
    public function setInformationUrl($informationUrl): void
    {
        $this->informationUrl = $informationUrl;
    }

     
    public function getInformationBody()
    {
        return $this->informationBody;
    }

     
    public function setInformationBody($informationBody): void
    {
        $this->informationBody = $informationBody;
    }

     
    public function getParticipants()
    {
        return $this->participants;
    }

     
    public function setParticipants($participants): void
    {
        $this->participants = $participants;
    }

     
    public function getRecordinginprogress()
    {
        return $this->recordinginprogress;
    }

     
    public function setRecordinginprogress($recordinginprogress): void
    {
        $this->recordinginprogress = $recordinginprogress;
    }

     
    public function getMutestatus()
    {
        return $this->mutestatus;
    }

     
    public function setMutestatus($mutestatus): void
    {
        $this->mutestatus = $mutestatus;
    }

     
    public function getEvncomplete()
    {
        return $this->evncomplete;
    }

     
    public function setEvncomplete($evncomplete): void
    {
        $this->evncomplete = $evncomplete;
    }

     
    public function getEvnMetadata()
    {
        return $this->evnMetadata;
    }

     
    public function setEvnMetadata($evnMetadata): void
    {
        $this->evnMetadata = $evnMetadata;
    }

     
    public function getEvnurl()
    {
        return $this->evnurl;
    }

     
    public function setEvnurl($evnurl): void
    {
        $this->evnurl = $evnurl;
    }
}