<?php

namespace server\model;

use GroupWareFolder;
use Sabre\VObject\Component;
use Sabre\VObject\Document;
use Sabre\VObject\Node;
use server\inc\gw\Search;
use SimpleXMLElement;
use tools\ParamLineParser;
use tools\Reader;
use tools\SearchParserYoda;

 
abstract class AGwEventItem
{
     
    protected $folder;
     
    protected $parent;
     
    protected $sourceSimpleXML;
     
    private $variablesSet = [];
    protected $skipInvitation = false;
    protected $evnclass;
     
    protected $evnCreated;
     
    protected $evnId;
     
    protected $_tzid;
    protected $evnModified;
    protected $evndeleted;
    protected $evneditcounter;
     
    protected $att_webdav_link;
     
    protected $gwApi;
     
    protected $folderSessionId;
    protected $xml;
    protected $backupState;
    protected $preparedParamLine = null;
    protected $finalized = false;
    protected $loaded = false;
    protected $filterTag = [];
    public $rename = false;
    public $duplicity = false;
    public $isLoaded = false;
    public $editChildren = true;

    private static $skipVariablesExport = [
        'parent',
        'skipInvitation',
        'tzid',
        'att_webdav_link',
        'folderSessionId',
        'folder',
        'atrValue',
        'loaded',
        'filterTag',
    ];
    private static $dependableVariablesExport = [
        'evndescformat'=>['evntitle','evnnote'],
    ];

    private static $renameVariablesExport = [
        'folder' => 'EVNFOLDER'
    ];

    private static $lengthVariablesExport = [

    ];

    public static $models = [
        'N' => Note::class,
        'E' => Event::class,
        'C' => Contact::class,
        'T' => Task::class,
        'I' => TeamChat::class,
        'F' => \server\model\Document::class,
        'J' => Journal::class,
        'G' => GroupWareTrash::class,
        'M' => TeamChat::class,
        'U' => Upload::class,

         
    ];

    public static $disableUpdate = [];

     
    public function __construct(\Folder $folder, string $id = null, AGwEventItem $parent = null)
    {
        $this->setParent($parent);
        $this->setFolder($folder)
            ->setEvnclass($this->getType() ?? null)
            ->setTzid($_SESSION['CLIENT_TIMEZONE'] ? $_SESSION['CLIENT_TIMEZONE'] : $_SESSION['SERVER_TIMEZONE']);
        if(!empty($id)){
            $this->isLoaded = $this->loadById($id);
            $this->backupState();
        }
    }

    protected function backupState()
    {
        $this->backupState = $this->getAsArray(false, true, [], [], [], []);
    }

    public function getChangedFromBackup(bool $encode = false, bool $skipEmpty = true, array $allowEmpty = [])
    {
        
        $result = self::difference($this->backupState ?? [], $this->getAsArray(false, $skipEmpty, [], [], [], []));
        unset($result['CTZ'], $result['EVN_CREATED']);
        if(!empty($this->getId())){
            unset($result['EVNCLASS']);
        }
        if($encode) $result = array_map('rawurlencode', $result);
        return $result;
    }

     
    public function getSearchSqlQuery(Search $search, array $conditions, & $filter = null): string
    {
        $result = '';
        $optionals = [];
        foreach ($conditions as $condition) {
            if(is_callable([$this, $condition['keyword'] . 'Condition'])){
                                                  $part = $this->{$condition['keyword'] . 'Condition'}($search, trim(\slToolsString::removeQuotes($condition['value']), " \t\n\r\x0B'"), $filter);
                if(!empty($part)){
                    if($condition['operator']=='?'){
                        $optionals[] = '('.$part.')';
                    }else{
                        $result .= $result ? ' AND ' : '';
                        $result .= ($condition['operator'] == '-' ? 'NOT ' : '');
                        $result .= $part;
                    }
                }
            }
        }
        
		if (!empty($optionals)){
            if($result){
			    $result = '('.$result.') OR ';
            }
			$result.= '('.join(' OR ',$optionals).')';
		}
        return $result;
    }

     
    public static function implodeAlternate(array ... $array) : string
    {
        $result = '';
        do{
            foreach ($array as $key => $parameter) {
                $result .= array_shift($parameter);
                $array[$key] = $parameter;
                if(empty($parameter)) unset($array[$key]);
            }
        }while(!empty($array));
        return $result;
    }

     
    public function placeValueToCondition(string $condition, $value , $numReplacements = null) : string
    {
        $escape = false;
        if (!is_array($value)){
            if(is_null($numReplacements)) $numReplacements = substr_count($condition, '%s');
            $value = array_fill(0, $numReplacements, $value);
        }
        foreach ($value as $key => $item) {
            $item = str_replace("'","''", $item);
            $value[$key] = $item;
        }
        return vsprintf($condition,$value);
    }

    protected function strToDelphiDate($str) : int
    {
        if(strcasecmp($str, 'now') === 0) $str = date('Y-m-d');
        $arr = explode('-',$str);
        return GregorianToJD($arr[1], $arr[2], $arr[0]);
    }

    protected function strToDelphiDateBefore($str) : int
    {
        return self::strToDelphiDate($str) + 1;
    }

    protected function strToTime($str)
    {
        return \slToolsDate::iso86012calendartime($str);
    }

    protected function strToTimeBefore($str)
    {
        return \slToolsDate::iso86012calendartime($str) + 1;
    }

     
    public static function checkMinVersion(float $minVersion): bool
    {
        return preg_match('/^(?P<version>\\d+\\.\\d+)\\./', $_SESSION['WM_VERSION'], $matches ) && floatval($matches['version']) >= $minVersion;
    }

    public function getCreatedData() : array
    {
        $result = [];
        $result['id'] = $this->getId();
        $result['created'] = $this->getEvnCreated(false);
        $result['att_webdav_link'] = $this->getAttWebdavLink();
        return $result;
    }

     
    public static function getFromIdByFolder(\Folder $folder, string $id, AGwEventItem $parent = null)
    {
        if(!isset(self::$models[$folder->getType()])) return false;
        return new self::$models[$folder->getType()]($folder, $id, $parent);
    }

     
    public static function createFromXMLByFolder(\Folder $folder, string $xml, bool $save = false, AGwEventItem $parent = null)
    {
        if(!isset(self::$models[$folder->getType()])) return false;
        return self::$models[$folder->getType()]::createFromXML($folder, $xml, $save, $parent);
    }

     
    public static function createFromXMLByClass(string $class, \Folder $folder, string $xml, bool $save = false, AGwEventItem $parent = null)
    {
        if(!isset(self::$models[$class])) return false;
        return self::$models[$class]::createFromXML($folder, $xml, $save, $parent);
    }

    public static function difference(array $oldArr, array $newArr)
    {
        $result = [];
        foreach($newArr as $key => $val){
            if(!isset($oldArr[$key]) || $oldArr[$key]!=$val){
                $result[$key] = $val;
            }
        }
        return $result;
    }

    public static function getEditedFromXMLByFolder(\Folder $folder, string $id, string $xml, bool $save = true, AGwEventItem $parent = null)
    {
        $new = \server\model\AGwEventItem::createFromXMLByFolder($folder, $xml);
        $original = \server\model\AGwEventItem::getFromIdByFolder($folder, $id);
        if(!$original) throw new \Exc('item_create', $id);
        $disableUpdate = array_merge(self::$disableUpdate, $original::$disableUpdate ?? []);
 
        $originalArr = $original->getAsArray(false,true,[],$disableUpdate,[],[]) ?? [];
        $newArr = $new->getAsArray(false,true,[],$disableUpdate,[],[]) ?? [];

        $original->fillFromArray(self::difference($originalArr, $newArr),false,[],true);

        if($new->editChildren) $original->editChildren($new);
        if($save) $original->save();
        return $original;
    }

     
    public function getAttWebdavLink()
    {
        return $this->att_webdav_link;
    }

     
    public function setAttWebdavLink($att_webdav_link): AGwEventItem
    {
        $this->att_webdav_link = $att_webdav_link;
        return $this;
    }

    public static function getModelByFolder(\Folder $folder, string $id = null)
    {
        if(!isset(self::$models[$folder->getType()])) return false;
        return new self::$models[$folder->getType()]($folder, $id);
    }

    public static function getModelsByFilter(\Folder $folder, &$filter) : array
    {
        if(!isset(self::$models[$folder->getType()])) return [];
         
        $model = self::$models[$folder->getType()];
        return $model::getByFilter($folder, $filter);
     }

    public static function getModelCountByFilter(\Folder $folder, $filter)
    {
        if(!isset(self::$models[$folder->getType()])) return false;
         
        $model = self::$models[$folder->getType()];
        if(!is_callable([$model, 'getCountByFilter'])) return false;
        return $model::getCountByFilter($folder, $filter);
    }

     
    public function loadById(string $id)    {
     }
    
    public function fillFromArray(array $array, $loadAllAddons = false, array $loadAddons = [],bool $markVariablesSet = false)
    {
        foreach ($array as $property => $value) {
            if(empty($property)) continue;
            $setter = 'set' . ucfirst(self::camelize($property));
            if(is_callable([$this, $setter])){
                $this->{$setter}($value);
                if ($markVariablesSet){
                    $this->variablesSet[strtolower($property)] = true;
                }
            }
        }
        if($loadAllAddons) {
            $this->loadAddons(Attachment::class);
            $this->loadAddons(ItemAttribute::class);
            $this->loadAddons(EventReaction::class);
        }else{
            foreach ($loadAddons as $loadAddon) {
                $this->loadAddons($loadAddon);
            }
        }
        $this->afterLoad();
    }

    protected function loadAddons($class)
    {
        $reflection = new \ReflectionClass($class);
        if($reflection->implementsInterface(ISubItem::class)){
            $interface = $class::getControlInterface();
            if(!$this instanceof $interface) return;
            $this->{$class::getLoadingFunction()}();
        }

        if($reflection->isInterface()){
            if(!$this instanceof $class) return;
        }
        if($reflection->isTrait()){
            if(!in_array($class, class_uses($this))) return;
        }

    }

    public function fillFromVObject(Node $vObject)
    {
                 if(empty($this->getId())) $this->setId((string)$vObject->{'X-SERVER-UID'} ?? null);
        if(isset($vObject->CREATED)) $this->setEvnCreated($vObject->CREATED);
        if(isset($vObject->{'LAST-MODIFIED'})) $this->setEvnModified($vObject->{'LAST-MODIFIED'});
        if(isset($vObject->DELETED)) $this->setEvndeleted($vObject->DELETED);

        if($this instanceof IHasAttachments) $this::addAttachmentsFromVObject($this, $vObject);
        if($this instanceof IHasAttendees) $this::addAttendeesFromVObject($this, $vObject);
        if($this instanceof IHasCertificates) $this::addCertificatesFromVObject($this, $vObject);
        if($this instanceof IHasLocations) $this::addLocationsFromVObject($this, $vObject);
        if($this instanceof IHasRecurrences) $this::addRecurrencesFromVObject($this, $vObject);
        if($this instanceof IHasReminders) $this::addRemindersFromVObject($this, $vObject);
        if($this instanceof IHasItemAttributes) $this::addItemAttributesFromVObject($this, $vObject);
    }

     
    public function getTzid()
    {
        return $this->_tzid;
    }

     
    public function setTzid($_tzid): AGwEventItem
    {
        $this->_tzid = $_tzid;
        return $this;
    }

     
    public function getEvnId()
    {
        return $this->evnId;
    }

     
    public function setEvnId($evnId): AGwEventItem
    {
        $this->evnId = (string)$evnId;
        return $this;
    }

     
    public function getEvnCreated($object = true)
    {
        return ($object) ? $this->evnCreated : $this->getItemScalarValue($this->evnCreated);
    }

     
    public function setEvnCreated($evnCreated = null): AGwEventItem
    {
        $this->evnCreated = self::getDateTime($evnCreated);
        return $this;
    }

     
    public function getEvnclass()
    {
        return $this->evnclass ?? $this->getFolder()->getType();
    }

     
    public function setEvnclass($evnclass): AGwEventItem
    {
        $this->evnclass = $evnclass;
        return $this;
    }

     
    public function getSkipInvitation()
    {
        return $this->skipInvitation;
    }

     
    public function setSkipInvitation($skipInvitation): AGwEventItem
    {
        $this->skipInvitation = $skipInvitation;
        return $this;
    }

     
    public function getSourceSimpleXML(): ?SimpleXMLElement
    {
        return $this->sourceSimpleXML;
    }

     
    public function setSourceSimpleXML(?SimpleXMLElement $sourceSimpleXML): AGwEventItem
    {
        $this->sourceSimpleXML = $sourceSimpleXML;
        return $this;
    }

     
    public function getParent(): ? AGwEventItem
    {
        return $this->parent;
    }

     
    public function setParent(? AGwEventItem $parent = null): AGwEventItem
    {
        $this->parent = $parent;
        return $this;
    }

     
    public function getEvnModified()
    {
        return $this->evnModified;
    }

     
    public function setEvnModified($evnModified): AGwEventItem
    {
        if(!empty($evnModified)) $this->evnModified = self::getDateTime($evnModified);
        return $this;
    }

     
    public function getEvndeleted()
    {
        return $this->evndeleted;
    }

     
    public function setEvndeleted($evndeleted): AGwEventItem
    {
        if(!empty($evndeleted)) $this->evndeleted = self::getDateTime($evndeleted);
        return $this;
    }

     
    public function getEvneditcounter()
    {
        return $this->evneditcounter;
    }

     
    public function setEvneditcounter($evneditcounter): AGwEventItem
    {
        $this->evneditcounter = $evneditcounter;
        return $this;
    }

     
    public function setFolder(\Folder $folder) : AGwEventItem
    {
        $this->folder = $folder;
        return $this;
    }

    protected function getCalendarToDateTime($date, $time = 0, $timezone = null)
    {
        $result = new \DateTime();
        if(!empty($timezone)) $result->setTimezone(new \DateTimeZone($timezone));
        $result->setTimestamp(\MerakGWAPI::calendar2unixTime($date,$time));
        return $result;
    }

    protected function getDateTimeToCalendarDate(\DateTimeInterface $dateTime)
    {
        return \MerakGWAPI::unix2calendardate($dateTime->getTimestamp());
    }

    protected function getDateTimeToCalendarTime(\DateTimeInterface $dateTime)
    {
        return \MerakGWAPI::unix2calendartime($dateTime->getTimestamp()) - ($_SESSION['CLIENT_TIMEZONE_OFFSET'] ?? 0);
    }

    public static function getDateTime($mixed)
    {
        if($mixed instanceof \DateTime){
            return $mixed;
        }elseif (is_numeric($mixed)){
            $result = new \DateTime();
            $result->setTimestamp($mixed);
        }else{
            $result = new \DateTime($mixed ?? 'now');
        }
        return $result;
    }

    public function editChildren(AGwEventItem $source)
    {
        if(get_class($this) != get_class($source)) return;
        if($this instanceof IHasEventReactions) $this->updateEventReactionsFromObject($source);
        if($this instanceof IHasAttendees) $this->updateAttendeesFromObject($source);
        if($this instanceof IHasAttachments) $this->updateAttachmentsFromObject($source);
        if($this instanceof IHasItemAttributes) $this->updateItemAttributesFromObject($source);
        if($this instanceof IHasCertificates) $this->updateCertificatesFromObject($source);
        if($this instanceof IHasLocations) $this->updateLocationsFromObject($source);
        if($this instanceof IHasOccurrences) $this->updateOccurrencesFromObject($source);
        if($this instanceof IHasRecurrences) $this->updateRecurrencesFromObject($source);
        if($this instanceof IHasReminders) $this->updateRemindersFromObject($source);
    }

    public static function camelize(string $input, string $separator = '_')
    {
        return str_replace($separator, '', ucwords($input, $separator));
    }

    public static function snakeize(string $input, string $separator = '_')
    {
        if(!preg_match_all('/((?:(?:^[a-z])|(?:[A-Z]))[^A-Z]+)/', $input, $matches)) return $input;
        return implode($separator, $matches[0]);
    }

     
    public static function fillValues(AGwEventItem $item, SimpleXMLElement $simpleXMLElement = null)
    {
        if(isset($simpleXMLElement)){
            $item->setSourceSimpleXML($simpleXMLElement);
        }
        $simpleXMLElement = $item->getSourceSimpleXML();
        if(isset($simpleXMLElement->values)){
            foreach ($simpleXMLElement->values->children() as $property => $value) {
                $setter = 'set' . self::camelize($property);
                if(is_callable([$item, $setter])){
                     @$item->$setter((string)$value);
                     $item->variablesSet[strtolower($property)] = true;
                }
            }
        }
        if (filter_var($item->getSourceSimpleXML()->auto_save, FILTER_VALIDATE_BOOLEAN)){
            return $item;
        }
        if($item instanceof IHasAttachments) $item::addAttachmentsFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasAttendees) $item::addAttendeesFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasCertificates) $item::addCertificatesFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasLocations) $item::addLocationsFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasRecurrences) $item::addRecurrencesFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasReminders) $item::addRemindersFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasItemAttributes) $item::addItemAttributesFromSimpleXML($item, $item->getSourceSimpleXML());
        if($item instanceof IHasEventReactions) $item::addEventReactionsFromSimpleXML($item, $item->getSourceSimpleXML());
        return $item;
    }

     
    public static function fillValuesArray(AGwEventItem $item, array $values = null)
    {
        foreach ($values as $property => $value) {
            $setter = 'set' . self::camelize($property);
            if(is_callable([$item, $setter])) @$item->$setter((string)$value);
        }
        return $item;

     }

     
    protected function addMultipleToICalendar(Component $document, array $items) : void
    {
        foreach ($items as $item) {
            $item->addToICalendar($document);
        }
    }

     
    protected function serialize(Component $document, Component $addonsPart = null) : string
    {
        $addonsPart = $addonsPart ?? $document;
        if($this instanceof IHasAttachments) $this->addMultipleToICalendar($addonsPart, $this->getAttachments());
        if($this instanceof IHasAttendees) $this->addMultipleToICalendar($addonsPart, $this->getAttendees());
        if($this instanceof IHasCertificates) $this->addMultipleToICalendar($addonsPart, $this->getCertificates());
        if($this instanceof IHasLocations) $this->addMultipleToICalendar($addonsPart, $this->getLocations());
        if($this instanceof IHasRecurrences) $this->addMultipleToICalendar($addonsPart, $this->getRecurrences());
        if($this instanceof IHasReminders) $this->addMultipleToICalendar($addonsPart, $this->getReminders());
        if($this instanceof IHasItemAttributes) $this->addMultipleToICalendar($addonsPart, $this->getItemAttributes());
        if($this instanceof IHasEventReactions) $this->addMultipleToICalendar($addonsPart, $this->getEventReactions());
        return $document->serialize();
    }

     
    public static function createFromXML(\Folder $folder, string $xml, bool $save = false, AGwEventItem $parent = null)
    {
        $class = get_called_class();
         
        $item = new $class($folder);
        if($parent instanceof AGwEventItem) $item->setParent($parent);
        self::fillValues($item, simplexml_load_string($xml));
        if($save) $item->save();
        return $item;
    }

    public function setDuplicity($duplicity) {}

     
    public function prepareParamLine(string $prefix = '', string $postfix = '', bool $skipEmpty = true, array $allowEmpty = []) : bool
    {
        if(!empty($this->getId())){
            $changes = $this->getChangedFromBackup(false, $skipEmpty, $allowEmpty);
            $skipVariablesExport = array_merge(self::$skipVariablesExport, static::$skipVariablesExport ?? [], static::$skipVariablesExportGW ?? []);
            $dependableVariablesExport = array_merge(self::$dependableVariablesExport, static::$dependableVariablesExport ?? [], static::$dependableVariablesExport ?? []);
            $renameVariablesExport = array_merge(self::$renameVariablesExport, static::$renameVariablesExport ?? [], static::$renameVariablesExportGW ?? []);
            $changesKeys = array_keys($changes);
            foreach ($changes as $property => $value) {
                $check = lcfirst(self::camelize(strtolower($property)));
                if(in_array($check, $skipVariablesExport)){
                    unset($changes[$property]);
                }elseif(isset($renameVariablesExport[$check])){
                    $changes[$renameVariablesExport[$check]] = $value;
                    unset($changes[$property]);
                }
                if(isset($dependableVariablesExport[$check])){
                    $found = false;
                    foreach($dependableVariablesExport[$check] as $dependableVariable){                     
                        if(in_array(strtoupper($dependableVariable), $changesKeys)){
                            $found = true;
                        }
                    } 
                    if(!$found){
                        unset($changes[$property]);
                    }  
                }  
            }
        }else{
            $changes = $this->getAsArrayGW(false, $skipEmpty, $allowEmpty);
        }
        if(empty($changes)) return false;
        $result = [];
        if(!empty($prefix)) $result[] = $prefix;
        $result[] = http_build_query($changes, '', '&', PHP_QUERY_RFC3986);
        if (method_exists($this, 'getDuplicity') && $this->getDuplicity() == 'replace') {
            $this->setDuplicity(null);
            $result[] = 'forcereplace=1';
        }
        if(!empty($postfix)) $result[] = $postfix;
        $this->preparedParamLine = $result;
        return true;
    }

     
    public function getParamLine()
    {
        if(empty($this->preparedParamLine)) return "";
        $result = implode('&', $this->preparedParamLine);
        $this->preparedParamLine = null;
        $this->backupState();
        return $result;
    }

    public function getAsArrayGW($encode = false, $skipEmpty = true, array $allowEmpty = [])
    {
        $skipVariablesExport = array_merge(self::$skipVariablesExport, static::$skipVariablesExport ?? [], static::$skipVariablesExportGW ?? []);
        $renameVariablesExport = array_merge(self::$renameVariablesExport, static::$renameVariablesExport ?? [], static::$renameVariablesExportGW ?? []);
        $lengthVariablesExport = array_merge(self::$lengthVariablesExport, static::$lengthVariablesExport ?? [], static::$lengthVariablesExportGW ?? []);
        return $this->getAsArray($encode, $skipEmpty, $allowEmpty, $skipVariablesExport, $renameVariablesExport, $lengthVariablesExport);
    }

    public function getAsArray($encode = false, $skipEmpty = true, array $allowEmpty = [], array $skipVariablesExport = null, array $renameVariablesExport = null, array $lengthVariablesExport = null)
    {
        try {
            if(is_null($skipVariablesExport)) $skipVariablesExport = array_merge(self::$skipVariablesExport, static::$skipVariablesExport ?? []);
            if(is_null($renameVariablesExport)) $renameVariablesExport = array_merge(self::$renameVariablesExport, static::$renameVariablesExport ?? []);
            if(is_null($lengthVariablesExport)) $lengthVariablesExport = array_merge(self::$lengthVariablesExport, static::$lengthVariablesExport ?? []);
            $this->xml = '';
            $result = [];
            $reflection = new \ReflectionClass(static::class);
            $properties = array_merge($reflection->getProperties(), $allowEmpty);
            foreach ($properties as $property) {
                $propertyName = (is_object($property)) ? $property->getName() : $property;
                $getter = 'get' . self::camelize($propertyName);
                $setter = 'set' . self::camelize($propertyName);
                if(!method_exists($this, $getter) || in_array($propertyName, $skipVariablesExport)) continue;
                $value = $this->getItemScalarValue($this->$getter());
                $name = $renameVariablesExport[$propertyName] ?? strtoupper(self::snakeize($propertyName));
                if(isset($result[$name])) continue;
                if($skipEmpty && $value == '' && !in_array(strtolower($name), $allowEmpty) && !isset($this->variablesSet[strtolower($propertyName)])) continue;
                if($lengthVariablesExport[$propertyName]) $value = iconv_substr(\slToolsString::utf8_bad_replace($value), 0, $lengthVariablesExport[$propertyName], 'utf-8');

                $result[$name] = ($encode) ? rawurlencode($value) : $value;
                $nameXML = strtolower($name);
                if($value === ''){
                    $this->xml .= '<' . $nameXML . '/>' . PHP_EOL;
                }else{
                    $this->xml .= '<' . $nameXML . '>' . htmlspecialchars($result[$name]) . '</' . $nameXML . '>' . PHP_EOL;
                }
            }
        }catch (\Exception $e){
            throw new \Exc();
        }
        return $result;
    }

    public function getAsArrayFE($encode = false, $skipEmpty = true, array $allowEmpty = [], array $skipFields = [], array $renameFields = [])
    {
        $skipVariablesExport = array_merge(self::$skipVariablesExport, static::$skipVariablesExport ?? [], static::$skipVariablesExportFE ?? [], $skipFields);
        $renameVariablesExport = array_merge(self::$renameVariablesExport, static::$renameVariablesExport ?? [], static::$renameVariablesExportFE ?? [], $renameFields);
        $lengthVariablesExport = array_merge(self::$lengthVariablesExport, static::$lengthVariablesExport ?? [], static::$lengthVariablesExportFE ?? []);
        return $this->getAsArray($encode, $skipEmpty, $allowEmpty, $skipVariablesExport, $renameVariablesExport, $lengthVariablesExport);
    }

    protected function getItemScalarValue($value)
    {
        if(is_scalar($value)) return $value;
        if($value instanceof \DateTime) return $value->getTimestamp();
        if($value instanceof \Folder) return $value->name;
        return false;
    }

    public function getArray()
    {
        $result = [];
        foreach (get_object_vars($this) as $property => $value) {
            if(!is_scalar($value)) continue;
            $result[strtoupper($property)] = $value;
        }
        return $result;
    }

    public function getId()
    {
        return $this->getEvnId();
    }

     
    public function save() : AGwEventItem
    {
        if(is_callable([$this, 'beforeSave'])){
            if(!$this->beforeSave()) return $this;
        }
                 $sEvnId = $this->getGwApi()->FunctionCall('ProcessVCalendar', $this->getFolderSessionId(), $this->getId() ?? '', $this->getICalendar(), 'rfc6638=&action=edit');
                  
        $this->setId($sEvnId);
        $this->setAttWebdavLink(\User::addTokenToTicket($this->getGwApi()->FunctionCall('GetAttachmentPathLocal', $this->getFolderSessionId(), $this->getId(), '', 'TICKET')));
        return $this;
    }

     
    public function delete()
    {
        if(empty($this->getId())) return;
         
        $gwApi = $this->getFolder()->account->gwAPI;
        $folderSessionId = $this->getFolder()->openAccess();
        return $gwApi->FunctionCall('DeleteEvent', $folderSessionId, $this->getId(), '0');
    }

    public function setId($value)
    {
        if(empty($this->getId())) $this->setEvnId($value);
        return $this;
    }

    protected function getFolderSessionId()
    {
        if(!empty($this->folderSessionId)) return $this->folderSessionId;
        return $this->folderSessionId = $this->getFolder()->openAccess() ?? $this->folder->account->gwAPI->OpenGroup();
    }

    protected function replaceCID(string & $data)
    {
        if(!$this instanceof IHasAttachments) return;
        $pattern = '/(?P<prefix>\<img[^>]+src=").*?(?:(?:sid=' . preg_quote($_SESSION['SID']) . ')|(?:dlsess=' . $_SESSION['DLSESS'] . '))&amp;class=(?P<class>[^&]+)&amp;fullpath=(?P<fullpath>(?:[^\'"&\/]+\/)+(?P<name>[^"&]+)[^"]*)(?P<suffix>[^>]+>)/i';
        if(preg_match_all($pattern, $data, $matches, PREG_SET_ORDER)){
            foreach ($matches as $match) {
                $attachment = new \server\model\Attachment($this->getFolder());
                $attachment->setFullpath(urldecode($match['fullpath']))
                    ->setClass($match['class'])
                    ->setDescription(urldecode($match['name']))
                    ->setCid('attachment_id_'.md5($attachment->getDescription()));
                $this->addAttachment($attachment);
                $data = str_replace($match[0], $match['prefix'] . $attachment->getCid() . $match['suffix'], $data);
            }
        }
        return $data;
    }

    public function getGroupWareItem()
    {
        $folder = $this->getFolder();
        if(!$folder instanceof GroupWareFolder){
            $folder = $folder->account->gwAPI->getFdr($folder);
        }
        $gwi = new \GroupWareItem($folder, $this->getAsArray());
        $gwi->sFields = $this->xml;
        $gwi->itemID = $this->getId();
        $gwi->att_webdav_link = $this->getAttWebdavLink();
        return $gwi;
    }

    public function addChildrenTemplateDataArray(array & $result, array $children, $groupName, $name, $uidField = null, array $skipFields = [], array $renameFields = [], array $allowEmpty = [])
    {
        $useUid = false;
        $getter = '';
        if(!empty($uidField)){
            $getter = 'get' . self::camelize($uidField);
        }
        if(is_callable([current($children), $getter])){
            $useUid = true;
        }
        $xml = [];
        foreach ($children as $child) {
            $result[$groupName][] = $child->getAsArrayFE(false, true, $allowEmpty, $skipFields, $renameFields);
            if($useUid){
                $xml[$child->$getter()] = $child->xml;
            }else{
                $xml[] = $child->xml;
            }
        }
        $resultXml = '';
        if(!empty($xml)){
            $resultXml .= "<$groupName>";
            foreach ($xml as $uid => $part) {
                $resultXml .= "<$name" . ($useUid ? ' uid="' . $uid . '"' : '') . "><values>$part</values></$name>";
            }
            $resultXml .= "</$groupName>";
        }

        $result['sFields'] .= $resultXml;
    }

     
    public static function arrayOrderBy(array &$array, array $rules)
    {
        $getters = [];
        $test = current($array);
        if(!$test instanceof AGwEventItem) return $array;
        foreach ($rules as $rule) {
            $function = 'get' . self::camelize($rule['key']);
            if(is_callable([$test, $function])) $getters[$rule['key']] = $function;
        }
        usort($array, function ($a, $b) use ($rules, $getters) {
            $retval = 0;
            foreach ($rules as $rule) {
                $retval = strcmp($a->{$getters[$rule['key']]}(), $b->{$getters[$rule['key']]}());
                if($retval != 0) break;
            }
            if (strcasecmp($rule['direction'], 'DESC') === 0) $retval *= -1;
            return $retval;
        });
        return $array;
    }

     public function getFolder() : \Folder
    {
        return $this->folder;
    }
    abstract public function getICalendar() : string;
    abstract public function getType() : string;
    abstract protected function afterLoad() : void;

     
    abstract public function getTemplateDataArray(array $allowEmpty = []) : array;

     
    public function setFilterTag(array &$filterTag) : AGwEventItem
    {
        $this->filterTag = $filterTag;
        if(isset($filterTag['tag'])){
            $this->filterTag['tag'] = explode(',', $filterTag['tag']);
            $this->filterTag['tag'] = array_map('trim', $this->filterTag['tag']);
        }
        return $this;
    }

     
    public function getFilterTag(): array
    {
        return $this->filterTag;
    }

     
    public function getGwApi(): \MerakGWAPI
    {
        if(! $this->gwApi instanceof \MerakGWAPI) $this->gwApi = $this->getFolder()->account->gwAPI ?? $this->getFolder()->account->gwAccount->gwAPI;
        return $this->gwApi;
    }

    static protected function fixEscape($dbtype, $dbsyntax, $condition)
	{
		switch(strtolower($dbtype)){
			case 'mysql':
				$condition = preg_replace('/(\{(ESCAPE \'.\')\})/si','',$condition);
				$condition = str_replace('\\','\\\\',$condition);
			break;
			case 'mssql':
			case 'odbc':
			case 'oci':
				if($dbsyntax=='oracle'){
					$condition = preg_replace('/(\{(ESCAPE \'.\')\})/si','',$condition);
				}
				$condition = str_replace('\\','\\\\',$condition);
				$condition = preg_replace('/(\{(ESCAPE \'([^\']+)\')\})/si','{ESCAPE \'\\\'}',$condition);
			break;
			case 'sqlite':
			default:
				$condition = str_replace('\\','\\\\',$condition);
				$condition = preg_replace('/(\{(ESCAPE \'([^\']+)\')\})/si','ESCAPE \'\\\'',$condition);
			break;
		}
		
		return $condition;
	}

}
