<?php

namespace server\inc\imap;

use tools\SearchParserYoda;

 
class Search extends SearchParserYoda
{
     
    public $folder;

     
    public static $_allowedKeywords = [
        'from',
        'to',
        'subject',
        'after',
        'before',
        'has',
        'fulltext',
        'is',
        'tag',
        'taglist',
        'yoda',
        'with',
        'last',
        'color',
        'bcc',
        'cc',
        'message-id',
        'greater',
        'smaller',
    ];

     
    public static $_keywordTranslations = [
        'after' => 'SINCE',
        'has' => 'HEADER',
        'fulltext' => 'TEXT',
        'tag' => 'KEYWORD',
        'taglist' => 'KEYWORD',
        'yoda' => 'X-YODA',
        'with' => 'X-WITH',
        'last' => 'SINCE',
        'color' => 'X-COLOR',
        'message-id' => 'HEADER MESSAGE-ID',
        'greater' => 'LARGER',
        'smaller' => 'SMALLER',
    ];

    public static $xColor = [
        'Y' => 'DONE',
        'Z' => 'NIL',
        1 => 'RED',
        2 => 'BLUE',
        3 => 'GREEN',
        5 => 'ORANGE',
        8 => 'PURPLE',
        'A' => 'YELLOW',
    ];

    static public $_keywordValuesTranslations = [
        'is' => ['unread' => 'UNSEEN', 'flagged' => 'FLAGGED', 'genuine' => 'NOT X-ISSPAM', 'spam' => 'X-ISSPAM'],
        'has' => ['attachment' => 'HASATTACHMENT'],
    ];

    protected static $skipSearchConditions = [];

    public static $_default_keyword = 'fulltext';

     
    protected $conditions = [];

    protected $isVirtual = false;
    protected $imap;
    public $enabled = true;

     
    public function __construct(\Folder $folder, string $searchCondition = '')
    {
        $this->enabled = $_SESSION['FULLTEXT_SUPPORT'];
        if(in_array($searchCondition, self::$skipSearchConditions)) $this->enabled = false;
        if($folder instanceof \VirtualFolder) $this->isVirtual = true;
        $this->folder = $folder;
                 try {
            $this->conditions = self::parseExpression($searchCondition);
        }catch (\Exception $e){
            $this->enabled = false;
        }
    }

     
    public function getCriteria() : string
    {
        return $this->__toString();
    }

     
    public function getSearchResults()
    {
        $imap = $this->getImap();
        if(!$imap instanceof \IMAP) return false;
        $imap->openMailbox($this->folder->name);
        $result = imap_search($imap->getImapResource(), $this->getCriteria(), SE_UID);
        return $result;
    }

     
    public function isSearchEnabled(): bool
    {
        if(!$this->enabled) return false;
        $imap = $this->getImap();
        if(!$imap instanceof \IMAP) return false;
        if(!$this->isVirtual) return true;
        return $imap->getCapability('MULTISEARCH');
    }

     
    protected function getImap()
    {
        if(!$this->imap instanceof \IMAP) {
            try {
                $account = $this->folder->account;
                if ($this->isVirtual) $account = $account->account;
                if(!$account instanceof \IMAPAccount) return false;
                $this->imap = \IMAP::instance($account);
            }catch (\Exception $exception){
                return false;
            }
        }
        return $this->imap;
    }

     
    protected function getMultiSearchResults(& $folderIds = [], array $useFolders = [], $allFolders = false)
    {
        if(!empty($useFolders)){
            $folders = $useFolders;
        }elseif(!empty($this->folder->folders ?? null)){
            $folders = $this->folder->folders;
        }else{
            $account = $this->folder->account->account;
            $folders = $account->folders['main'];
            $allFolders = true;
        }
        if(!($this->folder instanceof \SnoozedFolder) && empty($folders)){
            $folders = [$this->folder];
            $allFolders = false;
        }
        $location = '(personal)';
                 if($this->folder instanceof \VirtualFolder){
            if($this->folder->getScope() != ''){
                $location = $this->folder->getScopeLocation();
                $allFolders = true;
            }
        }

        $imap = $this->getImap();
        if(!$imap instanceof \IMAP) return '(0 = 1)';
        $folderNames = [];
        foreach($folders as $folder){
            $encodedName = $imap->encode($folder->name);
            $folderIds[strtolower($encodedName)] = $folder->folderID;
            $folderNames[] = '"' . $encodedName . '"';
        }
        if(!$allFolders){
            $location = '(mailboxes (' . join(' ', $folderNames) . '))';
        }
        $criteria = $this->getCriteria();
        if(!$criteria){
            $criteria = 'TEXT "*"';
        }
        $ids = imap_msearch($imap->getImapResource(), $location, $criteria, '(X-NOFALLBACK ALL)');
        return $ids;
    }

     
    public function getMultiSearchSql($useFolders = [], $allFolders = false)
    {
        $imap = $this->getImap();
        $searchResult = $this->getMultiSearchResults($folderIds, $useFolders, $allFolders);
                 if($searchResult === false) {
            $this->fallback = true;
            return false;
        }
                 if(empty($searchResult)){
             return '(0 = 1)';
        }
                 $conditions = $ids = [];
        foreach ($searchResult as $folder => $list) {
            $folderName = strtolower(trim($folder));
            $ids[$folderName] = [];
            foreach($list as $key => $val){
                $ids[$folderName][$key] = \IMAP::fixID($val);
            }
            if(!empty($ids[$folderName])){
                $conditions[] = '(' . 'folder_id = \'' . $folderIds[$folderName] . '\' AND (RID IN (\'' . implode("','", $ids[$folderName]).'\')))';
            }
        }
        return !empty($conditions) ? '(' . implode(' OR ', $conditions) . ')' : '(0 = 1)';
    }

    protected function correctKeywordValue($value)
    {
        return \slToolsString::urlenquote($value);
    }

     
    protected function correctAfterValue($value)
    {
        return $this->correctDate($value);
    }

     
    protected function correctLastValue($value)
    {
        if(!is_numeric($value)) return $this->correctDate($value);
        $value = intval($value)-1;          $datetime = new \DateTime();
        $datetime->modify('-' . $value . ' days');
        return $datetime->format('d-M-Y');
    }

     
    protected function correctBeforeValue($value)
    {
        return $this->correctDate($value);
    }

     
    protected function correctDate($date, $scalar = true)
    {
        $datetime = date_create_from_format('Y-m-d', $date);
        if(!$datetime instanceof \DateTime) $datetime = new \DateTime();
        return $scalar ? $datetime->format('d-M-Y') : $datetime;
    }

    protected function correctColorValue($value)
    {
        $value = trim(strtoupper($value), '"\'');
        if(isset(self::$xColor[$value])) return self::$xColor[$value];
        return $value;
    }

     
    public function getSearchSql(&$alreadySearched = false, $useFolders = [], $allFolders = false)
    {
        if(!$this->enabled){
            return false;
        }
        $alreadySearched = true;
        if($this->isVirtual || $this->getImap()->getCapability('MULTISEARCH')){
            return $this->getMultiSearchSql($useFolders, $allFolders);
        }
        $searchResult = $this->getSearchResults();
        if($searchResult === false){
            $this->fallback = true;
            return false;
        }
        if(is_array($searchResult)) $searchResult = array_map('IMAP::fixID', $searchResult);
        return !empty($searchResult) ? '(RID IN (\'' . implode('\',\'', $searchResult) . '\'))' : '(0 = 1)';
    }

     
    protected function getFinalValue(array $condition)
    {
        $condition['value'] = \slToolsString::removeQuotes($condition['value']);
        $function = 'correct' . ucfirst(strtolower($condition['keyword'])) . 'Value';
        $function = preg_replace( '/[\W]/', '', $function);
        return is_callable([$this, $function]) ? $this->{$function}($condition['value']) : $condition['value'];
    }


     
    public function __toString() : string
    {
        foreach ($this->conditions as $key => $condition) {
            $this->conditions[$key]['value'] = $this->getFinalValue($condition);
        }
        return self::getImapCriteriaFromParsedData($this->conditions);
    }

     
    protected function correctGreaterValue($value)
    {
                 return (int)$this->kiloByteToByte($value);
    }

     
    protected function correctSmallerValue($value)
    {
                 return $this->kiloByteToByte($value);
    }

     
    protected function kiloByteToByte($value)
    {
        return (int)$value * 1024;
    }
}
