<?php

namespace server\inc\alfresco;

class Api
{
     
    protected $url;
     
    protected $authorization = null;
     
    protected $account;
     
    public $lastPagination = null;
     
    protected $cookies;
         protected $tokenPrefix = 'Bearer';
    protected $exception;
    protected $tokenTimeout;
    protected $useUsername;
    protected $superuser;
    protected $superuserPassword;
    protected $ticket;
    protected $tokenTimeoutDefault;

     
    public function __construct(string $url, string $token, \AlfrescoAccount $account, ? string $superuser, ? string $superuserPassword, $useUsername = false, int $tokenTimeout = 3600)
    {
                 if (!function_exists('curl_init')) include_once __DIR__ . '/CurlEmu.php';
        $this->url = $url;
        $this->account = $account;
        $this->tokenPrefix = $token;
        $this->useUsername = $useUsername;
        $this->superuser = $superuser;
        $this->superuserPassword = $superuserPassword;
        $this->tokenTimeoutDefault = $tokenTimeout;
    }

     
    public function curlResponseHeaderCallback($ch, $headerLine) {
        if (preg_match('/^Set-Cookie:\s*(?P<key>[^=;]+)?=?(?P<value>[^;]*)/mi', $headerLine, $match)){
            $this->cookies[$match['key'] ?? (count($this->cookies))] = $match['value'];
        }
        return strlen($headerLine);      }

     
    protected function getData(string $method, string $url, array $params = [], bool $plain = false)
    {
        $headers = ['Accept' => 'Accept: application/json'];
        if(!empty($this->authorization)){
            $headers['Authorization'] = 'Authorization: ' . $this->authorization;
        }else{
            $headers['Authorization'] = 'Authorization: Basic ' . base64_encode($params['userId'] . ":" . $params['password']);
        }

        $ch = curl_init($this->url . $url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if($method == 'POST'){
            $dataString = json_encode($params);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $dataString);
            $headers['Content-Length'] = 'Content-Length: ' . strlen($dataString);
            $headers['Content-Type'] = 'Content-Type: application/json';
        }elseif($method == 'GET'){
            $getParams = http_build_query($params);
            curl_setopt($ch, CURLOPT_URL, $this->url . $url . '?' . $getParams);
        }elseif ($method == 'FILE'){
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
                     }
        if(!empty($this->cookies)){
            foreach ($this->cookies as $key => $value) {
                $headers[] = 'Cookie: ' . (is_numeric($key) ? '' : $key . '=') . $value;
            }
        }
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
        curl_setopt($ch, CURLOPT_ENCODING, '');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_HEADERFUNCTION, [&$this, 'curlResponseHeaderCallback']);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
		curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);

        $serverOutput = curl_exec($ch);
        curl_close($ch);
        if($plain) return $serverOutput;
        $data = json_decode($serverOutput);

        if(!$data instanceof \stdClass){
            throw new \Exc(500, 'ALFRESCO_NO_DATA');
        }
        if(isset($data->error)){
                         if($data->error->statusCode == 401){
                $this->logout();
                $this->authorization = null;
                if($this->connect()) return $this->getData($method, $url, $params, $plain);
            }
            $this->handleError($data->error);
        }
        if(isset($data->list->pagination)){
            $this->lastPagination = $data->list->pagination;
        }
        return $data;
    }

     
    public function connect(
        ?string $username = null,
                 ?string $password = null
    ) : bool
    {
        try {
            if($this->tokenTimeout > time()) $this->logout();
            if(!empty($this->authorization)) return true;
            if(empty($username) || empty($password)){
                 
                $user = $_SESSION['user'];
                $username = ($this->useUsername) ? $_SESSION['U_USERNAME'] : current(explode('@', $user->username, 2));
                $account = new \IceWarpAccount();
                $account->Open($_SESSION['EMAIL']);
                $password = $user->getPassword();
                if($account->GetProperty('u_authmode') == 2){
                    $password = substr($password, 32);
                }
            }
            $data = $this->getData('POST', '/-default-/public/authentication/versions/1/tickets', ['userId' => $username, 'password' => $password]);
            $this->ticket = $data->entry->id;
            if($this->tokenPrefix == 'Basic') $this->ticket = base64_encode($this->ticket);
            $this->authorization = $this->tokenPrefix . ' ' . $this->ticket;
            $this->tokenTimeout = time()+$this->tokenTimeoutDefault;
            return true;
        }catch (\Exception $e){
            $this->exception = $e;
            return false;
        }
    }

     
    protected function handleError(\stdClass $error)
    {
        $briefSummary = preg_replace('/^[0-9\\s]+/', '', $error->briefSummary);
        throw new \Exc('', $briefSummary);
    }

    protected function logout()
    {
        try {
            $this->getData('DELETE', '/-default-/public/authentication/versions/1/tickets/-me-', [], true);
            unset($this->authorization, $this->cookies);
        }catch (\Exception $e){};
    }

     
    public function urlReachable() : bool
    {
        $ch = curl_init($this->url . '/-default-/public/authentication/versions/1/tickets/-me-');
        curl_setopt_array($ch, [
            CURLOPT_AUTOREFERER    => true,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_ENCODING       => "",
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 1,
            CURLOPT_NOBODY         => true,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT        => 5,
        ]);
        curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return (bool) $code;
    }


     
    public function checkUser() : bool
    {
        if(!$this->urlReachable()) return false;
        if(empty($this->superuser) || empty($this->superuserPassword)) return true;
        try{
            try{
                $passwordDecrypted = \slToolsCrypt::decryptSymmetric($this->superuserPassword);
            }catch(\Exception $e){ $passwordDecrypted = null;}
            if(empty($passwordDecrypted)){
                $passwordDecrypted = $this->superuserPassword;
                @$xml = \slToolsDOM::open(GLOBAL_SETTINGS_FILE,true);
                $xml->getElementsByTagName("alfrescosuperuserpassword")->item(0)->nodeValue = \slToolsCrypt::encryptSymmetric($this->superuserPassword);
                @$xml->save(GLOBAL_SETTINGS_FILE,true);
            }
            $user = ($this->useUsername) ? $_SESSION['U_USERNAME'] : current(explode('@', $_SESSION['user']->username, 2));
            $this->connect($this->superuser, $passwordDecrypted);
            $data = $this->getData('GET', '/-default-/public/alfresco/versions/1/people/' . $user);
        }catch (\Exception $e){}
        $this->logout();
        return $data->entry->enabled ?? false;
    }

     
    public function getDefaultFolderName()
    {
        return '-root-';
    }

     
    public function getFolders(\AlfrescoFolder $folder, $limit = 0, $offset = 0)
    {
        $result = [];
        $params = $this->getParamsFromPaging($limit, $offset);
        $params['where'] = '(isFolder=true)';
        $params['orderBy'] = 'name';
        $data = $this->getData('GET', '/-default-/public/alfresco/versions/1/nodes/' . $folder->folderID . '/children', $params);
        foreach ($data->list->entries as $item) {
            if(!$item->entry->isFolder) continue;
            $baseName = $item->entry->name;
            $name = ($folder->name == $this->getDefaultFolderName()) ? $item->entry->name : $folder->name . '/' . $item->entry->name;
            $entry = new \AlfrescoFolder($this->account, $name, $baseName, $item->entry->id);
            $result[$name] = $entry;
        }
        return $result;
    }

     
    protected function getItemFromEntry(\stdClass $entry)
    {
        return [
            'EVN_ID' => $entry->id,
            'EVNCLASS' => 'F',
            'EVNTITLE' => $entry->name,
            'EVNLOCATION' => $entry->name,
            'EVN_MODIFIED' => strtotime($entry->modifiedAt),
            'EVN_CREATED' => strtotime($entry->createdAt),
            'EVNCOMPLETE' => $entry->content->sizeInBytes,
        ];
    }

     
    protected function getParamsFromPaging(int $limit, int $offset)
    {
        $params = [];
        if($limit > 0) $params['maxItems'] = $limit;
        if($offset > 0) $params['skipCount'] = $offset;
        return $params;
    }

     
    public function getItems(? \AlfrescoFolder $folder, string $orderBy = '', int $limit = 0, int $offset = 0, ? string $search = null)
    {
        if(!$folder instanceof \AlfrescoFolder){
            $folder = new \AlfrescoFolder($this->account, 'Alfresco', $this->getDefaultFolderName(), $this->getDefaultFolderName());
        }
        $filterBySearch = !empty($search);
        $result = [];
        $params = $this->getParamsFromPaging($limit, $offset);
        $params['where'] = '(isFile=true)';
        $data = $this->getData('GET', '/-default-/public/alfresco/versions/1/nodes/' . $folder->folderID . '/children', $params);
        foreach ($data->list->entries as $item) {
            if(!$item->entry->isFile || ($filterBySearch && stripos($item->entry->name, $search) === false)) continue;
            $result[] = new \AlfrescoItem($folder, $this->getItemFromEntry($item->entry));
        }
        return $result;
    }

     
    public function getItem(\AlfrescoFolder $folder = null, string $id = null)
    {
        $data = $this->getData('GET', '/-default-/public/alfresco/versions/1/nodes/' . $id);
        if(!$data->entry->isFile){
            if(!$data->entry->isFolder) return null;
            $baseName = $data->entry->name;
            $name = $data->entry->name;
            if($folder instanceof \AlfrescoFolder){
                $name = ($folder->name == $this->getDefaultFolderName()) ? $data->entry->name : $folder->name . '/' . $data->entry->name;
            }
            return new \AlfrescoFolder($this->account, $name, $baseName, $data->entry->id);
        }
        return new \AlfrescoItem($folder, $this->getItemFromEntry($data->entry));
    }

     
    public function getFileContent(string $id)
    {
        return $this->getData('GET', '/-default-/public/alfresco/versions/1/nodes/' . $id . '/content', ['attachment' => 'true'], true);
    }

     
    public function getSearchedItems(? \AlfrescoFolder $folder, array $filter)
    {
        $result = [];
        $search = $filter['search'];
        if(strlen($search) < 3) return [];
        $search = [
            'query' => [
                'query' => "select * from cmis:document WHERE CONTAINS('cmis:name:*$search*')",
                'language' => 'cmis',
            ],
            'include' => ['path']
        ];
        $data = $this->getData('POST', '/-default-/public/search/versions/1/search', $search);
        foreach ($data->list->entries as $item) {
            if(!$item->entry->isFile) continue;
            $result[] = $this->getItemFromSearchResult($item->entry);
        }
        $this->lastPagination = null;
        return $result;
    }

     
    protected function getItemFromSearchResult(\stdClass $itemData)
    {
        $stripElement = array_shift($itemData->path->elements);
        $resultFolderPath = str_replace('/' . $stripElement->name . '/','',$itemData->path->name);
        if(!isset($this->account->folders[$resultFolderPath])){
            $pathFolders = [];
            foreach ($itemData->path->elements as $element) {
                $pathFolders[] = $element->name;
                $fullPath = implode('/', $pathFolders);
                if(!isset($this->account->folders[$fullPath])){
                    $this->account->folders[$fullPath] = new \AlfrescoFolder($this->account, $fullPath, $element->name, $element->id);
                }
            }
            $this->account->folders[$resultFolderPath] = new \AlfrescoFolder($this->account, $resultFolderPath, $itemData->name, $itemData->id);
        }
        $folder = $this->account->folders[$resultFolderPath];
        return new \AlfrescoItem($folder, $this->getItemFromEntry($itemData));
    }

     
    public function getSearchResult(string $search)
    {
        $result = [];
        if(strlen($search) < 3) return $result;
        $search = [
            'query' => [
                'query' => "select * from cmis:folder WHERE CONTAINS('cmis:name:*$search*')",
                'language' => 'cmis',
            ],
            'include' => ['path']
        ];
        $data = $this->getData('POST', '/-default-/public/search/versions/1/search', $search);
        foreach ($data->list->entries as $item) {
            if(!$item->entry->isFolder) continue;
            $entry = $this->getFolderFromSearchResult($item->entry);
            $result[$entry->name] = $entry;
        }
        $this->lastPagination = null;
        return $result;
    }

     
    protected function getFolderFromSearchResult(\stdClass $itemData)
    {
        $stripElement = array_shift($itemData->path->elements);
        $resultFolderPath = str_replace('/' . $stripElement->name . '/','',$itemData->path->name . '/' . $itemData->name);
        if(isset($this->account->folders[$resultFolderPath])) return $this->account->folders[$resultFolderPath];
        $pathFolders = [];
        foreach ($itemData->path->elements as $element) {
            $pathFolders[] = $element->name;
            $fullPath = implode('/', $pathFolders);
            if(!isset($this->account->folders[$fullPath])){
                $this->account->folders[$fullPath] = new \AlfrescoFolder($this->account, $fullPath, $element->name, $element->id);
            }
        }
        $result = new \AlfrescoFolder($this->account, $resultFolderPath, $itemData->name, $itemData->id);
        $this->account->folders[$resultFolderPath] = $result;
        return $result;
    }

     
    public function create(\Folder $parentFolder, string $filePath, string $fileName)
    {
        $curlFile = curl_file_create(realpath($filePath), '', $fileName);
        $data = $this->getData('FILE', '/-default-/public/alfresco/versions/1/nodes/' . $parentFolder->folderID . '/children', ['filedata' => $curlFile, 'autoRename' => 'true']);
        if(!$data->entry->isFile) return null;
        return $data->entry->id;
    }

     
    public function getException()
    {
        return $this->exception;
    }
}