(function () {
	var files = [];
	var conversations = [];
	var processQueue = {};

	function processProcessQueue(conversation) {
		if (processQueue[conversation.id] && processQueue[conversation.id].length && !conversation.processing) {
			processQueue[conversation.id].pop()
			conversation.process();
		}
	}

	function WMChatGPT() {
		this.models = [];
	}

	function normalizeAlias(alias) {
		return (alias || '').replace(/^\*/g, '').replace(/[^\w]/g, '_');
	}

	WMChatGPT.getModel = function(model) {
		for (var i in this.models) {
			if (this.models[i].model === model) {
				return this.models[i];
			}
		}
		return {};
	};

	WMChatGPT.getModelName = function(model) {
		return (model || '').replace(/dalle(\d+).*/, 'DALL·E $1').replace(/gpt-([\d.]+).*/, 'GPT $1');
	};
	
	WMChatGPT.prototype.modelsList = function (callback) {
		if (this.models.length) {
			return callback && callback(true, this.models);
		}

		request({
			url: 'model'
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MODELS_LIST'}});
				return callback && callback(bOK, response);
			}

			this.models = response.map(function(model) {
				model.label = WMChatGPT.getModelName(model.model);
				return model;
			}, this);

			this.models.unshift({
				label: getLang('CHATGPT::ASSISTANT'),
				model: 'assistant',
				type: 'assistant'
			})

			callback && callback(bOK, this.models);
		}.bind(this));
	};

	WMChatGPT.prototype.getConversation = function(id, callback) {
		if (!conversations.some(function(conversation) {
			if (conversation.id === id) {
				callback && callback(true, conversation);
				return true;
			}
		})) {
			var conversation = new ChatGPTConversation().data({id: id});
			conversations.push(conversation);
			conversation.detail(callback);
		}
	};

	WMChatGPT.prototype.conversationsList = function (limit, offset, callback) {
		request({
			url: 'conversation',
			data: {
				limit: limit || 5,
				offset: offset || 0
			}
		}, function (bOK, response) {
			response = response || [];

			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATIONS_LIST'}});
				return callback && callback(bOK, response);
			}

			response = response.map(function(data) {
				var conversation;
				for (var i in conversations) {
					if (conversations[i].id === data.id) {
						conversation = conversations[i];
						break;
					}
				}
				if (!conversation) {
					conversation = new ChatGPTConversation();
					conversations.push(conversation)
				}

				return conversation.data(data);
			});

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.getConversationByAlias = function(data, callback) {
		if (!data.alias) {
			return callback && callback(false);
		}

		var normalizedAlias = normalizeAlias(data.alias + '_' + (data.openai_model || /*dataSet.get('chatgpt', ['openai_model']) ||*/ 'assistant'));
		for (var i in conversations) {
			if (normalizeAlias(conversations[i].alias) === normalizedAlias) {
				return callback(true, conversations[i]);
			}
		}

		request({
			url: 'conversation/alias/' + normalizedAlias,
		}, function(bOK, response) {
			if (!bOK) {
				return callback && callback(bOK, response);
			}

			var conversation;
			for (var i in conversations) {
				if (conversations[i].id === data.id) {
					conversation = conversations[i];
					break;
				}
			}

			if (!conversation) {
				conversation = new ChatGPTConversation();
				conversations.push(conversation)
			}

			conversation.data(response);

			gui.__exeEvent('ChatGPTConversationCreate', conversation);
			callback && callback(bOK, conversation);
		});
	};

	WMChatGPT.prototype.conversationCreate = function (data, callback) {
		function create() {
			data.openai_model = data.openai_model || /*dataSet.get('chatgpt', ['openai_model']) || */'assistant';

			if (!data.alias) {
				delete data.alias;
			} else {
				data.alias = normalizeAlias(data.alias + '_' + data.openai_model);
			}

			request({
				url: 'conversation',
				method: 'POST',
				data: {
					label: data.label || '',
					openai_model: data.openai_model,
					alias: data.alias,
				}
			}, function (bOK, response) {
				if (!bOK) {
					
					// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATION_CREATE'}});
					return callback && callback(bOK, response);
				}
	
				var conversation = new ChatGPTConversation().data(response);
				conversations.push(conversation);
	
				gui.__exeEvent('ChatGPTConversationCreate', conversation);
				callback && callback(bOK, conversation, true);
			});
		}

		if (!data.alias) {
			return create(data, callback);
		}

		this.getConversationByAlias(data, function (bOK, response) {
			if (!bOK) {
				if (response.status === 404) {
					return create(data, callback);
				}
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATION_GET'}});
				return callback && callback(bOK, response);
			}

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.conversationCreateFake = function (data, callback) {
		data = data || {};
		data.fake = true;

		data.openai_model = data.openai_model || 'assistant';

		data.messages = (data.messages || []).map(function(message) {
			return new ChatGPTConversationMessage().data({
				message: message,
				role: 'assistant',
				status: 'SUCCESS',
				conversation: this
			});
		}, this);

		var conversation = new ChatGPTConversation().data(data);
		conversations.push(conversation);

		gui.__exeEvent('ChatGPTConversationCreate', conversation);
		callback && callback(true, conversation);
	};

	WMChatGPT.prototype.filesList = function (limit, offset, callback) {
		request({
			url: 'files',
			data: {
				limit: limit || 5,
				offset: offset || 0
			}
		}, function (bOK, response) {
			response = response || [];

			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::FILES_LIST'}});
				return callback && callback(bOK, response);
			}

			response = response.map(function(data) {
				var file;
				for (var i in files) {
					if (files[i].alias === data.alias) {
						file = files[i];
						break;
					}
				}
				if (!file) {
					file = new ChatGPTFile();
					files.push(file)
				}

				return file.data(data);
			});

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.createFile = function(data, callback) {
		if (data.alias && files.some(function(file) {
			if (file.alias === data.alias) {
				callback && callback(true, file);
				return true;
			}
		})) {
			return;
		}

		request({
			url: 'files',
			method: 'POST',
			data: {
				link: data.url,
				name: data.name,
				alias: data.alias,
				extension: '.' + data.name.split('.').pop()
			}
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::FILE_CREATE'}});
				return callback && callback(bOK, response);
			}

			var file = ChatGPTFile.instance(response);

			callback && callback(bOK, file);
		});
	};

	WMChatGPT.prototype.images = function(data, callback) {
		if (!data || !data.prompt) {
			// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::DALLE_EMPTY'}});
			return callback && callback(false);
		}
		
		var model = 'dalle2';
		if (this.models.some(function(model) {
			return model.model === 'dalle3';
		})) {
			model = 'dalle3';
		}

		request({
			url: 'images',
			method: 'POST',
			data: Object.assign({
				model: model,
				prompt: '',
				quality: 'hd', // hd | standard
				size: "1024", // 256 | 512 | 1024 | 1792x1024 | 1024x1792
				style: "natural" // natural | vivid
			}, data)
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::DALLE'}});
				return callback && callback(bOK, response);
			}

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.textToSpeech = function(data, callback) {
		if (!data.input) {
			return callback && callback(false);
		}

		data = Object.assign({
			format: 'mp3', // mp3 | opus | aac | flac
			input: '',
			model: 'tts-1', // tts-1 | tts-1-hd | canary-tts
			speed: 0,
			voice: 'alloy' // alloy | echo | fable | onyx | nova | shimmer
		}, data);

		request({
			url: 'speecher',
			method: 'POST',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::SPEECHER'}});
				return callback && callback(bOK, response);
			}

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.transcriptFile = function(data, callback) {
		if (!data.files) {
			return callback && callback(false);
		}

		request({
			url: 'transcriptor',
			method: 'POST',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::TRANSCRIPTOR'}});
				return callback && callback(bOK, response);
			}

			callback && callback(bOK, response);
		});
	};

	WMChatGPT.prototype.tokenizer = function (data, callback) {
		request({
			url: 'tokenizer',
			method: 'POST',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::TOKENIZER'}});
				return callback && callback(bOK, response);
			}

			callback && callback(bOK, response.tokens);
		});
	};

	WMChatGPT.prototype.ping = function(callback) {
		request({
			url: 'ping',
			method: 'GET'
		}, callback);
	};

	WMChatGPT.getFile = function (fileId) {
		return files[fileId];
	};

	function ChatGPTConversation() {
		this.messages = [];
	}

	ChatGPTConversation.prototype.data = function (data) {
		if (data.message_first) {
			if (!this.messages.some(function(message) {
				return message.id === data.message_first.id;
			})) {
				this.messages.push(new ChatGPTConversationMessage(this).data(data.message_first));
			}

			if (data.message_last && data.message_last.id !== data.message_first.id) {
				if (!this.messages.some(function(message) {
					return message.id === data.message_last.id;
				})) {
					this.messages.push(new ChatGPTConversationMessage(this).data(data.message_last));
					delete data.message_last;
				}
			}

			delete data.message_first;
		}

		for (var i in data) {
			this[i] = data[i];
		}

		return this;
	};

	ChatGPTConversation.prototype.detail = function (callback) {
		if (!this.id) {
			return callback(true, this);
		}

		if (this.__gettingDetail) {
			return;
		}
		this.__gettingDetail = true;
		request({
			url: 'conversation/' + this.id
		}, function (bOK, response) {
			this.__gettingDetail = false;
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATION_DETAIL'}});
				return callback && callback(bOK, response);
			}

			this.data(response);

			callback && callback(bOK, this);
		}.bind(this));
	};

	ChatGPTConversation.prototype.process = function (data, callback) {
		if (!this.id || !this.messages.length) {
			return callback(false);
		}

		if (this.processing) {
			processQueue[this.id] = processQueue[this.id] || [];
			processQueue[this.id].push(callback);
			return;
		}
		this.processing = true;

		data = Object.assign(this.instance === 'image' ? {
			quality: 'hd',
			size: '1024',
			style: 'natural'
		} : {}, data || {});

		request({
			url: 'conversation/' + this.id + '/process',
			method: 'POST',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				this.processing = false;
				processProcessQueue(this);
				return callback && callback(bOK, response);
			}

			var message = new ChatGPTConversationMessage(this).data(response);
			this.messages.push(message);
			gui.__exeEvent('ChatGPTConversationMessageCreate', message);

			this.intensity = this.messages.length;
			this.updated_at = message.updated_at;

			gui.__exeEvent('ChatGPTConversationUpdate', this);
			callback && callback(bOK, message);
		}.bind(this));
	};

	ChatGPTConversation.prototype.update = function (data, callback) {
		function updateCallback(response) {
			this.data(response);

			gui.__exeEvent('ChatGPTConversationUpdate', this);
			callback && callback(true, this);
		}

		if (!this.id) {
			return updateCallback.call(this);
		}

		request({
			url: 'conversation/' + this.id,
			method: 'PATCH',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATION_UPDATE'}});
				return callback && callback(bOK, response);
			}

			updateCallback.call(this, response);
		}.bind(this));
	};

	ChatGPTConversation.prototype.cancel = function (data, callback) {
		if (!this.id) {
			return callback(true, this);
		}

		request({
			url: 'conversation/' + this.id + '/cancel',
			method: 'POST',
			data: data
		}, function (bOK) {
			callback && callback(bOK, this);
		}.bind(this));
	};

	ChatGPTConversation.prototype.delete = function (callback) {
		function deleteCallback(response) {
			conversations = conversations.filter(function (conversation) {
				return conversation !== this;
			}, this);

			gui.__exeEvent('ChatGPTConversationDelete', this);
			callback && callback(true, response);
		}

		if (!this.id) {
			return deleteCallback.call(this);
		}

		request({
			url: 'conversation/' + this.id,
			method: 'DELETE',
			data: {}
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::CONVERSATION_DELETE'}});
				return callback && callback(bOK, response);
			}

			deleteCallback.call(this, response);
		}.bind(this));
	};

	ChatGPTConversation.prototype.messagesList = function (callback) {
		if (!this.id) {
			return callback(true, this.messages);
		}

		request({
			url: 'conversation/' + this.id + '/message'
		}, function (bOK, response) {
			response = response || [];

			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MESSAGES_LIST'}});
				return callback && callback(bOK, response);
			}

			this.messages = response.map(function (data) {
				var message;
				for (var i in this.messages) {
					if (this.messages[i].id === data.id) {
						message = this.messages[i];
						break;
					}
				}
				message = (message || new ChatGPTConversationMessage(this)).data(data);

				return message;
			}, this);

			callback && callback(bOK, this.messages);
		}.bind(this));
	};

	ChatGPTConversation.prototype.messageCreate = function (data, callback) {
		if (!this.id) {
			return ChatGPT.conversationCreate({
				label: data.message.split(/.+\.\?!/)[0],
				alias: this.alias,
				openai_model: this.openai_model
			}, async function(bOK, conversation) {
				this.realConversation = conversation;
				await gui.__exeEvent('ChatGPTConversationUpdate', this);
				await gui.__exeEvent('ChatGPTConversationDelete', this);
				conversation.messageCreate(data, callback);
			}.bind(this));
		}

		request({
			url: 'conversation/' + this.id + '/message',
			method: 'POST',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MESSAGE_CREATE'}});
				return callback && callback(bOK, response);
			}

			var message = new ChatGPTConversationMessage(this).data(response);
			this.messages.push(message);

			this.intensity = this.messages.length;
			this.updated_at = response.updated_at;

			gui.__exeEvent('ChatGPTConversationMessageCreate', message);
			gui.__exeEvent('ChatGPTConversationUpdate', this);
			callback && callback(bOK, message);
		}.bind(this));
	};

	function ChatGPTConversationMessage(conversation) {
		this.files = [];
		this.conversation = conversation;
	}

	ChatGPTConversationMessage.statusMap = {
		0: 'PROCESSING',
		1: 'ERROR',
		2: 'SUCCESS',
		3: 'PROCESSING',
		4: 'CANCELED',
		5: 'EMPTY'
	};

	ChatGPTConversationMessage.prototype.data = function (data) {
		for (var i in data) {
			switch(i) {
				case 'status':
					this[i] = ChatGPTConversationMessage.statusMap[data[i]] || data[i];
					break;

				case 'files':
					(data.files || []).forEach(function(file) {
						var fileInstance = ChatGPTFile.instance(file);
						if (!~this.files.indexOf(fileInstance)) {
							this.files.push(fileInstance);
						}
					}, this);
					break;

				case 'next_message_id':
					if (data[i] && !this.conversation.messages.some(function(message) {
						return message.id === data[i];
					})) {
						var message = new ChatGPTConversationMessage(this.conversation).data({
							id: data[i]
						});
						this.conversation.messages.push(message);
						gui.__exeEvent('ChatGPTConversationMessageCreate', message);
					}
					this[i] = data[i];
					break;

				case 'image':
					if (data[i]) {
						var name = data[i].split('/').pop();
						this[i] = location.origin + '/teamchatapi/' + name + '?override_method=http.download&token=' + sPrimaryAccountTeamchatToken + '&url=' + encodeURIComponent(data[i].unentityify());
					}
					break;

				case 'message':
					var match, fileInstance;
					var matches = data.message.matchAll(/\((file-[^(]+)#sandbox:(.*?)\)/g);
					for (match of matches) {
						fileInstance = ChatGPTFile.instance({
							alias: match[1],
							name: match[2].split('/').pop()
						});
						if (!~this.files.indexOf(fileInstance)) {
							this.files.push(fileInstance);
						}
					}

					matches = data.message.matchAll(/!\[[^[]+\]\(([^(]+)\)/g);
					for (match of matches) {
						fileInstance = ChatGPTFile.instance({
							alias: match[1],
							name: 'img'
						});
						if (!~this.files.indexOf(fileInstance)) {
							this.files.push(fileInstance);
						}
					}

				default:
					this[i] = data[i];
			}
		}
		return this;
	};

	ChatGPTConversationMessage.prototype.detail = function (callback) {
		if (this.__gettingDetail) {
			return;
		}
		this.__gettingDetail = true;
		request({
			url: 'conversation/' + this.conversation.id + '/message/' + this.id
		}, async function (bOK, response) {
			this.__gettingDetail = false;
			var wasProcessing = !!this.conversation.processing;
			this.conversation.processing = ~['PROCESSING', 'EMPTY'].indexOf(response.status);
			if (wasProcessing !== this.conversation.processing) {
				gui.__exeEvent('ChatGPTConversationUpdate', this.conversation);
			}
			processProcessQueue(this.conversation);

			if (!bOK) {
				this.status = 'ERROR';
				var data = {};
				try {
					data = await response.json();
					this.error_code = data.code;
				} catch {}

				if (data.code !== 'ENTITY_NOT_FOUND') {
					// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MESSAGE_DETAIL'}});
				}
				gui.__exeEvent('ChatGPTConversationMessageUpdate', this);
				return callback && callback(bOK, response);
			}

			delete this.error_code;
			this.data(response);

			gui.__exeEvent('ChatGPTConversationMessageUpdate', this);

			callback && callback(bOK, this);
		}.bind(this));
	};

	ChatGPTConversationMessage.prototype.update = function (data, callback) {
		request({
			url: 'conversation/' + this.conversation.id + '/message/' + this.id,
			method: 'PATCH',
			data: data
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MESSAGE_UPDATE'}});
				return callback && callback(bOK, response);
			}

			this.data(response);

			var remove = false;
			this.conversation.messages = this.conversation.messages.filter(function(message) {
				if (remove) {
					gui.__exeEvent('ChatGPTConversationMessageDelete', message);
					return false;
				}
				if (message.id === this.id) {
					remove = true;
				}
				return true;
			}, this);
			this.conversation.intensity = this.conversation.messages.length;

			gui.__exeEvent('ChatGPTConversationMessageUpdate', this);
			this.conversation.updated_at = response.updated_at;
			gui.__exeEvent('ChatGPTConversationUpdate', this.conversation);
			callback && callback(bOK, this);
		}.bind(this));
	};

	ChatGPTConversationMessage.prototype.delete = function (callback) {
		request({
			url: 'conversation/' + this.conversation.id + '/message/' + this.id,
			method: 'DELETE',
			data: {}
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::MESSAGE_DELETE'}});
				return callback && callback(bOK, response);
			}

			this.conversation.messages = this.conversation.messages.filter(function (message) {
				return message.id !== this.id;
			}, this);
			gui.__exeEvent('ChatGPTConversationMessageDelete', this);

			this.conversation.intensity = this.conversation.messages.length;
			gui.__exeEvent('ChatGPTConversationUpdate', this.conversation);
			callback && callback(bOK);
		}.bind(this));
	};

	function ChatGPTFile() {

	}

	ChatGPTFile.instance = function(data) {
		if (!files[data.alias]) {
			files[data.alias] = new ChatGPTFile();
		}

		return files[data.alias].data(data);
	};

	ChatGPTFile.prototype.data = function (data) {
		for (var i in data) {
			switch (i) {
				case 'name':
					this[i] = data[i].replace(/^sandbox:.*?_/, '');
				break;

				case 'download_url':
					this[i] = location.origin + '/teamchatapi/http.download?token=' + sPrimaryAccountTeamchatToken + '&url=' + encodeURIComponent(data[i].unentityify());
				break;

				default:
					this[i] = data[i];
			}
		}
		return this;
	};

	ChatGPTFile.prototype.detail = function (callback) {
		if (this.__gettingDetail) {
			return;
		}
		this.__gettingDetail = true;
		request({
			url: 'files/' + this.alias
		}, function (bOK, response) {
			this.__gettingDetail = false;

			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::FILE_DETAIL'}});
				return callback && callback(bOK, response);
			}

			this.data(response);

			gui.__exeEvent('ChatGPTFileUpdate', this);

			callback && callback(bOK, this);
		}.bind(this));
	};

	ChatGPTFile.prototype.delete = function (callback) {
		request({
			url: 'files/' + this.alias,
			method: 'DELETE',
			data: {}
		}, function (bOK, response) {
			if (!bOK) {
				// gui.notifier._value({type: 'chatgpt', args: {header: 'CHATGPT_ERROR::FILE_DELETE'}});
				return callback && callback(bOK, response);
			}

			files = files.filter(function (file) {
				return file.alias !== this.alias;
			}, this);
			gui.__exeEvent('ChatGPTFileDelete', this);

			callback && callback(bOK);
		}.bind(this));
	};

	ChatGPTFile.prototype.download = async function() {
		if (this.alias && this.alias[0] === '*') {
			var data;
			try {
				data = (await WMItems.list({
					aid: sPrimaryAccount,
					fid: '__@@UPLOAD@@__',
					iid: this.alias
				}))[sPrimaryAccount]['__@@UPLOAD@@__'][this.alias];
				var attName;
				for (var i in data.ATTACHMENTS) {
					if (data.ATTACHMENTS[i].values.ATTTYPE === 'attachment') {
						attName = i;
						break;
					}
				}
				downloadItem(await Item.webdavURL([sPrimaryAccount, data.EVNFOLDER, this.alias], attName), true);
			} catch {
				//
			}
			return;
		}

		mkElement('a', {
			href: this.download_url,
			download: this.name
		}).click();
	};

	function request(options, callback) {
		var url = dataSet.get('dashboard', ['middleware']) + '/openai/' + options.url;
		var headers = {
			accept: 'application/json',
			Authorization: 'Bearer ' + icewarpapi.token.access,
		};

		var server_version = (dataSet.get('dashboard', ['config']) || {}).server_version
		if (server_version) {
			headers['iw-server-version'] = server_version;
		}

		var fetch_options = {
			method: options.method || 'GET',
			cache: 'no-cache',
			headers: headers
		};

		if (~['POST', 'PUT', 'PATCH'].indexOf(options.method)) {
			headers['Content-Type'] = 'application/json';
			fetch_options.body = options.data ? JSON.stringify(options.data) : void 0;
		} else if (options.data && Object.keys(options.data).length) {
			url += '?' + buildURL(options.data);
		}

		fetch(url, fetch_options).then(async function(response) {
			if (!response.ok) {
				return Promise.reject(response);
			}

			if (response.status === 401) {
				return gui.frm_main.dashboard.__refreshToken(function() {
					request(options, callback);
				});
			}
	
			var data;
			if (response.status !== 204) {
				data = response;
	
				if (!~['DELETE'].indexOf(options.method)) {
					try {
						data = await response.json();
						if (response.status !== 200) {
							data.status = response.status;
						}
					} catch {
						//
					}
				}
			}

			callback && callback(response.ok, data);
		}).catch(async  function(response) {
			return callback && callback(false, response);
		});
	};

	window.ChatGPT = new WMChatGPT();

	window.ChatGPT.getModel = WMChatGPT.getModel;
	window.ChatGPT.getModelName = WMChatGPT.getModelName;
	window.ChatGPT.getFile = WMChatGPT.getFile;
})();