
/*
	SIP bridge class
	Martin Ekblom 2014
*/

/*
	SIP states:
		online, offline, error

	SIP activities (busy state):
		ready, ringing, talking, calling, meeting

	SIP types:
		conference | phone
*/

var IceSIP = function(config) {
	IceSIP.instantiated = true;

	var me = this;

	// Internal state
	this.activity = null;
	this.state = null;
	this.type = null;

	// Custom parameters for IceWarp
	config.stun_servers = []; // "stun:stun.l.google.com:19302"
	var ws = location.protocol.replace('http','ws') + '//' + document.location.host;
	config.sockets = [new JsSIP.WebSocketInterface(ws)];

	config.register_expires = 120;
	config.session_timers = false;

	// PeerConstraint: {optional: [{DtlsSrtpKeyAgreement: 'true'}]}

	// Create SIP client
	this.sip = new JsSIP.UA(config);

	// Intercept and modify INFO requests because JsSIP does not support RFC 6068
	// this.sip.on('connected',function(e) {
	// 	var socket = e.socket;
	// });

	// Connecting registation events
	this.sip.on('registered',function(e){
		if(me.state!='online') {
			me.state = 'online';
			me.activity = 'ready';
			var host = e && e.data && e.data.response && e.data.response.from && e.data.response.from.uri && e.data.response.from.uri.host ? e.data.response.from.uri.host : '';
			me.onconnect({host: host});
		}
	});
	this.sip.on('unregistered',function(){
		me.state = 'offline';
		me.activity = null;
		me.ondisconnect({reason: 'offline'});
	});
	this.sip.on('registrationFailed',function(){
		me.state = 'failed';
		me.activity = null;
		me.ondisconnect({reason: 'failed'});
	});

	// Monitor incoming calls
	this.sip.on('newRTCSession', function(e){

		if(e.session) {

			// Monitor user media access
	/*		e.session.on('mediaRequested',function(e){
				me.onmedia({constraints: e});
			});
			e.session.on('mediaGranted',function(e){
				me.onmedia({granted: true});
			});
	*/		e.session.on('getusermediafailed',function(e){
				me.onmedia({granted: false, reason: e && e.name});
			});

			// Handle incoming calls
			if(e.originator=='remote') {

				// Do not allow more than one call
				for(var i in me.sip._sessions)
					if(e.session._id!=i) {
						e.session.terminate({status_code: 486});
						return;
					}

				me.activity = 'Ringing';

				// Initiate call and keep reference
				var call = new IceSIP.Call(me,e.session);
				var session = e.session;

				// Local references to media players
				var audio = false;
				var video = false;

				// Handling failed or ended call
				var ended = function(e) {
					if (audio)
						audio.pause();

					call.onhangup({reason: e.cause || 'Ended', remote: e.originator=='remote'});
				};
				session.on('ended',ended);
				session.on('failed',ended);

				// Handling started call
				session.on('confirmed',(function(call){
					return function() {
						me.activity = 'Talking';

						call.onanswer({
							name: session._remote_identity._display_name,
							user: session._remote_identity._uri._user + '@' + session._remote_identity._uri._host
						});
					};}(call))
				);

				// Determine if it's a audio, video or presentation call
				var sdp = e.request.body;
				var presentation = sdp && sdp.indexOf("m=video")!=-1 && sdp.indexOf("a=sendonly")!=-1;
				video = presentation || sdp && sdp.indexOf("m=video")!=-1 && sdp.indexOf("a=sendrecv")!=-1;

				// Only remote stream is ready in this stage
				me.oncall({
					name: session._remote_identity._display_name,
					user: session._remote_identity._uri._user + '@' + session._remote_identity._uri._host,
					call: call,
					video: video,
					show: presentation
				});
			}
		}
	});

	// Monitor incoming messages - both conference and common
	this.sip.on('newMessage',function(e){
		// Remote incoming message
		if(e && e.originator=='remote' && e.message) {
			var message = new IceSIP.Message(me, e.message._request.body);
			message.from = e.message._remote_identity._uri._user + '@' + e.message._remote_identity._uri._host;
			// Propagate to conference if it has conference header
			if(e.message._request.getHeader('X-IceWarp-Conference')) {
				var id = e.message._request.getHeader('X-IceWarp-Conference').match(/id=([0-9]+)/);
				if(id)
					message.conference_id = id[1];
				var data = e.message._request.getHeader('X-IceWarp-Message').split(',');
				var tmp = data.pop();
				do {
					if(tmp.indexOf("is_private=")===0)
						message.group = tmp=="is_private=0" ? true : false;
					else if(tmp.indexOf("from_id=")===0)
						message.from_id = tmp.substr(8);
				} while((tmp=data.pop()));
			}
			me.onmessage(message);
		}
	});

};

/* Static support method */

IceSIP.supported = function() {
	if(window.RTCPeerConnection && navigator.mediaDevices && navigator.mediaDevices.getUserMedia) // Cross browser check for native WebRTC support
		return true;
	else if(document.body && 'WebSocket' in window && 'ActiveXObject' in window) { // IE browser with WebSockets, check for plugin
		var supported = false;
		try {
			if(new ActiveXObject('Tem.TemWebRTCPlugin'))
				supported = true;
		} catch {
			//
		}
		return supported;
	} else if(window.webkitAudioContext) { // WebKit engine without native support (Safari), check for plugin
		if(navigator.plugins && navigator.plugins.namedItem && navigator.plugins.namedItem('TemWebRTCPlugin'))
			return true;
		else
			return false;
	} else return false;
};

/* Check connected devices */

IceSIP.devices = function(callback) {
	// Latest cross browser standard device detection
	if(navigator.mediaDevices && navigator.mediaDevices.enumerateDevices) {
		navigator.mediaDevices.enumerateDevices()
		.then(function(devices) {
			var media = {microphone: false, camera: false};
			devices.forEach(function(device) {
				switch(device.kind) {
					case 'audioinput':
						media.microphone = true;
						break;
					case 'videoinput':
						media.camera = true;
						break;
				}
			});
			callback(media);
		})
		.catch(function() {
			callback({});
		});
	} else callback({});
};

IceSIP.instantiated = false;

/* SIP connetion handling */

// Start SIP, initial start of all SIP functionality
IceSIP.prototype.start = function() {
	this.sip.start();
};

// Stop SIP, fully stop all SIP functionality
IceSIP.prototype.stop = function() {
	this.sip.stop();
};

// Temoporarily unregister (be offline)
IceSIP.prototype.unregister = function() {
	this.sip.unregister();
};

// Re-register after going offline
IceSIP.prototype.register = function() {
	this.sip.register();
};

// Force end calls
IceSIP.prototype.terminate = function() {
	for(var i in this.sip.sessions)
		this.sip.sessions[i].terminate();
};

// Check if SIP is online
IceSIP.prototype.online = function() {
	return this.sip.isConnected() && this.sip.isRegistered();
};

// Check if SIP is ready to make or accept calls
IceSIP.prototype.busy = function() {
	return !this.sip.isConnected() || !this.sip.isRegistered() || this.activity!='ready';
};

IceSIP.prototype.onconnect = function(e) {
	// Abstract method
	 console.warn('No handler assigned: Connected to '+e.host);
	// You are connected to host
};

IceSIP.prototype.ondisconnect = function() {
	// Abstract method
	 console.warn('No handler assigned: Disconnected');
	// Connection was lost
};

IceSIP.prototype.onmedia = function() {
	// Abstract method
	 console.warn('No handler assigned: Media');
	// Asking for or granted access to media
};

/* General helper functions */

IceSIP.attachVideoStream = function(stream) {
	var video;
	if(window.AdapterJS && AdapterJS.WebRTCPlugin && AdapterJS.WebRTCPlugin.plugin) {
		var elmId = 'WebRTCPluginVideo' + Math.random().toString(36).slice(2);
		var elm = document.createElement('div');
		// AdapterJS.WebRTCPlugin.pluginInfo.type
		elm.innerHTML =
		'<object id="' + elmId + '" type="application/x-temwebrtcplugin">' +
			'<param name="pluginId" value="' + elmId + '" /> ' +
			'<param name="pageId" value="' +  AdapterJS.WebRTCPlugin.pageId + '" /> ' +
			'<param name="windowless" value="true" /> ' +
			'<param name="streamId" value="' + stream.id + '" /> ' +
			'<param name="tag" value="video" /> ' +
		'</object>';
		video = elm.firstChild;
	} else {
		video = document.createElement('video');
		video.setAttribute('autoplay', true);

		video.srcObject = stream;

	}
	return video;
};

/* SIP Phone, Call handling */

IceSIP.Call = function(gear,session) {
	this.sip = gear.sip;
	this.state = session ? 'Ringing' : 'Calling';
	this.sip.activity = this.state;
	this.incoming = !!session;
	this.session = session;
};

IceSIP.Call.prototype.getLocalStreams = function(){
	var stream = new MediaStream(),
		senders = this.session._connection.getSenders();

	senders.forEach(function(sender) {
		stream.addTrack(sender.track);
		// if (sender.track.kind === 'video')
		// 	camera = true;
	});

	return stream;
};
IceSIP.Call.prototype.getRemoteStreams = function(){
	var stream = new MediaStream();
	this.session._connection.getReceivers().forEach(function(receiver) {
		stream.addTrack(receiver.track);
	});

	return stream;
};


// Make an outgoing call
IceSIP.Call.prototype.make = function(address,bVideo,bScreen) {
	var me = this;

	var mc = {audio: true, video: bVideo || false};
	if(bScreen)
		mc.screen = true;

	var oc = {
		offerToReceiveAudio: true,
		offerToReceiveVideo: bVideo || false
	};

	var audio = false;

	this.session = this.sip.call('sip:'+address,{eventHandlers: {
		progress: function() {
			me.state = 'Calling';
		},
		connecting: function() {
			me.oncalling({
				call: me.session,
				name: me.session._remote_identity._display_name,
				user: me.session._remote_identity._uri._user + '@' + me.session._remote_identity._uri._host
			});

		},
		failed: function(e) {
			me.state = 'Failed';
			me.onhangup({reason: e.cause ? e.cause : 'Unknown', remote: e.originator=='remote'});
		},
		confirmed: function() {
			if(me.session && me.session._remote_identity) {
				me.state = 'Talking';

				me.onanswer({
					name: me.session._remote_identity._display_name,
					user: me.session._remote_identity._uri._user + '@' + me.session._remote_identity._uri._host
				});
			}
		},
		ended: function(e) {
			me.state = 'Ended';
			if(audio)
				audio.pause();
			me.onhangup({reason: 'Ended', remote: e.originator=='remote'});
		},
		newDTMF: function(e) {
			if(e.originator=='remote')
				me.ondial({tone: e.dtmf.tone});
		},
		newInfo: function(e) {
			if(e.info) {
				if(e.info.contentType=="application/media_control+xml") {
					var xml = e.info.body;
					xml = new DOMParser().parseFromString(xml, 'text/xml');
					var command = xml.getElementsByTagName('to_encoder')[0];
					command = command.firstElementChild.nodeName;
					if(command=="picture_fast_update" && pc) {
						var videotrack = pc.getLocalStreams()[0].getVideoTracks()[0];
						videotrack.enabled = false;
						setTimeout(function(){
							videotrack.enabled = true;
						},250);
					}
				}
			}
		},
		refer: function(e) {
			e.accept();
		},
		replaces: function(e) {
			e.accept();
		}
	},
	mediaConstraints: mc,
	rtcOfferConstraints: oc
	});

	var pc = this.session._connection;
	pc.addEventListener('addstream',function(e){

		// Handle local streams
		var localstream = me.getLocalStreams(),
			camera = !mc.screen && localstream.getVideoTracks().length;

		var remote = false,
			local = false;

		var audiostream = false, videostream = false;
		if (e.stream.getVideoTracks().length) {
			videostream = e.stream;
		}
		else
		if (e.stream.getAudioTracks().length) {
			audiostream = e.stream;
			audiostream.onaddtrack = function(e) {
				// Video track added to audio after renegotiation
				if(e.track.kind=='video') {
					var localstream = me.getLocalStreams(),
						camera = localstream.getVideoTracks().length>0;

					var video = {
						remote: IceSIP.attachVideoStream(e.target),
						local: camera ? IceSIP.attachVideoStream(localstream) : false
					};
					me.onstreaming({
						camera: camera ? true : false,
						video: video,
						audio: false
					});
				}
			};
		}

		if (audiostream) {
			me.session.audio = audio = new Audio();
			me.session.audio.srcObject = e.stream;
			me.session.audio.play();
		}
		else
		if (videostream) {
			remote = IceSIP.attachVideoStream(e.stream);
			if(camera) {
				local = IceSIP.attachVideoStream(localstream);
				local.muted = true;
			}
		}

		var video = remote || local ? {remote: remote, local: camera ? local : false} : false;
		me.onstreaming({
			camera: camera ? true : false,
			video: video,
			audio: audiostream
		});
	},false);

};

// Answer to an incoming call
IceSIP.Call.prototype.answer = function(bShow) {
	var me = this;

	if(this.sip.isConnected() && this.sip.isRegistered() && this.session) {
		// Always send audio
		var mc = {audio: true /*, video: true */};

		// Only send video back if it's not a presentation
		if (bShow)
			mc.video = false;

		this.session.answer({
			mediaConstraints: mc
		});

		this.session._connection.addEventListener('addstream', function() {
			var remote, local;

			var ext_stream = me.getRemoteStreams(),
				video = ext_stream.getVideoTracks().length>0;

			var loc_stream = me.getLocalStreams(),
				camera = loc_stream.getVideoTracks().length>0;

			// Adding video stream
			if (video) {
				remote = IceSIP.attachVideoStream(ext_stream);
				if (camera){
					local = IceSIP.attachVideoStream(loc_stream);
					local.muted = true;
				}
			}
			else
			// Adding pure audio stream
			if (ext_stream.getAudioTracks().length) {
				me.session.audio = new Audio();
				me.session.audio.srcObject = ext_stream;
				me.session.audio.play();
			}

			me.onstreaming({
				name: me.session._remote_identity._display_name,
				user: me.session._remote_identity._uri._user + '@' + me.session._remote_identity._uri._host,
				camera: camera,
				video: video ? {remote: remote, local: local} : false,
				audio: video ? false : ext_stream
			});

		},false);

		return true;
	} else
		return false;
};

// Decline an incoming call
IceSIP.Call.prototype.decline = function() {
	if(this.sip.isConnected() && this.sip.isRegistered() && this.session)
		this.session.terminate();
	else
		return false;
};

// Dial numbers via keypad
IceSIP.Call.prototype.dial = function(n,bInband) {
	if(this.sip.isConnected() && this.sip.isRegistered() && this.session && (/^[*0-9#]+$/).test(n)) {
		var stream = this.session._localMediaStream;
		var audio = stream.getAudioTracks() || null;
		var peer = this.session._connection.pc;

		if(bInband && audio && audio.length && peer && peer.createDTMFSender) {
			var dtmf = peer.createDTMFSender(audio[0]);
			dtmf.ontonechange = function(e){
				if(e.tone) {
				//	console.log('Sending DTMF over RTP: ',e.tone);
				}
			};
			if(dtmf.canInsertDTMF)
				dtmf.insertDTMF(n);
		} else
			this.session.sendDTMF(n);
	}
};

// Mute user (silence outgoing stream)
IceSIP.Call.prototype.mute = function(mute) {
	if(this.session) {
		if(mute)
			this.session.mute();
		else
			this.session.unmute();

		return true;
	} else
		return false;
};

// Put call on hold
IceSIP.Call.prototype.hold = function(hold) {
	if(this.session) {
		if(hold)
			return this.session.hold();
		else
			return this.session.unhold();
	} else
		return false;
};

// Transfer call to other recipient
IceSIP.Call.prototype.transfer = function(sip) {
	if(this.session) {
		this.session.refer('sip:'+sip);
	};
};

// Hang up a call, regardless on incoming or outgoing
IceSIP.Call.prototype.hangup = function(code) {
	if(this.sip.isConnected() && this.sip.isRegistered() && this.session) {
		this.state = 'Ended';
		this.sip.activity = 'Ready';

		if (this.session._state != 8){
			if (this.session.audio)
				this.session.audio.pause();

			if(code)
				this.session.terminate({status_code: code});
			else
				this.session.terminate();
		}
	}
};

IceSIP.Call.prototype.addvideo = function() {
	if(this.sip.isConnected() && this.sip.isRegistered() && this.session) {
		var session = this.session;
		var connection = this.session._connection;
		var promise = navigator.mediaDevices.getUserMedia({audio: false, video: true});

		promise.then(function(stream){

			if ('addTrack' in connection){
				stream.getTracks().forEach(function(track) {
					if (track.kind == 'video')
						connection.addTrack(track, stream);
				});
			}
			// else
			// connection.addStream(stream);

			session.renegotiate({useUpdate:false},function(){
				 console.log("Renegotiate", arguments);
			});


		}).catch(function(){});
	}
};

// Add video to an ongoing call
IceSIP.Call.prototype.camera = function() {
	var me = this;

	var constraints = {
		audio: true, video: true
	};

	// Add new mediastream
	var addStream = function(stream) {
		me.session.rtcMediaHandler.addStream(stream);
	};

	// Attempt to get media stream from screen
	try {
		navigator.webkitGetUserMedia(constraints,
			addStream,
			function(){
				 console.log("No stream!");
			}
		);
	} catch(e){
		 console.log('Failed',e);
	}

};


IceSIP.Call.prototype.ondial = function(e) {
	// Abstract method
	 console.warn('No handler assigned: '+e.number+' was dialed');
	// You are connected to host
};

IceSIP.Call.prototype.onanswer = function(e) {
	// Abstract method
	 console.warn('No handler assigned: '+e.user+' has picked up');
	// Outgoing call was answered
};

IceSIP.Call.prototype.ondecline = function(e) {
	// Abstract method
	 console.warn('No handler assigned: '+e.user+' has declined call');
	// Outgoing call was declined
};

IceSIP.Call.prototype.oncalling = function(e) {
	// Abstract method
	 console.warn('No handler assigned: you are calling '+e.user);
	// Outgoing call was terminated
};

IceSIP.Call.prototype.onhangup = function(e) {
	// Abstract method
	 console.warn('No handler assigned: '+e.user+' has hanged up');
	// Outgoing call was terminated
};

/* SIP Messages */

IceSIP.Message = function(connection,content) {
	this.sip = connection.sip;
	this.content = content;
};

// Send message
IceSIP.Message.prototype.send = function(to) {
	var me = this;
	this.to = to;
	this.sip.sendMessage('sip:'+to,this.content,{
		eventHandlers: {
			succeeded: function() {
				me.onsent({failed: false});
			},
			failed: function() {
				me.onsent({failed: true});
			}
		},
		extraHeaders: []
	});
};