var _me;

window.browserEvent = function(event) {
	if (('on' + event) in window || event.replace(/end|start|update$/, '') in document.body.style) {
		return event;
	}
	if (('onwebkit' + event) in window) {
		return 'webkit' + this.capitalize(event[0]).replace(/(end|start|update)$/, this.capitalize);
	}
	return false;
};

window.clipboardNotifications = {};
window.toClipboard = function(v, notification_type = 'link') {
	if (v instanceof Object && navigator.clipboard && navigator.clipboard.write && 'ClipboardItem' in window) {
		var clipboardItemData = {};
		for (var i in v) {
			if (v[i] instanceof Blob) {
				clipboardItemData[i] = v[i];
				
			} else if (Is.String(v[i]) && v[i].length) {
				if (i.indexOf('image/') === 0) {
					clipboardItemData[i] = async function() {
						return await fetch(v[i]).then(function(response) {
							return response.blob();
						});
					};
				} else {
					clipboardItemData[i] = new Blob([v[i]], {type: i});
				}
			}
		}

		console.log('toClipboard', clipboardItemData);
		return navigator.clipboard.write([new ClipboardItem(clipboardItemData)]).then(function() {
			if (clipboardNotifications[notification_type]) {
				clipboardNotifications[notification_type].close();
			}

			clipboardNotifications[notification_type] = gui.notifier._value({
				type: 'success',
				args: {
					text_plain: getLang('notification::clipboard_' + notification_type )
				}
			});
		});
	}

	for (var i in v) {
		v = v[i];
		break;
	}

	if (Is.String(v[i]) && v[i].length)	{
		var inp = mkElement('textarea', {value:v, style:'position: absolute; opacity:0; z-index: -1000;'});
		document.body.appendChild(inp);
		inp.select();

		if (document.queryCommandSupported('copy') && document.queryCommandEnabled('copy') && document.execCommand('copy')) {
			gui.notifier._value({type: 'success', args: { text_plain: getLang('notification::' + (notification_type || 'clipboard_link')) }});

			setTimeout(function() {
				if (inp.parentNode)
					inp.parentNode.removeChild(inp);
			}, 220);

			return true;
		}
	}

	gui.notifier._value({type: 'alert', args: {header: '', text: 'ERROR::CLIPBOARD'}});
	return false;
};

/**
 * @brief: cross browser compatible helper to register for events
 **/
window.AttachEvent = function (obj, eventname, handler) {
	eventname = eventname.substr(2);
	if (eventname === "wheel"){
		obj.addEventListener(eventname, function(e){
			var evn = {
				originalEvent:e,
				type:e.type,
				target: e.target,
				deltaX:0,
				deltaY:0
			};

			if (navigator.userAgent.indexOf('WebKit') != -1) {
				evn.deltaX = e.deltaX / 40;
				evn.deltaY = e.deltaY / 40;
			} else {
				evn.deltaX = e.deltaX;
				evn.deltaY = e.deltaY;
			}
			handler(evn);

		}, { passive: true });
	} else {
		obj.addEventListener(eventname, handler, false);
	}
};

window.unique_id = function(){
	return (Math.random()*1000000000000000000)+''+(new Date).getTime();
};

/**
 * @brief	sends given download.php url into iframe
 * @date	24.7.2014
 */
window.downloadItem = function () {
	/**
	 * callback of the iframe onload
	 * - when server responds with 'text/html' content type and with an error in a body of the response
	 * - onload handler is not called when correct headers are sent in response (Content-Type: application/octet-stream)
	 * - content type cannot be 'application/json', because IE always downloads it (fixed in server)
	 * - content type cannot be checked in javascript - always try to JSON parse documentElement and get error from response
	 * - see WA-394 and/or WC-6923 and/or WC-6965 for details why it was implemented
	 * - note: SmartDiscover must be set-up correctly - opened URL in a browser must be same as the URL in SmartDiscover WebClient and teamchatapi
	 */
	function downloadItemOnloadCallback() {
		var	frm = this.contentDocument,
			body, error = '';

		try {
			body = frm.documentElement;
			error = JSON.parse(body.innerText || body.xtContent).error;
		}
		catch {
			// no action
		}

		if (error) {
			switch(error) {
				case 'no_permission':
					return gui.notifier._value({type: 'alert', args: {text: 'ERROR::E_ACCESS', args: [error]}});
				default:
					gui.notifier._value({type: 'alert', args: {text: 'ALERTS::GENERAL_ERROR', args: [error]}});
			}
		}
	};

	return function(path, full, bForceNewWindow){

		if (!Is.String(path)) {
			return;
		}

		var win,
			src = full ? path : sPrimaryAccountClient + 'server/download.php?' + path;

		// do not use IFRAME when https is currently used but download path is not secure (http)
		if (bForceNewWindow || (full && location.protocol == 'https:' && path.toLowerCase().indexOf('http:') == 0)) {
			win = window.open(src, 'downloadFile');
			if (win && win.document) {
				win.document.onload = function() {
					window.close();
				};
				return;
			}
		}

		var id = 'ifrm_download_' + unique_id(),
			frm = mkElement('iframe', {id: id, src: src});

		frm.style.position = 'absolute';
		frm.style.zIndex = '-10';

		// this allows to get an error response when download is not available
		frm.onload = downloadItemOnloadCallback;

		// this executes the download action
		document.body.appendChild(frm);

		// remove created iframe after 2 minutes
		setTimeout(function(){
			try{
				if (frm && frm.parentNode)
					frm.parentNode.removeChild(frm);
			}
			catch(e){
				console.log("downloadItem",src, e);
				window.Sentry && Sentry.captureException(e);
			}
		}, 120000);
		return true;
	};
}();

window.getRemoteFileContent = function(src, callback, responseType) {
	var xhr = new XMLHttpRequest();
	xhr.open('GET', src);
	if (responseType) {
		xhr.responseType = responseType;
	}
	xhr.onreadystatechange = function () {
		if (xhr.readyState === 4) {
			if (xhr.status === 200 || xhr.status == 0) {
				if (responseType === 'blob') {
					callback(xhr.response);
				} else {
					callback(xhr.responseText);
				}
			}
			else{
				callback(false);
			}
		}
	};
	xhr.send();
};

/**
 * @brief: Z-index whatever class :)
 **/

function cMaxZIndex(){};
cMaxZIndex.prototype.zindex = [500];
cMaxZIndex.prototype.get = function(b){
	var z = this.zindex[this.zindex.length-1]+1;
	if (!b) this.zindex.push(z);
	return z;
};
cMaxZIndex.prototype.remove = function(z){
	var pos = inArray(this.zindex,z);
	if (pos>-1) this.zindex.splice(pos,1);
};
window.maxZIndex = new cMaxZIndex();

/**
 * @brief   One-row createElement function
 * @author  DRZ 28.03.2005
 */
window.mkElement = function(tElm,eatt,doc,children) {
	var elm = (doc || document).createElement(tElm);
	if (typeof eatt === 'object') {
		for (var i in eatt) {
			if (eatt[i] === void 0) {
				continue;
			}
			try {
				switch(i) {

					case 'contenteditable':
					case 'for': elm.setAttribute(i,eatt[i]);
						break;

					case 'text':
						//elm.appendChild((doc || document).createTextNode(eatt[i]));
						elm.textContent = eatt[i];
						break;

					case 'style':
						if (Is.Object(eatt[i])){
							for (var j in eatt[i])
								elm.style[j] = eatt[i][j];
							break;
						}

					case 'href':
						if (!eatt[i])
							break;

					default:
						if (i in elm)
							elm[i] = eatt[i];
						else
							elm.setAttribute(i, eatt[i]);
				}
			} catch {
				//
			}
		}
	}
	(Array.isArray(children) ? children : [children]).forEach(function (child) {
		child && elm.appendChild(child);
	});

	return elm;
};

/**
 * Returns count of actually added styles
 * If style already exists it doesnt count to $out but it is styll placed to the end
 **/

window.addcss = function(elm){
    if (!elm) return;

	var arg = [];
	for (var a = 1;a<arguments.length;a++)
		if (Is.String(arguments[a]))
			arg = arg.concat(arguments[a].trim().split(' '));

	for (a = 0;a<arg.length;a++)
		if (arg[a])
			elm.classList.add(arg[a]);
};

window.removecss = function(elm){
    if (!elm) return;

	if (arguments.length<2)
		elm.className = '';
	else {
		for (var a = 1;a<arguments.length;a++)
			if (Is.String(arguments[a]))
				elm.classList.remove(arguments[a].indexOf(' ')>-1?arguments[a].trim():arguments[a]);
	}
};

/**
 * toggle css (using hascss and removecss or addcss)
 *
 * @param {object} elm
 * @param {string} sClass
 * @return {void}
 */
window.togglecss = function(elm, sClass, toggle) {
	if (toggle === void 0) {
		toggle = !hascss(elm, sClass);
	}
	if (toggle) {
		addcss(elm, sClass);
	} else {
		removecss(elm, sClass);
	}
};

window.hascss = function(elm,sClass){
	return elm.classList.contains(sClass);
};

/**
 * @brief   returns array containing element's size and position
 * @author  DRZ 10.03.2005
 *
 * @NOTE: ELEMENTS WITH owerflow: auto MUST HAVE position: relative
 **/

window.getSize = function(elm, bDontBubble){
	var r = {x:0,y:0,h:elm.offsetHeight,w:elm.offsetWidth},
		doc = elm.ownerDocument;

	if (elm.getBoundingClientRect){

		var box = elm.getBoundingClientRect();

			// Add the document scroll offsets
			r.x = box.left + Math.max(doc.documentElement.scrollLeft, doc.body.scrollLeft);
			r.y = box.top + Math.max(doc.documentElement.scrollTop, doc.body.scrollTop);
	}
	else if (elm.nodeType === 3) {
		var range = doc.createRange();
		range.selectNode(elm);
		var rect = range.getBoundingClientRect();
		
		r.h = rect.height;
		r.w = rect.width;
		// Add the document scroll offsets
		r.x = rect.left + Math.max(doc.documentElement.scrollLeft, doc.body.scrollLeft);
		r.y = rect.top + Math.max(doc.documentElement.scrollTop, doc.body.scrollTop);
	}
	//Others  - doesnt work scrollTop in Safari
	else{
		r.x = elm.offsetLeft;
		r.y = elm.offsetTop;

		while((elm = elm.offsetParent)){
			if (!elm || elm.tagName == 'BODY') break;
			r.x += elm.offsetLeft - elm.scrollLeft;
			r.y += elm.offsetTop - elm.scrollTop;
		}
	}

	// if within iframe
	if (!bDontBubble && doc.defaultView !== window && doc.defaultView.frameElement) {
		var size = getSize(doc.defaultView.frameElement);
		r.x += size.x;
		r.y += size.y;
	}

	return r;
};


////////////////////////////////////////////////////
//           TYPE DETECTION FUNCTIONS
////////////////////////////////////////////////////
window.Is = (function(){
	var reg = {
		email:		/^(?:[a-z0-9!#$%&'*+/=?^_`{|}~\-:]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~\-:]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])$/gi, //https://bit.ly/33cv2vn
		url:		/^http(s?):\/\/[a-z0-9]*/gi,
		filename:	/^(?!\.)(?!com[0-9]$)(?!con$)(?!lpt[0-9]$)(?!nul$)(?!prn$)[^<>:/\\|?*""]*$/gm
	};

	return {
		Boolean: function(a)  {
			return typeof a == 'boolean';
		},
		Array: function(a) {
			return Is.Object(a) && a.constructor == Array;
		},
		Empty: function(o) {
			if (Is.Object(o))
				if (Is.Array(o)){
					if (o.length)
						return false;
				}
				else
				for (var i in o)
					if (Is.Defined(o[i]))
						return false;

			return true;
		},
		Element: function(a){
			return a instanceof HTMLElement || a instanceof (((a.ownerDocument || {}) || {}).defaultView || window).HTMLElement;
		},
		Function: function(a) {
			return typeof a == 'function';
		},
		Number: function(a) {
			return typeof a == 'number' && isFinite(a);
		},
		Object: function(a) {
			return (a && typeof a == 'object') || Is.Function(a);
		},
		String: function(a) {
			return typeof a == 'string';
		},
		Email: function(a) {
			reg.email.lastIndex = 0;
			return Is.String(a) ? !!a.match(reg.email) || (((reg.email.lastIndex = 0) || true) && !!punycode.toASCII(a).match(reg.email)) : false;
		},
		URL: function(a) {
			if (!Is.String(a)) return false;
			return !!a.match(reg.url);
		},
		Filename: function(a){
			return (!Is.String(a) || !a.trim().length) ? false : !!a.match(reg.filename);
		},
		Defined: function(x){
			return x !== void 0;
		},
		Child: function (elm,eParent,eStop){

			if (Is.String(eParent)){
				eParent = eParent.toUpperCase();
				try{
					do {
						if (elm.tagName == eParent)
							return elm;

						if (eStop && eStop == elm)
							return false;
					}
					while((elm = elm.parentNode));
				}
				catch(r){ console.log(this._name||false,r)}
			}
			else
			if (eParent)
				//Modern
				if ('contains' in document.body)
					try{
						return eParent.parentNode && eParent.contains(elm);
					}
					catch(r){ console.log(this._name||false,r)}
				//Old
				else
					try{
						do {
							if (elm == eParent)
								return true;

							if (eStop && eStop == elm)
								return false;
						}
						while((elm = elm.parentNode));
					}
					catch(r){ console.log(this._name||false,r)}

			return false;
		},
		InstanceOf: function(instance, of) {
			if (instance instanceof window[of]) {
				return true;
			}
			for(var i = 0; window.frames[i]; i++) {
				if (!Object.keys(window.frames[i]).length) {
					continue;
				}
				try {
					if (window.frames[i][of] && instance instanceof window.frames[i][of]) {
						return true;
					}
				} catch {
					//
				}
			}
		}
	};
})();


////////////////////////////////////////////////////
//               STRING EXTENSIONS
////////////////////////////////////////////////////

/**
 * @brief	Replace & < > " by &amp; &lt; &gt; &quot;;
 * Example:
 *    - "&<>\"'".entityify() == "&amp;&lt;&gt;&quot;"
 */
String.prototype.entityify = function() {
	return this.replace(/&/g,"&amp;").replace(/"/g,"&quot;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
};
String.prototype.unentityify = function() {
	return this.replace(/&gt;/g,">").replace(/&lt;/g,"<").replace(/&quot;/g,"\"").replace(/&amp;/g,"&");
};

String.prototype.urlEncode = function() {
	return encodeURIComponent(this).replace(/!/g, '%21').replace(/'/g, '%27').replace(/\(/g, '%28').replace(/\)/g, '%29').replace(/\*/g, '%2A').replace(/%20/g, '+');
};
String.prototype.urlDecode = function() {
	return decodeURIComponent(this.replace(/%(?![\da-f]{2})/gi, function() { return '%25' }).replace(/\+/g, '%20'));
};

String.prototype.highlight_links = function (on){
	var str = this;

	if (this.indexOf('@')>0){ // && this.indexOf('/')<0
		on = on || '';
		var emailPattern = /(([a-z0-9'!#$%&+\-/=?^_`{|}~*]\.?)+@[a-z0-9]+([.\-_]?[a-z0-9])*\.[a-z]{2,})/gi;
		str = str.replace(emailPattern, "<a href=\"mailto:$1\""+(on?' '+on:'')+">$1</a>");
	}

	var urlPattern = /(?:=?(['"])?)([A-Za-z]{3,5}:\/\/[^\s<>]+?)([.,]\s|[\s<>]|$)/g;
	str = str.replace(urlPattern, function(match, quote, uri, suffix) {
		if (quote) {
			return uri;
		}
		var url = '';
		try {
			url = new URL(uri);
		} catch {
			try {
				url = new URL(uri.substring(0, uri.length - 1));
			} catch {
				//
			}
		}
		if (location.protocol === 'https:' && url && url.protocol === 'http:') {
			if (url.pathname.match(/\.\w{2,}$/)) { // redirect file to https proxy
				url = location.origin + '/teamchatapi/http.download?token=' + sPrimaryAccountTeamchatToken + '&url=' + encodeURIComponent(uri.unentityify());
			}
		}
		var opener = '';
		if (ShortURL.matches(url.href) || url.href.match(/\/collaboration\/\?ticket=(.*?)&url=(.*?)(?:&|$)/g)) {
			opener = ' rel="opener"';
		}
		return '<a href="' + url.toString() + '" target="_blank"' + opener + '>' + uri + '</a>' + suffix
	});

	return str;
};

String.prototype.highlight_links_array = function (bShort){
	var str = [],
		sPrefix = '~░',
		sSuffix = '░~',
		arr = [],
		elm = bShort?mkElement('A'):null;

	var stripPattern = /@\[\S+\]/gi,
		emailPattern = /([a-z0-9'!#$%&+\-/=?^_`{|}~*]\.?)+@[a-z0-9]+([.\-_]?[a-z0-9])*\.[a-z]{2,}/gi,
		urlPattern = /([A-Za-z-]{3,}:\/\/[^\s<>]+?)([.,]\s|[\s<>]|$)/g,
		splitStr = '```';

	if (~this.indexOf(splitStr))
		str = this.split(splitStr);
	else
		str = [this];

	for (var i = 0, j = str.length; i<j; i += 2){

		//replace emails
		if (!str[i].match(urlPattern) && str[i].indexOf('@')>0){

			//skip mention @[email]
			str[i] = str[i].replace(stripPattern, function(){
				return sPrefix + (arr.push(arguments[0]) - 1) + sSuffix;
			});

			str[i] = str[i].replace(emailPattern, function(){
				return sPrefix + (arr.push('<a href="mailto:' + arguments[0] + '">' + arguments[0] + '</a>') - 1) + sSuffix;
			});
		}

		str[i] = str[i].replace(urlPattern, function(x, s){
			var uri = s;
			if (bShort && s.length>48){
				elm.href = s;
				s = elm.protocol + '//' + elm.hostname;

				if (elm.pathname.length>24)
					s += elm.pathname.substr(0,24) + '...';
				else{
					s += elm.pathname;
					if (elm.search)
						s+='?...';
				}
			}
			var url = '';
			try {
				url = new URL(uri);
			} catch {
				try {
					url = new URL(uri.substring(0, uri.length - 1));
				} catch {
					//
				}
			}
			if (location.protocol === 'https:' && url && url.protocol === 'http:') {
				if (url.pathname.match(/\.\w{2,}$/)) { // redirect file to https proxy
					uri = location.origin + '/teamchatapi/http.download?token=' + sPrimaryAccountTeamchatToken + '&url=' + encodeURIComponent(uri.unentityify());
				}
			}
			var opener = '';
			if (ShortURL.matches(url.href) || url.href.match(/\/collaboration\/\?ticket=(.*?)&url=(.*?)(?:&|$)/g)) {
				opener = ' rel="opener"';
			}
			return sPrefix + (arr.push('<a href="'+ uri + '" target="_blank"' + opener + '>'+ s.entityify() +'</a>') - 1) + sSuffix + arguments[2];
		});

	}

	return {string: str.join(splitStr), array: arr, replace: sPrefix +'(\\d+)'+ sSuffix};
};

// Charactar data escape complying with xml 1.0 rfc, http://www.w3.org/TR/REC-xml/#syntax
String.prototype.escapeXML = function(bAttr) {
	var s = this.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
	if(bAttr) s = s.replace(/'/g,"&apos;").replace(/"/g,"&quot;");
	return s;
};

String.prototype.unescapeXML = function() {
	return this.replace(/&gt;/gi,">").replace(/&lt;/gi,"<").replace(/&quot;/gi,'"').replace(/&apos;/gi,'"').replace(/&amp;/gi,'&');
};

/**
 * @brief	Replace &amp; &lt; &gt; &quot; &#039; by & < > " '
 *  htmlspecialchars
 * Example:
 *		"&amp;&lt;&gt;&quot;&#039;".unentityify() == "&<>\"'"
 */

String.prototype.escapeHTML = function() {
	var div = document.createElement('div');
		div.appendChild(document.createTextNode(this));
	return div.innerHTML;
};

String.prototype.unescapeHTML = function() {
	if (this.indexOf('&')<0) return this.toString();

	var div = document.createElement('div');
		div.innerHTML = this.toString();
		div = div.childNodes[0];

	var out = div.textContent;
	while (true) {
		if((div = div.nextSibling)){
			if (div.nodeValue)
				out += div.nodeValue;
		}
		else
			break;
	}

	div = null;

	return out;
};

/**
 * @brief   Remove HTML tags from string.
 *			inline	= ''
 *			block   = ' '
 * Example:
 * 			"<b>Lorem ipsum <i>dolor</i> sit amet</b>".removeTags() == "Lorem ipsum dolor sit amet"
 */
String.prototype.removeTags = function(str){
	return this.replace(/<[!/]?([-a-zA-Z0-9]+)[^>^<]*>/gm,str || '').replace(/&nbsp;/g,' ');
};

/**
 * @brief   Quote characters that are not digits nor alphas.
 * Example:
 *    - "@#$%".quoteMeta() == "\@\#\$\%"
 */
String.prototype.quoteMeta = function(){
	return this.replace(/([!#$%^@.&*()\-_=+:;"'\\/?<>~[]{}`])/g , "\\$1" );
};

////////////////////////////////////////////////////obj_timetable
//                 DATE TOOLS
////////////////////////////////////////////////////

/**
 * Translates seconds to minutes and seconds.
 * @param[in]	iTime	[number]	Number of seconds.
 *
 * @return	[string]	Of format 'minutes:seconds'.
 * Example:
 * 		parseJulianTime(3601) == '1:01';
 */
window.parseJulianTime = function(iTime){
	var H  = (iTime-iTime%3600)/3600,
		M = Math.ceil(iTime%3600/60);
		M = M<10?'0'+M:M;

	if (GWOthers.getItem('LAYOUT_SETTINGS','time_format')>0)
    	return (H % 12 || 12) + ':' + M + (H<12?" AM":" PM");
	else
		return H + ':' + M;
};

/**
 * @brief   Search value in string, array or object.
 * @param[in]  sElm  [string|array|object]
 * @author  DRZ 10.03.2005
 * @return  If the parameter is:
 *    - string: then returns position of 'sElm' in string.
 *    - array:  index in the array.
 *    - object: name of the property which has the same value as 'sElm'.
 *
 *    If the value isn't found, return -1.
 *
 * Example:
 *    - inArray("lorem ipsum", "ipsum") == 6
 *    - inArray("lorem ipsum", "dolor") == -1
 *    - inArray(["lorem", "ipsum"], "ipsum") == 1
 *    - inArray({"prop1": "lorem", "prop2": "ipsum"}, "ipsum") == "prop2"
 */
window.inArray = function (aArray,sElm) {
	if (Is.Array(aArray))
		return aArray.indexOf(sElm);
	else
	for(var i in aArray)
		if(aArray[i]==sElm) return i;

	return -1;
};

/**
 * @brief   Reverse the order of items in the array of object.
 * @param[in]  oObj  [array|object]
 * @return  Array or object with items in reversed order.
 *
 * Example:
 *    - reverse(["lorem", "ipsum"] == ["ipsum", "lorem"]
 *    - reverse({"lorem": 1, "ipsum": 2}) == {"ipsum": 2, "lorem": 1}
 */
window.reverse = function(oObj){
	// reverse array
	if (oObj.constructor == Array)
		return oObj.reverse();

	// reverse object
	var key=[],oOut = {};
	for (var i in oObj)
		key.push(i);

	key.reverse();
	for (i in key)
		oOut[key[i]] = oObj[key[i]];

	return oOut;
};

/**
 * @brief   Concat unlimited number of associative arrays.
 * @param[in]  '...'  [object] Unlimited number of associative arrays.
 * @author  DRZ 22.05.2005
 * @warning Doesn't work on normal arrays!
 * @return  Union of all arrays.
 *
 * Example:
 *    // {"color": "red", "shape": "triangle", "size": "small", "position": "bottom"}
 *    arrConcat({"color": "red", "shape": "triangle"}, {"size": "small"}, {"position": "bottom"});
 *
 *     // BAD USAGE, returns ["blue", "green"]
 *    arrConcat(["red", "green"], ["blue"]);
 */
window.arrConcat = function(){
	var main = {};
	for (var a = 0;a<arguments.length;a++)
		for (var i in arguments[a]) main[i] = arguments[a][i];

	return main;
};

/**
 * @brief   Makes multidimensional array.
 * @param[in]  keys  [array]  List of keys.
 * @param[out]  arr   [array]  Optional, create subarray into alredy exising array.
 * @author  DRZ 01.05.2005
 * @return  Multidimensional array indexed by keys from 'keys' argument.
 *
 * Example:
 *    var aArray = {"first": []};
 *
 *    // Prints only structure of created subarray.
 *    // [first]
 *    //    [second]
 *
 *
 *    // But the whole array looks like this:
 *    // [first]
 *    //    [second]
 *    //       [third]
 *
 *
 * 22.7.2008 12:01:58 -  zmena z [] na {}
 */
window.mkArrayPath = function(keys,arr,val){

	if(typeof arr != 'object') arr = {};

	var out = arr;

	for (var i in keys){

		if (arguments.length>2 && keys.length-1 == i)
			arr[keys[i]] = val;
		else
		if (typeof arr[keys[i]] != 'object')
			arr[keys[i]]={};

		arr = arr[keys[i]];
	}

	return out;
};

window.arrayPath = function(aData,aDPath){
	for(var i in aDPath){
        aData = aData[aDPath[i]];
		if (typeof aData == 'undefined')
		    return;
	}

	return aData;
};

/**
 * @brief   Return lowest free key in array. (Array must be classical starting from zero!).
 * @param[in]  arr   [array]
 * @return  The first index which isn't defined. If array doesn't contain any gaps,
 * return index rigth after the last item.
 * @date : 10.5.2006 9:32:22
 *
 * Example:
 *    var aArray = ["one", "two", "three"];
 *    delete aArray[1]; // make 'hole'
 *
 *    // prints 1
 *    alert(getFreeKey(aArray));
 */
window.getFreeKey = function(arr){
	for(var i = 0;;i++)
		if (typeof arr[i] == 'undefined')
		    return i;
};

window.arrUnique = function(arr){
	return arr.filter(function(v,i,a){
		return a.indexOf(v) === i;
	});
};

/**
 * @brief   Return number of values in array or object.
 * @param[in]  arr   [array|object]
 * @date : 10.5.2006 9:31:05
 *
 * Example:
 *    - count({"one": "red", "two": "green"}) == 2
 *    - count(["one", "two", "three"]) == 3
 */
window.count = function(arr){
	if (Array.isArray(arr))
		return arr.length;
	else
	if (arr != null && typeof arr == 'object')
		return Object.keys(arr).length;

	return -1;  // invalid argument
};

/**
 * @brief   Check if two one dimensional arrays are identical (keys must be the same!).
 * Items are compared one by one so be careful when comparing multidimensional arrays.
 * This function works on one-dimensional array as you expected (items are atomic values).
 * When comparing subarrays, the are considered the same only if they are the same
 * references (they point to the same address in the memory).
 * @param[in]  arr1  [array]
 * @param[in]  arr2  [array]
 * @return  true/false
 *
 * @warning arrayCompare(["one", "two"], ["two", "one"]) == false because of different order!
 * (["one", "two"] == {"0": "one", "1": "two"} != {"0": "two", "1": "one"}).
 * @warning Only first dimension is compared so arrayCompare([[1]], [[1]]) == false.
 *
 * Example:
 *    - arrayCompare(["one", "two"], ["one", "two"]) == true
 *    - arrayCompare(["one", "two"], ["two", "one"]) == false
 *    - arrayCompare(["one", "two"], {"1": "two", "0": "one"}) == true
 *    - arrayCompare([[1]], [[1]]) == false
 *
 *    // Special case with multidimensional array
 *    var aArray = [1];
 *    arrayCompare([aArray], [aArray]); // == true
 */
window.arrayCompare = function(arr1, arr2) {
	var length = 0;

	for (var key in arr1) {
		if (arr1[key] != arr2[key]) {
			return false;
		}
		length++;
	}
	if (count(arr2) == length)
		return true;
	else
		return false;
};

////////////////////////////////////////////////////
//                   URL  TOOLS
////////////////////////////////////////////////////

/**
 * @brief   Function build URL GET string from JS array.
 * Don't worry about special characters like & or %, they are URL encoded.
 * @param[in]  varList  [object] Associative array of name->value.
 * @return GET string composed from pairs (name,value) from 'varList'.
 * @author DRZ 06.12.2012
 *
 * Example:
 *    - buildURL({"size": "normal", "special": "%^&"}) == "size=normal&special=%25%5E%26"
 */
window.buildURL = function(varList) {
	var url = [];
	for (var name in varList)
		url.push(encodeURIComponent(name) + (varList[name] !== void 0 ? '=' + encodeURIComponent(varList[name].toString()) : ''));

	return url.join('&');
};

/**
 * @brief   Function parses URL GET variables to JS array.
 * @param[in]  url   [string] URL encoded string.
 * @return  Associative array of name->value obtained from url.
 * @author  DRZ 06.12.2012
 *
 * DOESNT WORK WITH UTF16 (%uXXXX)!
 *
 * Example:
 *    - parseURL("size=normal&special=%25%5E%26") == {"size": "normal", "special": "%^&"}
 */
window.parseURL = function(url){
	var p,argList,newArg,output = {};

	if (!Is.Defined(url))
		url = self.location.search;

	if (typeof url === 'object') {
		return url;
	}

	if (typeof url == 'string'){
		//strip # part
		if ((p = url.indexOf('#')) > -1)
			url = url.substr(0,p);

		if ((p = url.indexOf('?'))>-1 && (p<url.indexOf('=')))
			url = url.substr(p+1);

		argList = url.split('&');
		for (var i = 0; i<argList.length; i++)
			if (argList[i]){
				newArg = argList[i].split('=');
				try{
					output[decodeURIComponent(newArg[0])] = newArg[1] ? decodeURIComponent(newArg[1].toString()) : '';
				}
				catch {
					return {};
				}
			}
	}

	return output;
};

/**
 *	parse filesize from bytes to kB/MB
 *	1.2.2010 16:16:39
 */
window.parseFileSize = function(i){
	function ceilFloat(num,n){
		n = Math.pow(10, parseInt(n,10)) || 1;
		return Math.ceil(parseFloat(num) * n)/n;
	};

	if ((i = parseInt(i,10)) && Is.Number(i)){
		i = ceilFloat(i/1024,1);
		if (i>=1024*1024)
			return ceilFloat(i/1024/1024,1).toLocaleString(document.documentElement.lang) + ' ' + getLang('UNITS::GB');
		else if (i>=1024)
			return ceilFloat(i/1024,1).toLocaleString(document.documentElement.lang) + ' ' + getLang('UNITS::MB');
		else
			return i.toLocaleString(document.documentElement.lang) + ' ' + getLang('UNITS::KB');
	}
	else {
		i = 0;
		return i.toLocaleString(document.documentElement.lang) + ' ' + getLang('UNITS::KB');
	}
};

////////////////////////////////////////////////////
//                 Miscellaneous
////////////////////////////////////////////////////

// TODO unless executeCallbackFunction reorders the parameters
// this hacker function wouldn't be nedeed
window.pushParameterToCallback = function(aResponse, arg) {
	if (Is.Function(aResponse[0])) {
		if (Is.Array(aResponse[1]))
			aResponse[1].push(arg);
		else
			aResponse[1] = [arg];
	}
	else
	if (Is.Object(aResponse[0])) {
		if (Is.Array(aResponse[2]))
			aResponse[2].push(arg);
		else
			aResponse[2] = [arg];
	}
	else
		throw 'pushParameterToCallback - Invalid argument';
};

/**
 * @brief	Implements calling registered function.
 * This function is used everywhere the callback is needed, mostly in reaction on some event
 * in the form. Etc. when the user submit some form and we need extra actions connected to that event.
 * @param[in]	aResponse	[array]
 * 		[0] - [Object|Function]		Depends on whether we call standalone function or member function.
 * 		[1] - [String|Array|Function]
 * 									If the first parameter is Object, the second must be string (name
 * 									of the function, otherwise array containing arguments to the function.
 * 		[3]	- [Array]				Optional. It can be preset in case the first parameter is object and
 * 									it represents argument to the function.
 *
 * @param[in]				[all]	Optional. The callback function is then called with these parameters and
 * 									they have 'higher priority' to aResponse[3] in that way they are first.
 *
 * @TODO	Don't reorder parameters!
 *
 * Example 1:
 * 		var func = function(arg) {
 * 			alert(arg1);
 * 			alert(arg2);
 * 		}
 *
 *		executeCallbackFunction([func, ['2']], '1');	// Will print '1' and '2'
 *
 * Example 2:
 * 		var date = new Date();
 *
 *		executeCallbackFunction([date, 'setHours', [2, 3, 0], xml|array (default)|text], 1);	// Sets 1 hours, 2 minutes, 3 seconds and 0 milliseconds.
 */
window.getCallbackFunction = function(aResponse,bAlways){
	if (aResponse){
		if (Is.Function(aResponse[0]))
			return aResponse[0];
		else
		if (Is.Function(aResponse[1]))
			return aResponse[1];
		else
		if (bAlways || !aResponse[0]._destructed)
			try{
				return aResponse[0][aResponse[1]];
			}
			catch{
				//
			}
	}
	return false;
};

window.executeCallbackFunction = async function(aResponse) {
	if (Is.Array(aResponse) && ((Is.Object(aResponse[0]) && (Is.String(aResponse[1]) || Is.Function(aResponse[1]))) || Is.Function(aResponse[0])))
		try
		{
			//Method or function?
			var nIndex;
			if (Is.Function(aResponse[0]))
				nIndex = 1;
			else
				nIndex = 2;

			//Prepare arguments
			var args = [];
			for (var i = 1; i < arguments.length; i++)
				args.push(arguments[i]);

			if (Is.Array(aResponse[nIndex]))
				args = args.concat(aResponse[nIndex]);

			// [this,'method',[argument after],argument before, argument before...]
			for (i = nIndex+1; i < aResponse.length; i++)
				args.unshift(aResponse[i]);

			var bOut;
			// function, [args]
			if (nIndex == 1) //Is.Function(aResponse[0])
				bOut = await aResponse[0].apply(null,args);
			else
			// object, method, [args]
			if (Is.Function(aResponse[1]))
				bOut = await aResponse[1].apply(aResponse[0], args);
			// object, "method", [args]
			else{
				bOut = await aResponse[0][aResponse[1]].apply(aResponse[0],args);

				// if (aResponse[0]._destructed == true)
				// 	return false;
			}

			return bOut; //true;
		}
		catch(e){

			var err = '';
			if (Is.String(aResponse[0]))
				err = "Error while executing "+aResponse[0]+"()";
			else
				err = "Error while executing "+(aResponse[0] && aResponse[0]._pathName?aResponse[0]._pathName:'oObject')+"."+aResponse[1]+"()";

			console.error('browser_ext:executeCallbackFunction', err, e);
			window.Sentry && Sentry.captureException(e);
		}
	else
		return false;
};

window.createNameFromLocation = function(aLCTval){
    if (Is.Object(aLCTval)){
		var a = [];
		if (aLCTval.ITMFIRSTNAME)
		    a.push(aLCTval.ITMFIRSTNAME);
		if (aLCTval.ITMMIDDLENAME)
		    a.push(aLCTval.ITMMIDDLENAME);
		if (aLCTval.ITMSURNAME)
		    a.push(aLCTval.ITMSURNAME);
		if (aLCTval.ITMSUFFIX)
		    a.push(aLCTval.ITMSUFFIX);

		return (aLCTval.ITMTITLE?aLCTval.ITMTITLE+' ':'') + a.join(' ');
	}

	return '';
};
window.parseNameToLocation = function(sName,aLCTval){
	sName = (sName || '').trim();

	if (Is.Object(aLCTval)){
		aLCTval.ITMCLASSIFYAS = sName;
		aLCTval.ITMTITLE = '';
		aLCTval.ITMFIRSTNAME = '';
		aLCTval.ITMMIDDLENAME = '';
		aLCTval.ITMSURNAME = '';
		aLCTval.ITMSUFFIX = '';
	}
	else
		aLCTval = {ITMCLASSIFYAS:sName};

	if (!sName.length) return aLCTval;

	var	tmp,p,aName = sName.split(' ');

	//clear blank spaces
	for (var i = aName.length-1;i>-1;i--)
	    if (!(aName[i] = aName[i].trim()))
	    	aName.splice(i,1);

	if (aName.length){

		//Prepare from Lang
		var aLang = getLang('NAME_PREFIX'),
			aPref = {},aSuff = {};
		for (i in aLang)
            if (aLang[i])
				aPref[aLang[i].toUpperCase()] = true;

		aLang = getLang('NAME_SUFFIX');
		for (i in aLang)
            if (aLang[i])
				aSuff[aLang[i].toUpperCase()] = true;

		//Prefix
		while (Is.String(aName[0]))
			if (aPref[aName[0].toUpperCase()])
	            aLCTval.ITMTITLE = (aLCTval.ITMTITLE || '') + aName.shift();
			else
			if ((p = aName[0].lastIndexOf('.'))>-1){
				if (p == aName[0].length-1)
	                aLCTval.ITMTITLE = (aLCTval.ITMTITLE || '') + aName.shift();
				else{
	               tmp = aName[0].split('.');
	               aName[0] = tmp.pop();
	               aLCTval.ITMTITLE = (aLCTval.ITMTITLE || '') + tmp.join('.') + '.';
				}
			}
			else
			    break;

		//FirstName
		aLCTval.ITMFIRSTNAME = aName.shift();

        //Suffix 1
		while(true){
			if ((tmp = aName[aName.length-1]) && (tmp.toUpperCase() === tmp || (p = tmp.indexOf('.'))>-1 || aSuff[tmp.toUpperCase()])){
				if (p>-1){
					tmp = tmp.split('.');
					if (tmp[0] && tmp[1]){
						aLCTval.ITMSUFFIX = tmp[1];
						aName[aName.length-1] = tmp[0];
					}
					else
						aLCTval.ITMSUFFIX = aName.pop() + (aLCTval.ITMSUFFIX?' '+aLCTval.ITMSUFFIX:'');
				}
				else
					aLCTval.ITMSUFFIX = aName.pop() + (aLCTval.ITMSUFFIX?' '+aLCTval.ITMSUFFIX:'');
			}
			else
			    break;
		}

		//MiddleName
		if (aName.length>1 && aName[0].indexOf('.')<0)
			aLCTval.ITMMIDDLENAME = aName.shift();

		//SurName
		if ((tmp = aName.shift())){
			aLCTval.ITMSURNAME = tmp;

			//Suffix 2
			if ((tmp = aName.join(' ')))
				aLCTval.ITMSUFFIX = tmp + (aLCTval.ITMSUFFIX?' '+aLCTval.ITMSUFFIX:'');
		}
		else
		if (aLCTval.ITMMIDDLENAME){
            aLCTval.ITMSURNAME = aLCTval.ITMMIDDLENAME;
			aLCTval.ITMMIDDLENAME = '';
		}
		else
		if (aLCTval.ITMFIRSTNAME){
			aLCTval.ITMSURNAME = aLCTval.ITMFIRSTNAME;
			aLCTval.ITMFIRSTNAME = '';
		}
	}

	return aLCTval;
};

window.getPrimaryAccountFromAddress = function(bPrintable) {
	var aAccInfo = dataSet.get('accounts',[sPrimaryAccount]);
	if (aAccInfo['FULLNAME'])
		return MailAddress.createEmail(aAccInfo['FULLNAME'],sPrimaryAccount,bPrintable);
	else
		return sPrimaryAccount;
};

window.getSubobjects = function(oObject,aObjects) {
    aObjects = aObjects || {};
	for (var i in oObject)
		if (i.charAt(0) != '_' && i.substr(0,2) != 'X_' && i.substr(0,2) != 'x_' && oObject[i].constructor && oObject[i].constructor === Gui)
			if (oObject[i]._type == 'obj_tabs' || oObject[i]._type == 'obj_tab')
				getSubobjects(oObject[i], aObjects);
			else
				aObjects[i] = oObject[i];

	return aObjects;
};

window.getAuxiliarySubobjects = function(oObject,aObjects) {
	aObjects = aObjects || {};
	for (var i in oObject)
		if (i.charAt(0) != '_')
			if (i.substr(0,2) == 'X_' || i.substr(0,2) == 'x_')
				aObjects[i] = oObject[i];
			else
			if (oObject[i]._type == 'obj_tabs' || oObject[i]._type == 'obj_tab')
				getAuxiliarySubobjects(oObject[i], aObjects);

	return aObjects;
};

window.loadDataIntoForm = function(oObject, aValues) {
	var aObjects = getSubobjects(oObject);
	for (var i in aObjects)
		if (Object.prototype.hasOwnProperty.call(aObjects, i) && Is.Defined(aValues[i]) && aObjects[i]._value)
			aObjects[i]._value(aValues[i]);
};

window.loadDataIntoFormOnAccess = function(oObject, aValues, bIgnoreDomain){
	var aObjects = getSubobjects(oObject,aObjects);
	if (aValues['USERACCESS'] || aValues['DOMAINADMINACCESS'])
		var aAuxiliaryObjects = getAuxiliarySubobjects(oObject);

	for (var i in aObjects) {
		if (aValues['ACCESS'][i]) {
			switch(aValues['ACCESS'][i]){
				case 'full':
					aObjects[i]._value(aValues['VALUES'][i]);
					break;
				case 'view':
					aObjects[i]._value(aValues['VALUES'][i]);
					aObjects[i]._disabled(true);
					break;

				case 'none':
					aObjects[i]._main.parentNode.style.display = "none";
			}
		}
		if (aValues['DOMAINADMINACCESS'] && !bIgnoreDomain) {
			if (aAuxiliaryObjects['x_'+i+'_set']) {
				aAuxiliaryObjects['x_'+i+'_set'].domadmin._value((aValues['DOMAINADMINACCESS'][i] == 'view') ? true : false);

				if (aValues['ACCESS'][i] == 'view')
					aAuxiliaryObjects['x_'+i+'_set'].domadmin._disabled(true);
			}
		}
		if (aValues['USERACCESS']) {
			if (aAuxiliaryObjects['x_'+i+'_set']) {
				aAuxiliaryObjects['x_'+i+'_set'].user._value((aValues['USERACCESS'][i] == 'view') ? true : false);

				if (aValues['ACCESS'][i] == 'view')
					aAuxiliaryObjects['x_'+i+'_set'].user._disabled(true);
			}
		}
	}
};

window.storeDataFromFormWithAccess = function(oObject, aValues, aAccess) {
	var aObjects = getSubobjects(oObject);
	var aAuxiliaryObjects = getAuxiliarySubobjects(oObject);

	var value;
	for(var i in aObjects) {
		if (Is.Defined((value = aObjects[i]._value())))
			aValues[i] = value;
		else
			aValues[i] = '';

		if (aAuxiliaryObjects['x_'+i+'_set'] && aAuxiliaryObjects['x_'+i+'_set'].domadmin) {
			if (!aAccess['DOMAINADMINACCESS']) aAccess['DOMAINADMINACCESS'] = {};
			aAccess['DOMAINADMINACCESS'][i] = aAuxiliaryObjects['x_'+i+'_set'].domadmin._value() ? 'view' : 'full';
		}

		if (aAuxiliaryObjects['x_'+i+'_set']) {
			if (!aAccess['USERACCESS']) aAccess['USERACCESS'] = {};
			aAccess['USERACCESS'][i] = aAuxiliaryObjects['x_'+i+'_set'].user._value() ? 'view' : 'full';
		}
	}
};

window.storeDataFromForm = function(oObject, aValues) {
	var aObjects = getSubobjects(oObject),
		value;
	for(var i in aObjects){
		if (Is.Function(aObjects[i]._value)){
			if (Is.Defined((value = aObjects[i]._value()))) {
				aValues[i] = value;
			} else
				aValues[i] = '';
		}
	}
};

window.shiftObject = function(oObject) {
	if (Is.Object(oObject)) {
		for (var i in oObject) {
			var result = oObject[i];
			delete oObject[i];
			return result;
		}
		return null;
	}
	else
		return null;
};

window.isFormEmpty = function(aValues) {
	for (var i in aValues) {
		if (Is.Object(aValues[i])) {
			if (!isFormEmpty(aValues[i])) return false;
		}
		else
		if (aValues[i] != '') return false;
	}
	return true;
};


////////////////////////////////////////////////////
//                 COLOR TOOLS
////////////////////////////////////////////////////

/**
 * @brief: color conversions etc...
 * @note : adopted from http://www.easyrgb.com/math.html
 * @date : 19.9.2006 10:38:52
 **/

function cColors(){
	this.hexchars = "0123456789ABCDEF";
};

//Used for Tags
cColors.prototype.fast_contrast = function (hexcolor, options) {
	options = Object.assign({
		dark: '#333333',
		light: '#ffffff',
		threshold: 50
	}, options || {});
	var rgb = this.hex2rgb(hexcolor);
	var lightness = this.lightness(rgb[0], rgb[1], rgb[2]);
	return lightness >= options.threshold ? options.dark : options.light;
};

cColors.prototype.lightness = function(Rint,Gint,Bint) {

    var Rlin = (Rint / 255) ** 2.218;
    var Glin = (Gint / 255) ** 2.218;
    var Blin = (Bint / 255) ** 2.218;

    var Ylum = Rlin * 0.2126 + Glin * 0.7156 + Blin * 0.0722;

    return Ylum ** 0.43 * 100;
}

/**
 * Converts an HSL color value to RGB. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
 * Assumes h, s, and l are contained in the set [0, 1] and
 * returns r, g, and b in the set [0, 255].
 *
 * @param   Number  h       The hue
 * @param   Number  s       The saturation
 * @param   Number  l       The lightness
 * @return  Array           The RGB representation
 */
cColors.prototype.hsl2rgb = function(h,s,l){
    var r, g, b;

    if(s == 0){
        r = g = b = l; // achromatic
    }else{
        var hue2rgb = function hue2rgb(p, q, t){
            if(t < 0) t += 1;
            if(t > 1) t -= 1;
            if(t < 1/6) return p + (q - p) * 6 * t;
            if(t < 1/2) return q;
            if(t < 2/3) return p + (q - p) * (2/3 - t) * 6;
            return p;
        };

        var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        var p = 2 * l - q;
        r = hue2rgb(p, q, h + 1/3);
        g = hue2rgb(p, q, h);
        b = hue2rgb(p, q, h - 1/3);
    }

    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
};

/**
 * Converts an RGB color value to HSL. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
 * Assumes r, g, and b are contained in the set [0, 255] and
 * returns h, s, and l in the set [0, 1].
 *
 * @param   Number  r       The red color value
 * @param   Number  g       The green color value
 * @param   Number  b       The blue color value
 * @return  Array           The HSL representation
 **/
cColors.prototype.rgb2hsl = function(r, g, b){
    r /= 255, g /= 255, b /= 255;
    var max = Math.max(r, g, b), min = Math.min(r, g, b);
    var h, s, l = (max + min) / 2;

    if(max == min){
        h = s = 0; // achromatic
    }else{
        var d = max - min;
        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
        switch(max){
            case r: h = (g - b) / d + (g < b ? 6 : 0); break;
            case g: h = (b - r) / d + 2; break;
            case b: h = (r - g) / d + 4; break;
        }
        h /= 6;
    }

    return [h, s, l];
};

cColors.prototype.hex2rgb = function(str) {
	str = str.replace('#','');
	return	[
			(this.toDec(str.substr(0, 1)) * 16) + this.toDec(str.substr(1, 1)),
			(this.toDec(str.substr(2, 1)) * 16) + this.toDec(str.substr(3, 1)),
			(this.toDec(str.substr(4, 1)) * 16) + this.toDec(str.substr(5, 1))
			];
};

cColors.prototype.toDec = function(hexchar) {
	return this.hexchars.indexOf(hexchar.toUpperCase());
};

cColors.prototype.rgb2hex = function(r,g,b) {
	return '#' + this.toHex(r) + this.toHex(g) + this.toHex(b);
};

cColors.prototype.toHex = function(n) {
	n = n || 0;
	n = parseInt(n, 10);
	if (isNaN(n)) n = 0;
	n = Math.round(Math.min(Math.max(0, n), 255));

	return this.hexchars.charAt((n - n % 16) / 16) + this.hexchars.charAt(n % 16);
};
window.colors = new cColors();

window.getCurrentEventTime = function() {
	var now = new IcewarpDate();
	var julian = now.format(IcewarpDate.JULIAN);
	var time = now.format(IcewarpDate.JULIAN_TIME);
	time += 30 - time%30;

	return {
		'EVNSTARTDATE': julian,
		'EVNSTARTTIME': time,
		'EVNENDDATE': julian,
		'EVNENDTIME': time+30
	};
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                       Class MailAddress
////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * @class	MailAddress
 * @date    26.4.2012 11:29:10
 */
window.MailAddress = {
	//Create Email address
	createEmail:function(name,email,bNoQuotes){
		var out = '';
		if(email.indexOf('[') === 0 && email.indexOf(']') === email.length - 1) {
			name = '';
		}
		if (name){
			if (bNoQuotes)
				out = name;
			else{
				name = name.replace(/(["\\])/g,'\\$1').trim();
				out = /[\s,;@<]/.test(name)?'"'+name+'"':name;

				// out = (
				// 	name.indexOf(' ')>-1 ||
				// 	name.indexOf('\\')>-1 ||
				// 	name.indexOf(',')>-1 ||
				// 	name.indexOf(';')>-1 ||
				// 	name.indexOf('@')>-1 ||
				// 	name.indexOf('<')>-1?'"'+name+'"':name);
			}
		}

		if (email){
			if (bNoQuotes){
				if (out)
					out += ' ('+ email.toLowerCase() +')';
				else
					out = email.toLowerCase();
			}
			else{
				if (out) out += ' ';
				out += '<'+ email.toLowerCase() +'>';
			}
		}

		return out;
	},
	appendEmail:function(sIn, email){

		if (email){

			if (!sIn)
				return email;

			var aMail = MailAddress.splitEmailsAndNames(email),
				aTmp = MailAddress.splitEmailsAndNames(sIn);

			if ((aMail = aMail[0]) && aMail.email){
				for(var i = aTmp.length-1;i>-1;i--)
					if (aTmp[i].email == aMail.email)
						return sIn;

				aTmp.push(aMail);

				var aOut;
				for(aOut = [],i = 0; i<aTmp.length;i++){
					if (!aTmp[i].name && aTmp[i].email.indexOf('[') === 0 && aTmp[i].email.match(/^\[.+\]$/g))
						aOut.push(aTmp[i].email);
					else
						aOut.push(MailAddress.createEmail(aTmp[i].name,aTmp[i].email));
				}

				return aOut.join(', ');
			}
		}

		return sIn;
	},
	//Split Emails into Array
	splitEmails:function(sIn){
		var aOut = [],
			sChar,
			sTemp = '',
			f = {block:false, block2:false, slash:false};

		if (Is.String(sIn)){
			for(var i = 0;i<sIn.length;i++){
				sChar = sIn.charAt(i);

				switch(sChar){
				case ',':
				case ';':
					if (!f.block && !f.block2 && !f.slash){
						if ((sTemp = sTemp.trim()).length){
							aOut.push(sTemp);
							sTemp = '';
						}

						continue;
					}

					break;

				case '"':
					if (!f.slash)
						if (sTemp == '' || f.block)	//Edit 21.9. ab <a"b@ab.com>, ab <email2>';
							f.block = !f.block;

					break;

				case '[':
					if (!f.block && !f.slash)
						f.block2 = true;

					break;

				case ']':
					if (f.block2 && !f.slash)
						f.block2 = false;

					break;

				case '\\':
					if (!f.slash){
						f.slash = true;
						sTemp += sChar;
						continue;
					}

				//ltrim
				case ' ':
					if (sTemp == '')
						continue;
				}

				sTemp += sChar;
				f.slash = false;
			}

			if ((sTemp = sTemp.trim()).length)
				aOut.push(sTemp);
		}

		return aOut;
	},
	//Split Emails and Names into Array
	splitEmailsAndNames:function(sIn){
		var aOut = [],
			aMails = MailAddress.splitEmails(sIn),
			aUniq = {};

		for (var i = 0;i<aMails.length;i++){
			//do parsing
			if (aMails[i].charAt(aMails[i].length-1) == '>' || aMails[i].charAt(aMails[i].length-1) == ')'){

				var sTemp = '',
					sEmail,
					sChar,
					f = {block:false, slash:false};

				for(var j = 0, k = aMails[i].length;j<k;j++){
					sChar = aMails[i].charAt(j);

					switch(sChar){
					case '<':
					case '(':
						if (!f.block && !f.slash && Math.max(aMails[i].lastIndexOf('<'), aMails[i].lastIndexOf('(')) === j) {
							sTemp = sTemp.trim();
							sEmail = aMails[i].substring(j+1,k-1).trim();

							if (sEmail && !aUniq[sEmail]){
								aUniq[sEmail] = true;
								aOut.push({
									name:sTemp,
									email:sEmail
								});
							}
							if(!sEmail ){
								aOut.push({
									name:sTemp,
									email:sEmail
								});
							}

							//end automat
							j = k;
							continue;
						}

						break;

					case '"':
						if (!f.slash){
							if (sTemp == '' || f.block){ //Edit 21.9. " only on the begining
								f.block = !f.block;
								continue;
							}
						}
						break;

					case '\\':
						if (!f.slash){
							f.slash = true;
							continue;
						}
					}

					sTemp += sChar;
					f.slash = false;
				}

			}
			//its email
			else
			if (!aUniq[aMails[i]]){
				aUniq[aMails[i]] = true;
				aOut.push({name:'',email:aMails[i]});
			}
		}

		for (var k in aOut) {
			if (!~(aOut[k].email || '').indexOf('@') && ~(aOut[k].name || '').indexOf('@')) {
				// if email does not contain @, but name contains @, switch them
				var name = aOut[k].name;
				aOut[k].name = aOut[k].email;
				aOut[k].email = name;
			}
		}

		return aOut;
	},
	//Use instead of name_list
	splitNames:function(sIn){
		var aMails = MailAddress.splitEmailsAndNames(sIn),
			aOut = [];

		for(var i = 0; i<aMails.length;i++)
			aOut.push(aMails[i].name || aMails[i].email);

		return aOut.join(', ');
	},
	//Distribution Lists
	findDistribList:function(aType)
	{
		var aEmails,sEmail,sDistribList,aDistribList,sAccId,sFolId,aName,sName;
		var aResult = {};
		var aDistrib = {};

		//Procházíme jednotlivé typy
		for(var sType in aType)
		{
			aResult[sType] = '';

			if (!aType[sType])
				continue;

			aEmails = MailAddress.splitEmails(aType[sType]);

			//Procházíme jednotlivé emaily
			for(var n in aEmails)
			{
				sEmail = aEmails[n];

				//Nasli jsme distribuční seznam?
				if (sEmail.charAt(0) == '[' && sEmail.charAt(sEmail.length-1) == ']')
				{
					sDistribList = sEmail.substr(1,sEmail.length-2);
					aDistribList = sDistribList.split('::');

					sAccId = '';
					sFolId = '';
					aName = [];

					//Identifikace moľného distribučního seznamu
					switch(aDistribList.length <= 3 ? aDistribList.length : 3)
					{
						case 3:
							sAccId = aDistribList.shift();

						case 2:
						sFolId = aDistribList.shift();

						case 1:
							for(var m in aDistribList)
								aName.push(aDistribList[m]);

							sName = aName.join('::');
					}
					//Není-li zadán account, bereme primární
					if (!sAccId)
						sAccId = sPrimaryAccount;

					//Není-li zadán folder, bereme __@@ADDRESSBOOK@@__
					if (!sFolId)
						sFolId = "__@@ADDRESSBOOK@@__";//Mapping.getDefaultFolderForGWType('C');

					//Naplnění distribučního seznamu
					if (!aDistrib[sAccId])
						aDistrib[sAccId] = {};

					if (!aDistrib[sAccId][sFolId])
						aDistrib[sAccId][sFolId] = {'to':[],'cc':[],'bcc':[]};

					aDistrib[sAccId][sFolId][sType].push(sName);
				}
				else
				aResult[sType] += sEmail + ',';
			}
			aResult[sType] = aResult[sType].substr(0,aResult[sType].length-1);
		}
		aResult['distrib'] = aDistrib;

		return aResult;
	}
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                       Class Path
////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * @class	Path
 */
window.Path = {

	/**
	 * @brief   Split forder path into account path and folderID.
	 * This function is used with function which operate with AccountID and FolderID,
	 * e.g. item.copy(), item.move() etc.
	 *
	 * @param[in]  sFolderPath [string] Full folder name, e.g. 'admin@merakdemo.com/INBOX'.
	 * @return aOut  [object]
	 *    - [0]  Contains account id, e.g. 'admin@merakdemo.com'.
	 *    - [1]   Contains folder id, e.g. 'INBOX'. When only account is
	 *    specified, empty string is returned in this item.
	 *
	 * Example:
	 *    - Path.split(false) => ['', '']
	 *    - Path.split('admin@merakdemon.com/INBOX') == ['admin@merakdemon.com', 'INBOX']
	 *    - Path.split('admin@merakdemon.com') == ['admin@merakdemon.com', '']
	 */
	split:function(sFolderPath, bNamed, bIidIncluded) {
		var aid = [];
		var fid = '';
		var iid = '';

		if (Is.String(sFolderPath)) {
			var tmp = (sFolderPath || '').split('/');
			for (var i in tmp) {
				aid.push(tmp[i]);
				if (~tmp[i].indexOf('@')) {
					tmp = tmp.slice(+i + 1);
					break;
				}
			}
			if (bIidIncluded) {
				iid = tmp.pop();
			}
			fid = tmp.join('/');
		}
		aid = aid.join('/');

		return bNamed ? bIidIncluded ? {
			aid: aid,
			fid: fid,
			iid: iid
		} : {
			aid: aid,
			fid: fid
		} : bIidIncluded ? [ aid, fid, iid ] : [ aid, fid ];
	},

	/**
	 * @brief   Returns filename component of path.
	 * Given string containing a path to a file, this function will return the base name of the file.
	 * @param[in]	sPath	[string]	String containing a path to a file.
	 * @return	[string]    Filename component of path.
	 *
	 * Example: Path.basename('/home/httpd/html/index.php') == 'index.php'
	 */
	basename:function(sPath) {
		if (!Is.String(sPath)) return false;
		return sPath.split('/').pop();
	},

	basedir:function (sPath){
		if (!Is.String(sPath)) return '';
	    var tmp = sPath.split('/');
			tmp.pop();

		return tmp.join('/');
	},

	extension:function(sFile){
		return sFile&&sFile.indexOf('.')!=-1?sFile.split('?')[0].split('.').pop().toLowerCase():'';
	},

	slash:function(sPath){
		return sPath?sPath.replace(/\\/g,'/'):'';
	},
	backslash:function(sPath){
		return sPath?sPath.replace(/\//g,'\\'):'';
	},
	build:function(aPath){
		return [aPath.aid || aPath[0], aPath.fid || aPath[1]].join('/');
	}
};

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                       Class Mapping
////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * @class	Mapping
 */
window.Mapping = {

	/**
	 * @brief   Get default folder for GW item depending on its type.
	 * @param[in]   sType   [string]    One character type, e.g. 'C' or 'T'.
	 * @return  [string]    Default folder, e.g. 'Contacts'. This mapping is used:
	 *  - 'C' => 'Contacts'
	 *  - 'E' => 'Events'
	 *  - 'J' => 'Journal'
	 *  - 'N' => 'Notes'
	 *  - 'T' => 'Tasks'
	 *  - 'F' => 'Files'
	 *  - 'D' => 'Dashboard'
	 */
	getDefaultFolderForGWType: function(sType) {
		var sName = '';
		switch (sType) {
			case 'C':
			case 'L': sName = 'contacts'; break;
			case 'E': sName = 'events'; break;
			case 'J': sName = 'journal'; break;
			case 'N': sName = 'notes'; break;
			case 'T': sName = 'tasks'; break;
			case 'F': sName = 'files'; break;
			case 'D': sName = 'dashboard'; break;
			default: return false;
		}

		return Path.split(GWOthers.getItem('DEFAULT_FOLDERS',sName))[1];
	},

	/**
	 * @brief   Get form name depending on GW type.
	 * @param[in]   sType   [string]    One character type, e.g. 'C' or 'T'.
	 * @return  [string]    Default folder, e.g. 'frm_contact' or 'frm_event2'.
	 */
	getFormNameByGWType: function(sType) {
		switch (sType) {
			case 'C': return 'frm_contact';
			case 'E': return 'frm_event2';
			case 'N': return 'frm_note';
			case 'T': return 'frm_task';
			case 'J': return 'frm_journal';
			case 'L': return 'frm_distrib';
			case 'F': return 'frm_file';
			default: throw new Error('Not implemented');
		}
	},

	isGlobalSearchFolder: function(sFolder) {
		return sFolder.match(/__@@VIRTUAL@@__\/((@@SUBTREE@@)|(.*#))/) || sFolder === '__@@ARCHIVE@@__';
	}
};

window.makeIDFromIDS = function(ids, j) {
	try{
		return [ids[0], ids[1], ids[2][j]];
	}
	catch {
		//
	}
};

window.makeIDSFromID = function(id) {
	return [id[0], id[1], [id[2]]];
};

/**
 * get browser
 * Edit: 13.6.2008 12:15:57
 **/
window.currentBrowser = function (){
	var out = '',
		v = '',
		str = navigator.userAgent.toUpperCase();

	if (str.indexOf('CHROME')>-1) {
		out = 'Chrome';
		v = parseInt(str.substr(str.indexOf('CHROME/')+7).split('.')[0],10);
	} else if (str.indexOf('WEBKIT')>-1 || str.indexOf('SAFARI')>-1) {
		out = 'Safari';
		v = parseInt(str.substr(str.indexOf('SAFARI/')+7),10);
	} else if (str.indexOf('TRIDENT/')>-1) {
		out = 'MSIE11';
	} else if (str.indexOf('GECKO')>-1) {
		out = 'Mozilla';
		v = parseInt(str.substr(str.indexOf('GECKO/')+6),10);
	}

	return function(bV){
		return bV?v:out;
	};
}();

/**
 * sColor is optional
 **/

window.calendarPalette = {
	blue: '#0070EB',
	green: '#4FC980',
	brown: '#F48F51',
	purple: '#633CA3',
	yellow: '#F5C400',
	teal: '#008080',
	plum: '#dda0dd',
	red: '#EC3E53',
	aqua: '#8FDFFF',
	aquamarine: '#3DD8B4',
	orchid: '#7E5FF1',
	pink: '#ffc0cb',
	silver: '#9F9F9F',
	white: '#ffffff',
	gray: '#6B6B6B',
	lemon: '#ffff9f'
};
window.getCalendarColor = function(fid, sColor){
	var aColors = Cookie.get(['calendar_colors']) || {};

	if (!fid)
		return aColors;

	sColor = calendarPalette[sColor] || sColor;
	if (sColor){

		var old = aColors[sColor] && aColors[sColor] != fid ? aColors[sColor] : '';

		for (var i in aColors)
			if ((aColors[i] == fid) || (aColors[calendarPalette[i]] == fid)) {
				aColors[i] = '';
				Cookie.set(['calendar_colors', i]);
				break;
			}

		aColors[sColor] = fid;
		Cookie.set(['calendar_colors', sColor], fid);

		if (old)
			getCalendarColor(old);

		return sColor;
	}

	for (sColor in aColors)
		if (aColors[sColor] == fid)
			return sColor;

	var aFolders = dataSet.get('folders',[sPrimaryAccount,'__@@VIRTUAL@@__/__@@EVENTS@@__','VIRTUAL','FOLDERS']) || {};

	var freeColor = '#' + Math.floor(Math.random()*16777215).toString(16);
	for (sColor in calendarPalette) {
		sColor = calendarPalette[sColor] || sColor;
		if (!aColors[sColor] || !Is.Defined(aFolders[aColors[sColor]])){
			freeColor = sColor;
			break;
		}
	}

	aColors[freeColor] = fid;
	Cookie.set(['calendar_colors', freeColor], fid);
	return freeColor;
};

window.parseParamLine = function(sData){

	var col, aOut = [], arr;

	sData.split("\n").forEach(function(val,key){
		if (val && (val = val.trim())){
			if (key == 0){
				arr = val.toUpperCase().split('&');
				col = arr;
			}
			else{
				arr = val.split('&');
				var tmp = {};

				col.forEach(function(v,k){
					if (v && Is.String(arr[k]))
						tmp[v] = arr[k].urlDecode();
				});

				aOut.push({values:tmp});
			}
		}
	});

	return aOut;
};

window.CalendarFormatting = {
	formats: {
		"0": 'MM/DD/YY',
		"1": 'MM/DD/YYYY',
		"5": 'DD-MM-YY',
		"2": 'DD-MM-YYYY',
		"6": 'DD/MM/YY',
		"3": 'DD/MM/YYYY',
		"4": 'YYYY-MM-DD',
		"7": 'DD.MM.YY',
		"8": 'DD.MM.YYYY',
		"9": 'DD MMM YY',
		"10": 'DD MMM YYYY'
	},
	rtl_formats : false,
	rtl_langs : ['ar', 'fa', 'ps', 'ur', 'he'],
	formatRtlTransform : function () {
		var key;
		this.rtl_formats = {};
		for (key in this.formats) {
			this.rtl_formats[key] = this.formats[key].split('').reverse().join('');
		}
		return this.rtl_formats;
	},
	getFormats: function () {
		if (~this.rtl_langs.indexOf(document.documentElement.lang)) {
			return this.rtl_formats ? this.rtl_formats : this.formatRtlTransform();
		}
		return this.formats;
	},
	getFormat: function (id) {
		if (~this.rtl_langs.indexOf(GWOthers.getItem('LAYOUT_SETTINGS', 'language')||'en')) {
			return this.rtl_formats ? this.rtl_formats[id] : this.formatRtlTransform()[id];
		}
		return this.formats[id];
	},
	simplified: function (date, override) {
		return date.moment._isValid ? date.calendar(null, Object.assign({
			sameDay: 'LT',
			nextDay: '[' + getLang('CALENDAR::TOMORROW') + '] LT',
			nextWeek: (date.getMoment().localeData().longDateFormat('L').replace(/[/\-.\s]?[ij]?Y+[/\-.\s]?/, '') + ' LT'),
			lastDay: '[' + getLang('CALENDAR::YESTERDAY') + '] LT',
			lastWeek: (date.getMoment().localeData().longDateFormat('L').replace(/[/\-.\s]?[ij]?Y+[/\-.\s]?/, '') + ' LT'),
			sameElse: date.isSame(new IcewarpDate(), 'year') ? (date.getMoment().localeData().longDateFormat('L').replace(/[/\-.\s]?[ij]?Y+[/\-.\s]?/, '') + ' LT') : 'L LT'
		}, override || {})) : '';
	},
	normal: function (date, override) {
		return date.moment._isValid ? date.calendar(null, Object.assign({
			sameDay: '[' + getLang('CALENDAR::TODAY') + ']',
			nextDay: '[' + getLang('CALENDAR::TOMORROW') + ']',
			nextWeek: 'L',
			lastDay: '[' + getLang('CALENDAR::YESTERDAY') + ']',
			lastWeek: 'L',
			sameElse: 'L'
		}, override || {})) : '';
	},
	normalWithTime: function (date, override) {
		return date.moment._isValid ? date.calendar(null, Object.assign({
			sameDay: '[' + getLang('CALENDAR::TODAY') + '] LT',
			nextDay: '[' + getLang('CALENDAR::TOMORROW') + '] LT',
			nextWeek: 'L LT',
			lastDay: '[' + getLang('CALENDAR::YESTERDAY') + '] LT',
			lastWeek: 'L LT',
			sameElse: 'L LT'
		}, override || {})) : '';
	},
	normalWithWeekDay: function(date, override) {
		var today = new IcewarpDate().setTime(0);
		var diff = Math.abs(date.diff(today, 'days'));
		var withWeekDay = diff < 7 ? '[' + this.weekdayToString(date.day(), date.week() < today.week()) + ']' : 'L';
		return this.normal(date, Object.assign({
			sameElse: withWeekDay,
			lastWeek: withWeekDay,
			nextWeek: withWeekDay
		}, override || {}));
	},
	normalWithWeekDayAndTime: function(date, override) {
		var today = new IcewarpDate().setTime(0);
		var diff = Math.abs(date.diff(today, 'days'));
		var withWeekDay = diff < 7 ? '[' + this.weekdayToString(date.day(), date.week() < today.week()) + '] LT' : 'L LT';
		return this.normalWithTime(date, Object.assign({
			sameElse: withWeekDay,
			lastWeek: withWeekDay,
			nextWeek: withWeekDay
		}, override || {}));
	},

	formatStartDate: function (event) {
		var formattedDate = '';

		if (event.startdate>0){
			if (event.starttime > -1) {
				formattedDate = IcewarpDate.julian(event.startdate, event.starttime).format('L LT');
			} else {
				formattedDate = IcewarpDate.julian(event.startdate).format('L');
			}
		}

		return formattedDate;
	},
	weekdayToString: function(week_day, in_the_past) {
		return 'sunday,monday,tuesday,wednesday,thursday,friday,saturday'.split(',').map(function(day) {
			return getLang('DAYS::' + (in_the_past ? 'LAST_' : '') + day.toUpperCase());
		})[week_day];
	}
};

window.isIos = function() {
	return /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream
};

window.isoLanguageCode = function(sCode) {
	return {
		cn: 'zh',
		dk: 'da',
		jp: 'ja',
		no: 'nb',
		kr: 'ko',
		se: 'sv',
		ua: 'uk'
	}[sCode] || sCode;
};

window.getCssPath = function(sFilename, sSkin, sBase) {
	var skin = sSkin || GWOthers.getItem('LAYOUT_SETTINGS', 'skin');
	var skin_colors = JSON.stringify(dataSet.get('main', ['skin_colors']));

	return (sBase || '') + 'client/skins/' + skin + '/css/css.php?' + buildURL({
		file: sFilename,
		skin: skin,
		skin_colors: skin_colors
	});
};

window.normalizeTimezone = function(timezone) {
	/**
	 * X -> missing old timezone
	 * ? -> missing new timezone
	 */
	var timezoneMapping = {
		'UTC/GMT': 'Etc/GMT',
		'Africa/Accra': 'Africa/Abidjan',
		'Africa/Addis_Ababa': 'Africa/Nairobi',
		'Africa/Asmara': 'Africa/Nairobi',
		'Africa/Bamako': 'Africa/Abidjan',
		'Africa/Bangui': 'Africa/Algiers',
		'Africa/Banjul': 'Africa/Abidjan',
		'Africa/Blantyre': 'Africa/Johannesburg',
		'Africa/Brazzaville': 'Africa/Algiers',
		'Africa/Bujumbura': 'Africa/Johannesburg',
		'Africa/Conakry': 'Africa/Abidjan',
		'Africa/Dakar': 'Africa/Abidjan',
		'Africa/Dar_es_Salaam': 'Africa/Nairobi',
		'Africa/Djibouti': 'Africa/Nairobi',
		'Africa/Douala': 'Africa/Algiers',
		'Africa/Freetown': 'Africa/Abidjan',
		'Africa/Gaborone': 'Africa/Johannesburg',
		'Africa/Harare': 'Africa/Johannesburg',
		'Africa/Kampala': 'Africa/Nairobi',
		'Africa/Kigali': 'Africa/Johannesburg',
		'Africa/Kinshasa': 'Africa/Algiers',
		'Africa/Libreville': 'Africa/Algiers',
		'Africa/Lome': 'Africa/Abidjan',
		'Africa/Luanda': 'Africa/Algiers',
		'Africa/Lubumbashi': 'Africa/Johannesburg',
		'Africa/Lusaka': 'Africa/Johannesburg',
		'Africa/Malabo': 'Africa/Algiers',
		'Africa/Maseru': 'Africa/Johannesburg',
		'Africa/Mbabane': 'Africa/Johannesburg',
		'Africa/Mogadishu': 'Africa/Nairobi',
		'Africa/Niamey': 'Africa/Algiers',
		'Africa/Nouakchott': 'Africa/Abidjan',
		'Africa/Ouagadougou': 'Africa/Abidjan',
		'Africa/Porto-Novo': 'Africa/Algiers',
		'America/Anguilla': 'America/Barbados',
		'America/Antigua': 'America/Barbados',
		'America/Aruba': 'America/Barbados',
		'America/Atikokan': 'America/Bogota',
		'America/Blanc-Sablon': 'America/Barbados',
		'America/Cayman': 'America/Bogota',
		'America/Creston': 'America/Dawson',
		'America/Curacao': 'America/Barbados',
		'America/Dominica': 'America/Barbados',
		'America/Godthab': 'America/Nuuk', // -1 => -2
		'America/Grenada': 'America/Barbados',
		'America/Guadeloupe': 'America/Barbados',
		'America/Kralendijk': 'America/Barbados',
		'America/Lower_Princes': 'America/Barbados',
		'America/Marigot': 'America/Barbados',
		'America/Montserrat': 'America/Barbados',
		'America/Nassau': 'America/Bogota',
		'America/Nipigon': 'America/Chicago',
		'America/Pangnirtung': '', // X
		'America/Port_of_Spain': 'America/Barbados',
		'America/Rainy_River': '', // X
		'America/St_Barthelemy': 'America/Barbados',
		'America/St_Kitts': 'America/Barbados',
		'America/St_Lucia': 'America/Barbados',
		'America/St_Thomas': 'America/Barbados',
		'America/St_Vincent': 'America/Barbados',
		'America/Thunder_Bay': '', // X
		'America/Tortola': 'America/Barbados',
		'America/Yellowknife': '', // X
		'Antarctica/DumontDUrville': '', // ?
		'Antarctica/McMurdo': '', // X
		'Antarctica/Syowa': '', // X
		'Antarctica/Vostok': '', // ?
		'Arctic/Longyearbyen': '', // X
		'Asia/Aden': 'Europe/Istanbul', // Asia/+3 => Europe/+3
		'Asia/Bahrain': 'Europe/Istanbul', // Asia/+3 => Europe/+3
		'Asia/Brunei': 'Asia/Hong_Kong',
		'Asia/Istanbul': 'Europe/Istanbul',
		'Asia/Kuala_Lumpur': 'Asia/Hong_Kong',
		'Asia/Kuwait': 'Europe/Istanbul', // Asia/+3 => Europe/+3
		'Asia/Muscat': 'Asia/Baku',
		'Asia/Phnom_Penh': 'Asia/Bangkok',
		'Asia/Rangoon': 'Asia/Yangon',
		'Asia/Vientiane': 'Asia/Bangkok',
		'Atlantic/Reykjavik': '', // ?
		'Atlantic/St_Helena': '', // ?
		'Australia/Currie': 'Australia/Hobart',
		'Europe/Amsterdam': 'Europe/Andorra',
		'Europe/Bratislava': 'Europe/Andorra',
		'Europe/Busingen': 'Europe/Andorra',
		'Europe/Copenhagen': 'Europe/Andorra',
		'Europe/Guernsey': 'Europe/Dublin',
		'Europe/Isle_of_Man': 'Europe/Dublin',
		'Europe/Jersey': 'Europe/Dublin',
		'Europe/Kiev': 'Europe/Athens',
		'Europe/Ljubljana': 'Europe/Andorra',
		'Europe/Luxembourg': 'Europe/Andorra',
		'Europe/Mariehamn': 'Europe/Helsinki',
		'Europe/Monaco': 'Europe/Andorra',
		'Europe/Nicosia': 'Asia/Famagusta',
		'Europe/Oslo': 'Europe/Andorra',
		'Europe/Podgorica': 'Europe/Andorra',
		'Europe/San_Marino': 'Europe/Andorra',
		'Europe/Sarajevo': 'Europe/Andorra',
		'Europe/Skopje': 'Europe/Andorra',
		'Europe/Stockholm': 'Europe/Andorra',
		'Europe/Uzhgorod': 'Europe/Athens',
		'Europe/Vaduz': 'Europe/Andorra',
		'Europe/Vatican': 'Europe/Andorra',
		'Europe/Zagreb': 'Europe/Andorra',
		'Europe/Zaporozhye': 'Europe/Athens',
		'Indian/Antananarivo': '', // X
		'Indian/Cocos': '', // X
		'Indian/Comoro': '', // X
		'Indian/Christmas': '', // X
		'Indian/Kerguelen': 'Indian/Maldives',
		'Indian/Mahe': 'Indian/Mauritius',
		'Indian/Mayotte': '', // X
		'Indian/Reunion': 'Indian/Mauritius',
		'Pacific/Enderbury': 'Pacific/Apia',
		'Pacific/Funafuti': 'Pacific/Fiji',
		'Pacific/Chuuk': 'Pacific/Guam',
		'Pacific/Johnston': 'Pacific/Honolulu',
		'Pacific/Majuro': 'Pacific/Fiji',
		'Pacific/Midway': 'Pacific/Niue',
		'Pacific/Pohnpei': '', // X
		'Pacific/Saipan': 'Pacific/Guam',
		'Pacific/Wake': 'Pacific/Fiji',
		'Pacific/Wallis': 'Pacific/Fiji'
	};
	return timezoneMapping[timezone] || timezone;
};

async function replaceAsync(string, regexp, replacerFunction) {
	const replacements = await Promise.all(
		Array.from(string.matchAll(regexp),
			match => replacerFunction(...match)));
	let i = 0;
	return string.replace(regexp, () => replacements[i++]);
}

function sanitizeFilename(input = '') {
	var illegalRe = /[\/\?<>\\:\*\|"]/g;
	var controlRe = /[\x00-\x1f\x80-\x9f]/g;
	var reservedRe = /^\.+$/;
	var windowsReservedRe = /^(con|prn|aux|nul|com[0-9]|lpt[0-9])(\..*)?$/i;
	var windowsTrailingRe = /[\. ]+$/;

	function sanitize(input, replacement) {
		if (typeof input !== 'string') {
			throw new Error('Input must be string');
		}
		var sanitized = input
			.replace(illegalRe, replacement)
			.replace(controlRe, replacement)
			.replace(reservedRe, replacement)
			.replace(windowsReservedRe, replacement)
			.replace(windowsTrailingRe, replacement)
			.replace(new RegExp(replacement + '+', 'g'), replacement)
			.replace(new RegExp('^' + replacement), '');

		sanitized = sanitized.split('.');
		var ext = '';
		if (sanitized.length > 1) {
			ext = '.' + sanitized.pop();
		}
		sanitized = sanitized.join('.');

		return sanitized.substring(0, 255 - ext.length) + ext;
	}

	return sanitize(input, '-');
  }

if (typeof Promise.withResolvers === 'undefined') {
	window.Promise.withResolvers = function () {
		let resolve, reject;
		const promise = new Promise((res, rej) => {
			resolve = res;
			reject = rej;
		});
		return { promise, resolve, reject };
	};
}