<?php
	 

namespace tools;

use error\InvalidSearchExpressionException;

class SearchParserYoda
{
	static public $_allowedKeywords = [
		'from',
		'to',
		'subject',
		'after',
		'before',
		'has',
		'fulltext',
		'is',
		'tag',
		'taglist',
		'with',
		'yoda',
		'message-id',
	];

	static public $_keywordTranslations = [
		'after' => 'SINCE',
		'has' => 'HEADER',
		'fulltext' => 'TEXT',
		'tag' => 'KEYWORD',
		'taglist' => 'KEYWORD',
		'with' => 'X-WITH',
		'yoda' => 'X-YODA',
		'message-id' => 'HEADER MESSAGE-ID',
	];

	static public $_keywordValuesTranslations = [
		'is' => ['unread' => 'UNSEEN'],
		'has' => ['attachment' => 'HASATTACHMENT'],
	];

	static public $_default_keyword = '';

	protected $fallback = false;
	public $folder;
	public $enabled;

	protected $conditions = [];

	 
	static public function parseExpression($searchExpression)
	{
		$result = [];

		$length = strlen($searchExpression);
		$first = 0;
		$offset = 0;

		$escape = false;
		$quotes = false;
		$term = false;

		while ($offset < $length)
		{
			$ch = $searchExpression[$offset];
			if ($ch <= ' ')
			{
				if (!$quotes && $term)
				{
					$result[] = self::parseTerm(substr($searchExpression, $first, $offset - $first));
					$term = false;
				}

				$escape = false;
			}
			else
			{
				if ($ch == '\\')
				{
					$escape = !$escape;
				}
				else
				{
					if ($ch == '"')
					{
						if (!$escape)
							$quotes = !$quotes;
					}

					$escape = false;
				}

				if (!$term)
				{
					$first = $offset;
					$term = true;
				}
			}

			$offset++;
		}

		if ($term)
			$result[] = self::parseTerm(substr($searchExpression, $first, $offset - $first));

		return $result;
}


	 
	static private function parseTerm($term)
	{
		if (!preg_match_all('/^\(?(?P<operator>[+\-?])?(?P<keyword>[a-zA-Z\-]+:)?(?P<value>(".+")|([^\s+\-?:"{}]+))(?P<metadata>{[^\s"]+})?\)?$/u', $term, $matches, PREG_SET_ORDER))
			throw new InvalidSearchExpressionException("Invalid term found: '".$term."'");

		$match = $matches[0];

		$value = $match['value'];
		if ($value[0] == '"')
		{
			$escape = false;
			$length = strlen($value) - 1;
			for ($i = 1; $i < $length; $i++)
			{
				if (!$escape && $value[$i] == '"')
				throw new InvalidSearchExpressionException("Invalid value found: '".$value."' in term '".$term."'");

				if ($value[$i] == '\\')
					$escape = !$escape;
				else
					$escape = false;
			}

			if ($escape)
				throw new InvalidSearchExpressionException("Invalid value found: '".$value."' in term '".$term."'");

			$match['value'] = $value;
		}

		$result = [];
		$result['operator'] = $match['operator'];
		$result['keyword'] = !empty($match['keyword']) ? substr($match['keyword'], 0, strlen($match['keyword']) - 1) : get_called_class()::$_default_keyword;
		$result['value'] = $match['value'];
		$result['metadata'] = $match['metadata'] ?? '';

		return $result;
	}


	 
	static public function getImapCriteria($searchExpression, $defaultKeyword = 'fulltext')
	{
		return self::getImapCriteriaFromParsedData(self::parseExpression($searchExpression), $defaultKeyword);
	}


	 
	static public function getImapCriteriaFromParsedData($parsedData, $defaultKeyword = 'fulltext')
	{
		$result = '';

		if (!count($parsedData))
			return $result;

		foreach ($parsedData as $term)
		{
			$result .= $result ? ' ' : '';
			$result .= ($term['operator'] == '-' ? 'NOT ' : '').($term['operator'] == '?' ? 'X-OPT ' : '');

			$keyword = strtolower($term['keyword']);
			$value = strtolower($term['value']);
			if (!$keyword || (!in_array($keyword, get_called_class()::$_allowedKeywords) && !get_called_class()::$_keywordValuesTranslations[$keyword][$value]))
				$keyword = $defaultKeyword;

			$keywordValue = get_called_class()::$_keywordValuesTranslations[$keyword][$value];
			if ($keywordValue)
				$result .= $keywordValue;
			else
			{
				$keyword = get_called_class()::$_keywordTranslations[$keyword] ?? strtoupper($keyword);
				if ($keyword == 'HEADER')
				{
					$pos = strpos($term['value'], ':');
					if ($pos === false)
						throw new InvalidSearchExpressionException("Invalid header found: '".$term['value']."'.");

					$header = trim(substr($term['value'], 0, $pos), ' "');
					$value = trim(substr($term['value'], $pos + 1), ' "');
					if (($header == '') || ($value == ''))
						throw new InvalidSearchExpressionException("Invalid header found: '".$term['value']."'.");

					$result .= $keyword.' "'.$header.'"'.' "'.$value.'"';
				}
				else if ($keyword == 'KEYWORD')
					$result .= $keyword.' '. \SearchTool::urlEnquoteEscapeTag($term['value']);
				else {
					 					if (is_numeric($term['value'])) {
						$result .= $keyword.' '.$term['value'];  					} else {
						if(substr($term['value'],0,1)!='"' || substr($term['value'],strlen($term['value'])-1,1)!='"'){
							$term['value'] = '"'.$term['value'].'"';
						}
						$result .= $keyword.' '.$term['value'];
					}
				}
			}
		}

		return $result;
	}

	 
	static public function getSQLCriteriaFromParsedData(\SearchTool &$searchObject, $parsedData, $defaultKeyword = 'fulltext')
	{
		$result = '';
		if (!count($parsedData))
			return $result;

		$optionals = [];
		foreach ($parsedData as $term)
		{
			$val = $searchObject->termToSql($term);
			if($term['operator'] == '?'){
				$optionals[] = '('.$val.')';
			}else{
				$result .= $result ? ' AND ' : '';
				$result .= ($term['operator'] == '-' ? 'NOT ' : '');
				$result.= $val;
			}
		}
		if (!empty($optionals)){
            if($result){
			    $result = '('.$result.') OR ';
            }
			$result.= '('.join(' OR ',$optionals).')';
		}
		return $result;
	}

	public function sqlFallback(&$aFilterTag, &$oAccount, $field = 'search')
	{
		if(!$this->folder){
			throw new \Exception('search_missing_folder');
		}
		if(!$this->enabled || $this->fallback){
			require_once(SHAREDLIB_PATH . 'tools/search.php');
			$search = new \SearchTool();
			$search->setType($this->folder->getType());
			$search->setAccount($oAccount);
			$search->setFolder($this->folder);
			$search->setTimeZone($_SESSION['CLIENT_TIMEZONE'] ? $_SESSION['CLIENT_TIMEZONE'] : $_SESSION['SERVER_TIMEZONE']);

			if($this->folder->getType() == 'M' || $this->folder->getType() == 'SNOOZED'){
				$aFilterTag[$field] = str_replace('fulltext:', '', $aFilterTag[$field]);
				$aFilterTag[$field] = str_replace('tag:', 'taglist:', $aFilterTag[$field]);
			}
			$sqlCondition = $this->getSqlCondition($search);
			if(is_array($search->filter)){
				$aFilterTag = array_merge($aFilterTag, $search->filter);
			}
			if($sqlCondition){
				$aFilterTag['sql'] = $sqlCondition;
			}
		}else if(!$this->fallback){
			$aFilterTag['sql'] = '(0 = 1)';
		}

		 		if($this->folder->getType() == 'M' || $this->folder->getType() == 'SNOOZED'){
			if(isset($aFilterTag['sql'])) $aFilterTag['sql'] = preg_replace('/(?:\\s(?:OR|AND)[\\s]+)?(?\'bracket1\'\\()?{(FULLTEXT|TAG)}.*?{\\/\\2}(?(\'bracket1\')\\))/i', '', $aFilterTag['sql']);
		}
		return $aFilterTag['sql'];
	}

	public function getSqlCondition(\SearchTool &$searchObject)
	{
		return self::getSQLCriteriaFromParsedData($searchObject, $this->conditions);
	}

}
?>
