#!/usr/bin/perl

my $prog = "$0";
my $version = "1.0.1";

use strict;	

use Getopt::Long;
use Fcntl ':flock';
 # override CORE::glob in current package
use File::DosGlob 'glob';    # override CORE::glob in ALL packages (use with extreme caution!)
use File::DosGlob 'GLOBAL_glob';    
$/ = "F:\n";

my $opt_mintime = '';
my $opt_maxtime = '';
my $opt_ctid;
my $opt_hselect = "*";
my $opt_hwhere = "";
my $opt_preview = '';
my $opt_help = '';
my $opt_stdin = '';
my $opt_notsorted = '0';
my $opt_version = '';
my $opt_verbose = '';
my $opt_hcalc = '';
my $opt_fhide = '';
my $opt_rhide = '';
my $opt_chide = '';

my $result = GetOptions (
			"mintime=s" => \$opt_mintime,
			"maxtime=s" => \$opt_maxtime,
			"ctid=s"    => \$opt_ctid,
			"hselect=s"	=> \$opt_hselect,
			"hwhere=s" 	=> \$opt_hwhere,
			"preview"  	=> \$opt_preview,
			"hcalc=s"  	=> \$opt_hcalc,
			"version"  	=> \$opt_version,
			"notsorted" => \$opt_notsorted,
			"verbose"   => \$opt_verbose,
			"fhide"     => \$opt_fhide,
			"rhide"     => \$opt_rhide,
			"chide"     => \$opt_chide,
			""			=> \$opt_stdin,
			"help"		=> \$opt_help,
			);

die "$version\n" if ($opt_version);

usage() if ($opt_help);

my $finished = 0;

my $now = time;
my $minute = 60;
my $hour = 60 * $minute;
my $day = 24 * $hour;
my $week = 7 * $day;

my $search_mintime = evaltime ($opt_mintime);
my $search_maxtime = evaltime ($opt_maxtime);

my $htype;
my $htime;
my $dn;
my $ctid;
my $tid;
my $fc;
my $ip;
my $dm;
my $fs;
my $ipc;
my $rsk;
my $pt;
my $rq;
my $zh;
my $cl;

my $code;

my $header;
my $lines;
my @rvalues;
my $rlines;
my @cvalues;
my $clines;

$opt_hselect = "\$header" if ($opt_hselect eq "*");

$code = <<EOC;
:0			:while (<IN>) {
:0			:\tnext if (!parse_record());
:ctid       :\tnext unless (\$ctid eq "$opt_ctid");
:mintime	:\tnext if (\$htime lt \"$search_mintime\");
:maxtime0	:\tlast if (\$htime gt \"$search_maxtime\");
:maxtime1	:\tnext if (\$htime gt \"$search_maxtime\");
:hwhere		:\tnext if (!($opt_hwhere));
:hcalc		:\t$opt_hcalc;
:hselect	:\tprint \"$opt_hselect\\n\";
:rselect*   :\tprint \"\$rlines\";
:cselect*   :\tprint \"\$clines\";
:fdisplay   :\tprint \"F:\\n\";
:0	    	:}
:0			:return 1;
EOC

my $prepcode = prepare_code ($code);

sub prepare_code
{
	my $code = shift;
	
	$code =~ s/^\s*:0\s*://mg;
	$code =~ s/^\s*:ctid\s*://mg if ($opt_ctid);
	$code =~ s/^\s*:mintime\s*://mg if ($search_mintime);
	$code =~ s/^\s*:maxtime$opt_notsorted\s*://mg if ($search_maxtime);
	$code =~ s/^\s*:hwhere\s*://mg if ($opt_hwhere);
	$code =~ s/^\s*:hcalc\s*://mg  if ($opt_hcalc);
	$code =~ s/^\s*:hselect\s*://mg if ($opt_hselect ne "");
	$code =~ s/^\s*:rselect\*\s*://mg if (!$opt_rhide);
	$code =~ s/^\s*:cselect\*\s*://mg if (!$opt_chide);
	$code =~ s/^\s*:fdisplay\s*://mg if (!$opt_fhide);
	$code =~ s/^:.*\n//mg;
	
	return $code;
}

if ($opt_preview)
{
	print "\n";
	print "mintime: $search_mintime\n" if ($search_mintime);
	print "maxtime: $search_maxtime\n" if ($search_maxtime);
	print "-----------------------------------------------------\n";
	print $prepcode."\n";
	print "-----------------------------------------------------\n";

	exit;
}

die "missing files args\n" if (!$opt_stdin && $#ARGV < 0);

process_file ("&=STDIN",0) if ($opt_stdin);
process_files (\@ARGV, 1, 1);

########################################################################################

sub process_directory
{
	my $dir = shift;
	my $glob_expr; 
	if ($dir =~ /\s/)
	{
		$glob_expr = "\Q$dir";
	}
	else 
	{
		$glob_expr = "$dir";
	}

	print "processing directory: $glob_expr\n" if ($opt_verbose);

	my @files = glob ($glob_expr."/*");
	process_files(\@files, 0, 0);
}

sub process_wildcard
{
	my $wc = shift;
	my $glob_expr; 
	if ($wc =~ /\s/)
	{
		$glob_expr = "\Q$wc";
	}
	else 
	{
		$glob_expr = "$wc";
	}

	my @files = glob ($glob_expr);
	die "no files matching $wc \n" if ($#files < 0);
	process_files(\@files, 1, 0);
}

sub process_files
{
	my $files = shift;
	my $force = shift;
	my $wc = shift;
	
	foreach my $arg (@$files)
	{
		return if ($finished);
			
		if (-d $arg)
		{
			process_directory($arg);
		}
		elsif (-f $arg)
		{
			if ($force || $arg =~ /\.log$/)
 			{
				process_file ($arg,1);
 			}
 			else
 			{
 				print "skipping file: $arg\n" if ($opt_verbose);
 			}
		}
		elsif ($wc)
		{
			process_wildcard ($arg);
		}
		else
		{
			die "file: $arg not found\n";
		}
	}
}

sub process_file
{
	my $file = shift;
	my $regular_file = shift;
	my $code;
	
	print "processing file: $file\n" if ($opt_verbose);

	open (IN,"<$file") || die "could not open $file - $!\n";
	
	eval $prepcode || die "failed to evaluate code: $@\n";
	close (IN);
}

sub parse_record
{
	chomp;
	$rlines = "";
	$clines = "";
	($header, $lines) = split /\n/,$_,2;
	($htype,$htime,$dn,$ctid,$tid,$fc,$ip,$dm,$fs,$ipc,$rsk,$pt,$rq,$zh,$cl) = split /,/,$header;
	my @tmpvalues = split /\n/,$lines;
	my $newline = "";
	foreach my $tmpline (@tmpvalues)
	{
		if ( $tmpline=~ /^R:.*$/o)
		{
			push @rvalues, $tmpline;
			$rlines = $rlines .$tmpline."\n";
		}
		elsif ( $tmpline =~ /^C:/o)
		{
			push @cvalues, $tmpline;
			$clines = $clines .$tmpline."\n";
		}
	}

	return 1;
}

sub stime
{
	my $t = shift;
	my @ta = gmtime ($t);
	$t = sprintf ("%04d-%02d-%02d %02d:%02d:%02d", ($ta[5]+1900), $ta[4]+1, $ta[3], $ta[2], $ta[1], $ta[0]);
	return $t;
}

sub evaltime
{
	my $p = shift;
	return $p if ($p =~ /^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d$/);
	return $p if (!$p);
	my $v = eval ($p) || die "failed to evaluate time: $@\n";
	my $s = stime($v);
	return $s;
}

sub usage
{
	print "Usage:\n\t$prog [options] (files|dirs|-)\n\n"
		.'This tool does the equivalent of the following SQL statement:'."\n"
		.'         select <select option>'."\n"
		.'         from line'."\n"
		.'         where <where option>'."\n\n"
	.'Options are:'."\n"
		.'  --mintime <utc date time>'."\n\t".'Minimum date & time for header records.'."\n\n"
		.'  --maxtime <utc date time>'."\n\t".'Maximum date & time for header records.'."\n\n"
		.'  --hselect "[field, ...]"'."\n\t".'list of fields to select from the header line [H:1].'."\n\t".'When this string is empty, then the header lines are not printed.'."\n\n"
		.'  --hwhere "<line filter>"'."\n\t".'A boolean expression to filter specific line records'."\n\n"
		.'  --hcalc  "[xpression]"'."\n\t".'Enables creating on the fly new calculated fields from existing fields in order to print with the --hselect option'."\n\n"
		.'  --help'."\n\t".'print this usage information'."\n\n"
		.'  --version'."\n\t".'print the version'."\n\n"
		."\n"
		.'Optional input sources are:'."\n\n"
		.'Files - any file with .log extension. More than a single file may be used separated by spaces.'."\n"
		.'Dirs  - any directory containing files with .log extension.'."\n\t".'More than a single directory may be used separated by space.'."\n"
		.' "-" means stdin'."\n"
		.'A combination of files, directories, and stdin may be used separated by spaces.'."\n"
		
		.'Options for --hselect are:'."\n"
		.'  $htime'."\n"
		.'  $dn'."\n"
		.'  $ctid'."\n"
		.'  $tid'."\n"
		.'  $fc'."\n"
		.'  $ip'."\n"
		.'  $dm'."\n"
		.'  $fs'."\n"
		.'  $ipc'."\n"
		.'  $rsk'."\n"
		.'  $pt'."\n"
		.'  $rq'."\n"
		.'  $zh'."\n"
		.'  $cl'."\n"
 		.'Also available the following global fields:'."\n"
		.'  $now         - Current utc time in seconds since EPOCH'."\n"
		.'  $day         - Number of seconds per-day'."\n"
		.'  $hour        - Number of seconds per-hour'."\n"
		.'  $minute      - Number of seconds per-minute'."\n"
		."\n"
		.'examples:'."\n"
		."$prog --hselect '\$htime,\$ip' --hwhere '\$dm == 301'\n"
		."$prog --hselect '\$htime,\$dn,\$fullip,\$dm' --hcalc 'my \$fullip = printf(\"%03d.%03d.%03d.%03d\",\$1,\$2,\$3,\$4) if(\$ip=~/(\\d{1,3})\.(\\d{1,3})\.(\\d{1,3})\.(\\d{1,3})/)' --hwhere '\$pt == 2'\n";
	
	exit;
}
