#!/usr/bin/perl

use strict;
use Getopt::Long;
use Time::Local;

$| = 1;

my $pidfile="/tmp/ctipd_log_server.pid";
my ($opt_verbose,$opt_help)=('','');
my ($ip, $mintime, $maxtime, $force,$dmaction);
my ($permfail,$tempfail,$tempfthrottle,$accthrottle,$accept,$sort);
GetOptions ("ip=s" 	=> \$ip,
	"mintime=s"	=> \$mintime,
	"maxtime=s"	=> \$maxtime,
	"force"		=> \$force,
	"dmaction=s"	=> \$dmaction,
	"sort"		=> \$sort,
	"verbose"	=> \$opt_verbose,
	"help"		=> \$opt_help);	


if (!valid_params() || $opt_help) {
	usage();
	exit 0;
}

my $win_os=0;
$win_os=1 if ($^O eq "MSWin32");

close_file() if (defined($force));

my $dir;
if ($#ARGV < 0) {
	$dir=".";
} else { 
	$dir=join (" ",@ARGV);
}

my $cmd;
if ($win_os)
{
  $cmd=generate_cmd_win();
}
else
{
  $cmd=generate_cmd();
}
print "Command:\n".$cmd."\n" if ($opt_verbose);
system ($cmd);
exit 0;

sub generate_cmd
{
	my $tcmd="";
	if (defined ($ip)) {
		$ip=~ s/\,/\|/g if ($ip =~ /\,/);
		$ip=~s/\./\\\./g;
	}
	
	if (defined ($accept) && defined($accthrottle) && defined ($tempfail) && defined($tempfthrottle) && defined ($permfail)) 
	{
		print "All dmactions in use\n" if ($opt_verbose);
		$tcmd="./ipdlogc.pl ";
		$tcmd.=" --hwhere '\$ip=~/$ip/'" if (defined ($ip));
	}
	else
	{
		$tcmd="./ipdlogc.pl --hwhere '";
		my $condition=get_condition();
		$tcmd.="( $condition ) ";
		$tcmd.=" && \$ip=~/$ip/" if (defined ($ip));
	        $tcmd.="'";
	}

	if (defined $sort) {
		$tcmd.=" --hcalc '\$ip=~/(\\d+)\.(\\d+)\.(\\d+)\.(\\d+)/; my \$ipnum=\$1*(256**3) + \$2*(256**2) + \$3*256 + \$4;' --hselect '\$ipnum,\$ip,\$htime,\$dm,\$ctid' --fhide --rhide --chide";
	} else {
		$tcmd.=" --hselect '\$ip,\$htime,\$dm,\$ctid' --fhide --rhide --chide";
	}
	$tcmd.=" --mintime '".$mintime."'" if (defined($mintime));
	$tcmd.=" --maxtime '".$maxtime."'" if (defined($maxtime));

	$tcmd.=" $dir";
	if (defined $sort) {
		$tcmd.=" | ./csvsort.pl --by '\#\$1,\$2,\$3' | perl -n -e ' \$_=~/^\\d+,(.*)/; print \"\$1\\n\";'"; 
	} else {
		$tcmd.=" | ./csvsort.pl --by '\$1,\$2'"; 
	}
	return $tcmd;
}	

sub generate_cmd_win
{
	my $tcmd="";
	if (defined ($ip)) {
		$ip=~ s/\,/\|/g if ($ip =~ /\,/);
		$ip=~s/\./\\\./g;
	}
	
	if (defined ($accept) && defined($accthrottle) && defined ($tempfail) && defined($tempfthrottle) && defined ($permfail)) 
	{
		print "All dmactions in use\n" if ($opt_verbose);
		$tcmd="perl ipdlogc.pl ";
		$tcmd.=" --hwhere ".'"'."\$ip=~/$ip/".'"' if (defined ($ip));
	}
	else
	{
		$tcmd="perl ipdlogc.pl --hwhere ".'"';
		my $condition=get_condition();
		$tcmd.="( $condition ) ";
		$tcmd.=" && \$ip=~/$ip/" if (defined ($ip));
	  $tcmd.='"';
	}

	if (defined $sort) {
		$tcmd.=" --hcalc ".'"'."\$ip=~/(\\d+)\.(\\d+)\.(\\d+)\.(\\d+)/; my \$ipnum=\$1*(256**3) + \$2*(256**2) + \$3*256 + \$4;".'"'." --hselect " .'"'."\$ipnum,\$ip,\$htime,\$dm,\$ctid".'"'." --fhide --rhide --chide";
	} else {
		$tcmd.=" --hselect ".'"'."\$ip,\$htime,\$dm,\$ctid".'"'." --fhide --rhide --chide";
	}
	$tcmd.=" --mintime ".'"'.$mintime.'"' if (defined($mintime));
	$tcmd.=" --maxtime ".'"'.$maxtime.'"' if (defined($maxtime));

	$tcmd.=" $dir";
	if (defined $sort) {
		$tcmd.=" | perl csvsort.pl --by ".'"'."\#\$1,\$2,\$3".'"'." | perl -n -e ".'"'." \$_=~/^\\d+,(.*)/; print ".'\"$1\n\"'.";".'"'; 
	} else {
		$tcmd.=" | perl csvsort.pl --by ".'"'."\$1,\$2".'"'; 
	}
	return $tcmd;
}	

sub close_file
{
	if($win_os)
        {
                require Win32::Event;
                my $event;
                $event = Win32::Event->open("log_server_event");
                $event->set;
        }
        else
        {
				open(FH,$pidfile) or die "can't open pid file $pidfile";
				my $pid = <FH>; 
				`kill  -s hup $pid`;
				close FH;
        }
}

sub get_condition
{
	my $str="";
	if (defined ($accept) && defined($accthrottle)) {
		$str.='$dm=~/^10/';
	} elsif (defined ($accept)) {
		$str.='$dm=~/100/';
	} elsif (defined($accthrottle)) {
		$str.='$dm=~/10[1-9]/';
	}
	
	if (defined ($tempfail) && defined($tempfthrottle)) {
		$str.=" || " if ($str);
		$str.='$dm=~/^20/'; 
	} elsif (defined ($tempfail)) {
		$str.=" || " if ($str);
                $str.='$dm=~/201/';
	} elsif (defined($tempfthrottle)) {
		$str.=" || " if ($str);
                $str.='$dm=~/20[2-9]/';
	}
	
	if (defined ($permfail)) {
		$str.=" || " if ($str);
		$str.='$dm=~/^3/';
	}
	return $str;
}

sub valid_params
{
	if (! defined ($mintime) && !$opt_help) {
		print "Missing [mintime] option.\n";
		return 0;
	}
	#return 0 if (defined ($mintime) && defined ($maxtime) && $mintime ge $maxtime);
	$maxtime = "\$now" if (! defined ($maxtime));
	if (! defined($dmaction) && !$opt_help) {
		print "Missing [dmaction] option.\n";
		return 0;
	}
	if (!action_init() && !$opt_help) {
		print "Missing or invalid [dmaction] option.\n";
		return 0;
	}
	return 1;
}

sub action_init
{
	my @arr=split /,/, $dmaction;
	foreach my $key(@arr) {
		$accept="1" if ($key eq "accept");
		$accthrottle="1" if ($key eq "acceptthrottle");
		$tempfail="1" if ($key eq "tempfail");
		$tempfthrottle="1" if ($key eq "tempfthrottle");
		$permfail="1" if ($key eq "permfail");
	}
	return 1 if (defined ($accept) || defined($accthrottle) || defined ($tempfail) || defined($tempfthrottle) || defined ($permfail));	
	return 0;
}


sub usage
{
	print "SYNOPSIS:\n\tperl get_action.pl [Options] [Directory]\n\n"
	."DESCRIPTION:\n\tPrint aggregated transactions by action from logfiles in Directory (the current directory by default)\n\n"
	."OPTIONS:\n"
	."\t--ip\t\tSingle IP address or list of IP addresses separated by comma\n"
        ."\t--mintime\tReport start-time, mandatory parameter (YYYY-MM-DD HH:MM:SS)\n"
        ."\t--maxtime\tReport  end-time, default now (YYYY-MM-DD HH:MM:SS)\n"
        ."\t--force\t\tForce include in scanning the currently-open logfile\n"
        ."\t--dmaction\tReport by the following actions:\n"
        ."\t\tpermfail\taction = permfail\n"
        ."\t\ttempfail\taction = tempfail\n"
	."\t\ttempfthrottle\taction = tempfail during throttling\n"
	."\t\tacceptthrottle\taction = accept during throttling\n"
	."\t\taccept\t\taction = accept\n"
	."\t--sort\t\tReport sorted by IP address\n"
        ."\t--help\t\tPrint usage\n"
        ."\t--verbose\tPrint debug info\n\n";

}

