Program Tool;
{$I LinuxMacros.inc}

{$IFDEF MSWINDOWS}
{$R version.res}

{$IFNDEF CPUX64}
{$R RunAsInvoker.X86.res}
{$ELSE}
{$R RunAsInvoker.AMD64.res}
{$ENDIF}



{$APPTYPE CONSOLE}
{$ENDIF}

Uses
  FirstUnit,
{$IFDEF MSWINDOWS}
  Windows, ComObj,
{$ELSE}
  nxi_core,
{$ENDIF}
{$IFNDEF TOOL_ACTIVEX}
  icewarpservercom,
{$ENDIF}
  StructureUnit,
  apiconst,
  SystemUnit,
  SysUtils,
  StringUnit,
  toolunit,
  toolhelp,
  Variants,
  DBUnit,
  MimeUnit,
  Numbers,
  CommandUnit,
  GWSessionCacheUnit,
  TypesUnit,
  AccountUnit,
  LinuxWrappers,
  FileUnit,
  CleanUnit,
  StreamTools,
  LoggerHelperUnit,
  FastLoggingUnit
{$IFDEF UNICODE},AnsiStrings{$ENDIF UNICODE};

Procedure CreateOutputFile;
var
  fileStream : TFileStreamReleasable;
begin
  fileStream := TFileStreamReleasable.Create(UTF8ToString(OutputFileName), fmOpenReadWrite or fmShareDenyNone);
  SaveStringToUTF8File('', OutputFileName);
  OutputFile := TStreamWriter.Create(fileStream);

  OutputToFile := FileExistsUTF8(OutputFileName);
  if not OutputToFile then
  begin
    writeln('Creating output file "'+ OutputFileName + '" was FAILED.');
    writeln('Please press some key to continue with standard output or break the program by ctrl+c');
    FreeNil(OutputFile);
    readln;
  end;
end;

Procedure ParamError;
Begin
  Writeln('Parameters error.');
  ExitCode := R_ERROR_PARAMS;
End;

Function TryOpenDomain(Const DomainName: AnsiString; Out Exists: Boolean): TXDomainObject;
Var
  vt: Word;
  OleDomain: OleVariant;
Begin
  OleDomain := FAPI.OpenDomain(UTF8ToWideString(DomainName));
  vt := VarType(OleDomain) and varTypeMask;
  Exists := not VarIsEmpty(OleDomain)
  {$IFDEF CPUX64}
            and (((vt = varUInt64) and (OleDomain <> 0))
  {$ELSE}
            and (((vt = varInteger) and (OleDomain <> 0))
  {$ENDIF}
  {$IFDEF MSWINDOWS}
            or ((vt = varDispatch) and (Assigned(IDispatch(OleDomain))))
  {$ENDIF}
            );
  Result := TXDomainObject(TXObject(OleDomain));
End;

Var
  FParam, FValue: AnsiString;
  SubParam: ShortString;
  Command: TProcessItem;

// *** Account Actions ***

Procedure ProcessAccountAction(FMode: TProcessItem; Email: ShortString; ParamIndex: Longint);
Var
  FLUser, FUser: TXAccountObject;
  FDom, FDomain: AnsiString;
  FI: Longint;
  FQuery: AnsiString;
  TryAgain: Boolean;

Procedure ProcessAccount(Email: ShortString; Paramindex: Longint; Mode: TProcessItem; Open: Boolean);
Var
  FExport: AnsiString;
  FUsr: TXAccountObject;
  SomeOK: Boolean;
Begin
  If Open Or (Mode = piDelete) Then
    FUsr := FUser
  Else
    FUsr := FLUser;

  If Open Or (Mode = piDelete) Then
    If Not FUsr.Open(UTF8ToWideString(Email)) Then
    Begin
      Writeln(Format(AccountNotFound, [Email]));
      ExitCode := R_ERROR_API_FAILURE;
      Exit;
    End;

  // Primary email
  Email := AnsiString(FUsr.EmailAddress);

  Case Mode Of
    piDisplay: Writeln(Email, True);
    piValidate:
    Begin
      GetNextParam(ParamIndex, FParam, FValue);
      If FUsr.ValidateData(UTF8ToWideString(FParam)) Then Exit;
      Writeln(Email, True);
    End;
    piExport: FExport := Email + cComma;
    piDelete:
    Begin
      FUser.Delete;
      Writeln(Format(AccountDeleted, [Email]));
      Exit;
    End;
  End;

  // Parameters
  SomeOK := False;
  While GetNextParam(ParamIndex, FParam, FValue, Mode <> piModify) Do
    Try
      Case Mode Of
        piModify: SomeOK := FUsr.SetProperty(FParam, FValue) or SomeOK;
        piDisplay, piExport:
        Begin
          If Pos(API_ALL_VARS_MASK, FParam) = 0 Then
            FValue := WideStringToUTF8(FUsr.GetProperty(FParam))
          Else
            FValue := cCRLF + WideStringToUTF8(FUsr.GetProperties(UTF8ToWideString(FParam)));

          If Mode = piDisplay Then
            Writeln(FParam + cDisplaySep + FValue, True)
          Else
            FExport := FExport + CommaString(FValue) + cComma;
        End;
      End;
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;

  Case Mode of
    piModify:
    Begin
      If SomeOK And FUsr.Save Then
        Writeln(Format(AccountUpdated, [Email]))
      Else
      Begin
        Writeln(Format(AccountProblem, [Email]));
        If SomeOK Then
          ExitCode := R_ERROR_API_FAILURE;
      End;
    End;
    piDisplay, piValidate: Writeln('', True);
    piExport: Writeln(FExport, True);
  End;
End;

Function ImportUsers(FileName: AnsiString):Boolean;
Var
  ODom: TXDomainObject;
  FDom, LastDom, Email, S1: AnsiString;
  S : WideString;
  FC, FP: Longint;
  FExists: Boolean;
  list: TAnsiStringList;
Begin
  FP := ParamIndex;
  LastDom := '';
  S1 := '';

  // Open
  list := GetFileParamContent(FileName);
  try
    Result:=True;
    // For each line
    for S1 in list do
    Begin
      S := UTF8ToWideString(S1);

      If Length(S) = 0 Then
        Continue;
      S := S + cComma;

      // Name
      Email := WideStringToUTF8(Copy(S, 1, Pos(cComma, S) - 1));

      // Open or new
      If Not FUser.Open(UTF8ToWideString(Email)) Then
      Begin
        // Check if domain exists
        FC := Pos(cAt, Email);
        If FC <> 0 Then
        Begin
          // Name
          FDom := Copy(Email, FC + 1, Length(Email) - FC);

          // Check if already checked
          If FDom <> LastDom Then
          Begin
            // Open
            ODom := TryOpenDomain(FDom, FExists);

            // Not exists create
            If Not FExists Then
            Begin
              ODom := TXDomainObject(TXObject(FAPI.NewDomain(UTF8ToWideString(FDom))));
              ODom.ApplyTemplate('');
              If Not ODom.Save Then
              Begin
                Writeln(Format(DomainProblem, [FDom]));
                ExitCode := R_ERROR_DOMAIN_PROBLEM;
                Continue;
              End;

            End;
          End;

          // Set
          LastDom := FDom;
        End;

        FUser.New(UTF8ToWideString(Email));

        // Set proper type
        If FindParam(ParamIndex, 'U_Type', FValue) Then
           FUser.SetProperty('U_Type', FValue);

        // Apply template
        FUser.ApplyTemplate(UTF8ToWideString(Template));
      End;

      // Properties
      FUser.SetProperty('U_PASS_POLICY_Temporary_Disable', True);  //temporarily disable password policy
      ParamIndex := FP;
      While GetNextParam(ParamIndex, FParam, FValue, True) Do
      Try
        FValue := GetCommaString(WideStringToUTF8(S), ParamIndex - FP);

        FUser.SetProperty(FParam, FValue);
      Except
        On E: Exception Do
        Begin
          Writeln(AnsiString(E.Message));
          ExitCode := R_ERROR_INVALID_PROPERTY;
        End;
      End;

      // Save
      If FUser.Save Then
        Writeln(Format(AccountCreated, [Email]))
      Else
      Begin
        Writeln(Format(AccountProblem, [Email]));
        ExitCode := R_ERROR_API_FAILURE;
      End;

      FUser.SetProperty('U_PASS_POLICY_Temporary_Disable',False);  //Enable Pass policy Again
    End;

  finally
    FreeNil(list);
  End;
End;

Begin //ProcessAccountAction(FMode: TProcessItem; Email: ShortString; ParamIndex: Longint);
  // Default user instance
  FUser := TXAccountObject(TXObject(CreateObject('IceWarpServer.AccountObject')));
  FUser.TokenHandle := TokenHandle;

  If FMode = piCreate Then
  Begin
    // Already exists
    If FUser.Open(UTF8ToWideString(Email)) Then
    Begin
      Writeln(Format(ToolUnit.AccountExists, [Email]));
      ExitCode := R_ERROR_ACCOUNT_CREATE_USER_EXISTS;
      Exit;
    End;

    // Create user
    FUser.New(UTF8ToWideString(Email));

    // Set proper type
    If FindParam(ParamIndex, 'U_Type', FValue) Then
      FUser.SetProperty('U_Type', FValue);

    // Apply template
    FUser.ApplyTemplate(UTF8ToWideString(Template));

    While GetNextParam(ParamIndex, FParam, FValue) Do
    Try
      If LowerCase(FParam) = 'u_password' Then
        If FAPI.GetProperty('C_Accounts_Policies_Pass_Enable')
          And (FAPI.ManageConfig('passpolicy', 'passpolicy', UTF8ToWideString('password='+URLEncode(FValue)+'&mailbox='+URLEncode(Email))) = '0') Then
        Begin
          Writeln(ToolUnit.PasswordPolicyViolation);
          ExitCode := R_ERROR_ACCOUNT_CREATE_PASSWORD_POLICY;
          Exit;
        End;
      FUser.SetProperty(FParam, FValue);
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      end;
    End;

    If FUser.Save Then
      Writeln(Format(AccountCreated, [Email]))
    Else
    Begin
      Writeln(Format(AccountProblem, [Email]));
      ExitCode := R_ERROR_API_FAILURE;
    End;
  End
  Else

  If (FMode = piModify) Or (FMode = piDisplay) Or (FMode = piValidate) Or (FMode = piExport) Or (FMode = piDelete) Then
  Begin
    // Default mask
    GetDefaultSafeMask(Email, FMode, {IsEmail}true);

    // If not any mask then modify one single domain
    If Not IsStringMask(Email) Then
      ProcessAccount(Email, ParamIndex, FMode, True)

    // Else check each domain
    Else
    Begin
      // Parse domain and alias
      FDom := '';
      FI := Pos(cAT, Email);
      If FI > 0 Then
      Begin
        FDom := Copy(Email, FI + 1, Length(Email) - FI);
        Email := Copy(Email, 1, FI - 1);
      End;

      // Prepare query
      FQuery := Email;
      FQuery := StringReplace(FQuery, cAsterisk, cPercent, [rfReplaceAll]);
      FQuery := StringReplace(FQuery, cAsk, cDownLine, [rfReplaceAll]);
      FQuery := Format(cUALIASLIKE, [FQuery]);
      If Length(Filter) > 0 Then
        FQuery := FQuery + ' AND (' + Filter + ')';

      // New User instance
      FLUser := TXAccountObject(TXObject(CreateObject('IceWarpServer.AccountObject')));
      FLUser.TokenHandle := TokenHandle;

      // Get every domain
      For FI := 1 To FAPI.GetDomainCount Do
      Begin
        // Get the domain
        FDomain := FDom;
        If (FDom = cAsterisk) Or (FDom = '') Then
          FDomain := AnsiString(FAPI.GetDomain(FI - 1));

        While True Do
        Begin
          TryAgain := False;

          // For every user meeting the search criteria
          If FLUser.FindInitQuery(UTF8ToWideString(FDomain), UTF8ToWideString(FQuery)) Then
          Begin
            While FLUser.FindNext Do
            Begin
              // Email
              Email := AnsiString(FLUser.EmailAddress);

              // Free iteration if Delete
              TryAgain := FMode = piDelete;
              If TryAgain Then
                FLUser.FindDone;

              // Process
              ProcessAccount(Email, ParamIndex, FMode, False);

              // Break if delete
              If TryAgain Then
                Break;
            End;

            // Do not free if delete
            If Not TryAgain Then
              FLUser.FindDone;
          End;

          // Try again
          If Not TryAgain Then
            Break;
        End;

        // Continue if all
        If FDom <> cAsterisk Then
          Break;
      End;

      Unset(TXObject(FLUser));
    End;
  End
  Else

  If FMode = piImport Then
  Begin
    If Not ImportUsers(Email) Then
      writeln(Format(ImportProblem,[Email]));
  End;


  Unset(TXObject(FUser));
End;

// *** Remote Account Actions ***

Procedure ProcessRemoteAccountAction(FMode: TProcessItem; ID: ShortString; ParamIndex: Longint);
Var
  FRemote: TXRemoteAccountObject;
  FExport, FDomain: AnsiString;
  FI: Longint;

Procedure ImportRemoteAccounts(FileName: AnsiString);
Var
  S: AnsiString;
  FP: Longint;
  list: TAnsiStringList;
Begin
  FP := ParamIndex;

  // Open
  list := GetFileParamContent(FileName);
  try
    // For each line
    for S in list do
    Begin
      If Length(S) = 0 Then
        Continue;

      // Create
      FRemote.New;
      FRemote.ApplyTemplate(UTF8ToWideString(Template));

      // Properties
      ParamIndex := FP;
      While GetNextParam(ParamIndex, FParam, FValue, True) Do
      Try
        FValue := GetCommaString(S, ParamIndex - FP);
        FRemote.SetProperty(FParam, FValue);
      Except
        On E: Exception Do
        Begin
          Writeln(AnsiString(E.Message));
          ExitCode := R_ERROR_INVALID_PROPERTY;
        End;
      End;

      // Save
      If FRemote.Save Then
        Writeln(Format(AccountCreated, ['']))
      Else
      Begin
        Writeln(Format(AccountProblem, ['']));
        ExitCode := R_ERROR_API_FAILURE;
      End;
    End;
  finally
    FreeNil(list);
  End;
End;

Procedure ProcessRemoteAccount(ID: ShortString; Paramindex: Longint; Mode: TProcessItem; Open: Boolean);
Var
  SomeOK: Boolean;
Begin
  If Open Or (Mode = piDelete) Then
    If Not FRemote.Open(StrToIntDef(String(ID), 0)) Then
    Begin
      Writeln(Format(AccountNotFound, [ID]));
      ExitCode := R_ERROR_API_FAILURE;
      Exit;
    End;

  Case Mode Of
    piDisplay: Writeln(ID, True);
    piExport: FExport := ID + cComma;
    piDelete:
    Begin
      FRemote.Delete;
      Writeln(Format(AccountDeleted, [ID]));
      Exit;
    End;
  End;

  // Parameters
  SomeOK := False;
  While GetNextParam(ParamIndex, FParam, FValue, Mode <> piModify) Do
    Try
      Case Mode Of
        piModify: SomeOK := FRemote.SetProperty(FParam, FValue) or SomeOK;
        piDisplay, piExport:
        Begin
          If Pos(API_ALL_VARS_MASK, FParam) = 0 Then
            FValue := AnsiString(FRemote.GetProperty(FParam))
          Else
            FValue := cCRLF + AnsiString(FRemote.GetProperties(UTF8ToWideString(FParam)));

          If Mode = piDisplay Then
            Writeln(FParam + cDisplaySep + FValue, True)
          Else
            FExport := FExport + CommaString(FValue) + cComma;
        End;
      End;
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;

  Case Mode of
    piModify:
    Begin
      If SomeOK And FRemote.Save Then
        Writeln(Format(AccountUpdated, [ID]))
      Else
      Begin
        Writeln(Format(AccountProblem, [ID]));
        If SomeOK Then
          ExitCode := R_ERROR_API_FAILURE;
      End;
    End;
    piDisplay: Writeln('', True);
    piExport: Writeln(FExport, True);
  End;
End;

Begin
  // Default user instance
  FRemote := TXRemoteAccountObject(TXObject(CreateObject('IceWarpServer.RemoteAccountObject')));
  FRemote.TokenHandle := TokenHandle;

  // Parse domain and id
  FDomain := cAsterisk;
  FI := Pos(cAT, ID);
  If FI > 0 Then
  Begin
    FDomain := Copy(ID, FI + 1, Length(ID) - FI);
    ID := Copy(ID, 1, FI - 1);
  End;

  If FMode = piCreate Then
  Begin
    // Create new
    FRemote.New;
    FRemote.ApplyTemplate(UTF8ToWideString(Template));
    While GetNextParam(ParamIndex, FParam, FValue) Do
    Try
      FRemote.SetProperty(FParam, FValue);
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;

    // Set new ID
    ID := StringUnit.IntToStr(FRemote.Count + 1);

    If FRemote.Save Then
      Writeln(Format(AccountCreated, [ID]))
    Else
    Begin
      Writeln(Format(AccountProblem, [ID]));
      ExitCode := R_ERROR_API_FAILURE;
    End;
  End
  Else

  If (FMode = piModify) Or (FMode = piDisplay) Or (FMode = piDelete) Or (FMode = piExport) Then
  Begin
    // If number then modify one single account
    If IsNumber(ID) Then
    Begin
      ProcessRemoteAccount(ID, ParamIndex, FMode, True)
    End
    // Else check each account
    Else
    Begin
      // Get every remote
      For FI := FRemote.Count DownTo 1 Do
      Begin
        // Open
        FRemote.Open(FI - 1);

        // Check if strings match then modify
        If MatchStrings(AnsiString(FRemote.GetProperty('RA_DomainString')), FDomain) Then
        Begin
           If MatchStrings(AnsiString(FRemote.GetProperty('RA_Name')), ID) Then
              ProcessRemoteAccount(StringUnit.IntToStr(FI - 1), ParamIndex, FMode, False);

        End;
      End;
    End;
  End
  Else
  If FMode = piImport Then
  Begin
    ImportRemoteAccounts(ID);
  End;

  Unset(TXObject(FRemote));
End;

// *** Domain Actions ***

Procedure ProcessDomainAction(FMode: TProcessItem; Domain: ShortString; ParamIndex: Longint);
Var
  FDomain: TXDomainObject;
  FDom: AnsiString;
  FI: Longint;

Procedure ProcessDomain(Domain: ShortString; Paramindex: Longint; Mode: TProcessItem);
Var
  FExport: AnsiString;
  FExists: Boolean;
Begin
  FDomain := TryOpenDomain(Domain, FExists);

  // Check if exists
  If Not FExists Then
  Begin
    Writeln(Format(DomainNotFound, [Domain]));
    ExitCode := R_ERROR_API_FAILURE;
    Exit;
  End;

  Case Mode Of
    piDisplay: Writeln(Domain, True);
    piExport: FExport := Domain + cComma;
    piDelete:
    Begin
      if FAPI.GetProperty('C_Domain_Cannot_be_deleted') then
      begin
        Writeln(DomainCannotBeDeleted);
        Exit;
      end;
      If FAPI.DeleteDomain(UTF8ToWideString(Domain)) Then
        Writeln(Format(DomainDeleted, [Domain]));
      Exit;
    End;
  End;

  // Parameters
  While GetNextParam(ParamIndex, FParam, FValue, Mode <> piModify) Do
  Try
    Case Mode Of
      piModify:
      begin
        if (LowerCase(FParam)='d_name') And FAPI.GetProperty('C_Domain_Cannot_be_renamed') then
        begin
          Writeln(DomainCannotBeRenamed);
          Exit;
        end;
        FDomain.SetProperty(FParam, FValue);
      end;
      piDisplay, piExport:
      Begin
        If Pos(API_ALL_VARS_MASK, FParam) = 0 Then
          FValue := AnsiString(FDomain.GetProperty(FParam))
        Else
          FValue := cCRLF + AnsiString(FDomain.GetProperties(UTF8ToWideString(FParam)));

        If Mode = piDisplay Then
          Writeln(FParam + cDisplaySep + FValue, True)
        Else
          FExport := FExport + CommaString(FValue) + cComma;
      End;
    End;
  Except
    On E: Exception Do
    Begin
      Writeln(AnsiString(E.Message));
      ExitCode := R_ERROR_INVALID_PROPERTY;
    End;
  End;

  Case Mode Of
    piModify:
    Begin
      If FDomain.Save Then
        Writeln(Format(DomainUpdated, [Domain]))
      Else
      Begin
        Writeln(Format(DomainProblem, [Domain]));
        ExitCode := R_ERROR_API_FAILURE;
      End;
    End;
    piDisplay: Writeln('', True);
    piExport: Writeln(FExport, True);
  End;
End;

Procedure ImportDomains(FileName: AnsiString);
Var
  Domain, S: AnsiString;
  FP: Longint;
  FExists: Boolean;
  list: TAnsiStringList;
Begin
  FP := ParamIndex;

  // Open
  list := GetFileParamContent(FileName);
  try
    // For each line
    for S in list do
    Begin
      If Length(S) = 0 Then
        Continue;

      // Name
      Domain := StrIndex(S,0,cComma);

      // Open or create
      FDomain := TryOpenDomain(Domain, FExists);
      If Not FExists Then
      Begin
        FDomain := TXDomainObject(TXObject(FAPI.NewDomain(UTF8ToWideString(Domain))));
        FDomain.ApplyTemplate(UTF8ToWideString(Template));
      End;

      // Properties
      ParamIndex := FP;
      While GetNextParam(ParamIndex, FParam, FValue, True) Do
      Try
        FValue := GetCommaString(S, ParamIndex - FP);
        FDomain.SetProperty(FParam, FValue);
      Except
        On E: Exception Do
        Begin
          Writeln(AnsiString(E.Message));
          ExitCode := R_ERROR_INVALID_PROPERTY;
        End;
      End;

      // Save
      If FDomain.Save Then
        Writeln(Format(DomainCreated, [Domain]))
      Else
      Begin
        Writeln(Format(DomainProblem, [Domain]));
        ExitCode := R_ERROR_API_FAILURE;
      End;
    End;
  finally
    FreeNil(list);
  End;
End;

Var
  FExists: Boolean;
Begin
  If FMode = piCreate Then
  Begin
    // Check if domain already exists
    FDomain := TryOpenDomain(Domain, FExists);
    If FExists Then
    Begin
      Writeln(Format(DomainExists, [Domain]));
      ExitCode := R_ERROR_API_FAILURE;
      Exit;
    End;

    // Create domain
    FDomain := TXDomainObject(TXObject(FAPI.NewDomain(UTF8ToWideString(Domain))));
    FDomain.ApplyTemplate(UTF8ToWideString(Template));
    While GetNextParam(ParamIndex, FParam, FValue) Do
    Try
      FDomain.SetProperty(FParam, FValue);
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;
    If FDomain.Save Then
      Writeln(Format(DomainCreated, [Domain]))
    Else
    Begin
      Writeln(Format(DomainProblem, [Domain]));
      ExitCode := R_ERROR_API_FAILURE;
    End;
  End
  Else

  If (FMode = piModify) Or (FMode = piDisplay) Or (FMode = piExport) Or (FMode = piDelete) Then
  Begin
    // Default mask
    GetDefaultSafeMask(Domain, FMode, {IsEmail}false);

    // If not any mask then modify one single domain
    If Not IsStringMask(Domain) Then
      ProcessDomain(Domain, ParamIndex, FMode)

    // Else check each domain
    Else
      // Get every domain
      For FI := FAPI.GetDomainCount DownTo 1 Do
      Begin
        FDom := AnsiString(FAPI.GetDomain(FI - 1));

        // Check if strings match then modify
        If MatchStrings(FDom, Domain) Then ProcessDomain(FDom, ParamIndex, FMode);
      End;
  End

  Else
  If FMode = piImport Then
  Begin
    // Import domains
    ImportDomains(Domain);
  End;
End;

// *** System Actions ***

Procedure ProcessSystemAction(FMode: TProcessItem; ParamIndex: Longint);
Var
  APIFail, SomeOK: Boolean;
Begin
  If FMode = piModify Then
  Begin
    APIFail := False;
    SomeOK := False;

    While GetNextParam(ParamIndex, FParam, FValue) Do
    Try
      SomeOK := FAPI.SetProperty(FParam, FValue);
      If Not SomeOK Then
      Begin
        APIFail := True;
        Break;
      End
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;

    If Not APIFail And SomeOK And FAPI.Save Then
      Writeln(SettingsUpdated)
    Else
    Begin
      Writeln(SettingsProblem);
      If APIFail Then
        ExitCode := R_ERROR_API_FAILURE;
    End;
  End
  Else

  If FMode = piDisplay Then
  Begin
    While GetNextParam(ParamIndex, FParam, FValue, True) Do
    Try
      If Pos(API_ALL_VARS_MASK, FParam) = 0 Then
        FValue := AnsiString(FAPI.GetProperty(FParam))
      Else
        FValue := cCRLF + AnsiString(FAPI.GetProperties(UTF8ToWideString(FParam)));

      Writeln(FParam + cDisplaySep + FValue, True);
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_INVALID_PROPERTY;
      End;
    End;
  End;
End;


// Certificate

function GetURIValue(const ALine: ansistring; const AID: ansistring): ansistring;
begin
  // Get URI value
  Result := GetURIValueString(AID, ALine);

  // Found
  if Result <> '' then
  begin
    // Add pair
    Result := AID + cDisplaySep + Result + LineEnding;
  end;
end;

function GetCertLineInfo(const ALine: ansistring): ansistring;
begin
  Result :=
    GetURIValue(ALine, 'id') +
    GetURIValue(ALine, 'ip') +
    GetURIValue(ALine, 'cert') +
    GetURIValue(ALine, 'cn') +
    GetURIValue(ALine, 'issuer') +
    GetURIValue(ALine, 'org') +
    GetURIValue(ALine, 'orgunit') +
    GetURIValue(ALine, 'expiration') +
    GetURIValue(ALine, 'subject') +
    GetURIValue(ALine, 'country') +
    GetURIValue(ALine, 'bits') +
    GetURIValue(ALine, 'fingerprint');
end;

Procedure ProcessCertificateAction(FMode: TProcessItem; Type_: ShortString; ParamIndex: Longint);
var
  sel, ip, junk: AnsiString;
  certs: ansistring;
  certList: TAnsiStringList;
  certIndex: longint;
  info: ansistring;
Begin
  junk := '';
  if LowerCase(Type_) = 'ca' then
    sel := 'system/certificates/ca certificates'
  else
    sel := 'system/certificates/server certificates';

  If FMode = piDisplay Then
  begin
    // Get cert list
    certs := WideStringToUTF8(FAPI.ManageConfig(UTF8ToWideString(sel), 'list', ''));

    // Create list
    certList := nil;
    CreateStringList(certs, cLF, certList, {Trim}true);
    try
      for certIndex := 0 to certList.Count - 1 do
      begin
        // Get info
        info := GetCertLineInfo(certList[certIndex]);

        // Output
        writeln(info);
      end;
    finally
      FreeNil(certList);
    end;
  end
  else

  if FMode = piDelete then
  begin
    GetNextParam(ParamIndex,FValue,junk, True);
    writeln(WideStringToUTF8(FAPI.ManageConfig(UTF8ToWideString(sel), 'del', UTF8ToWideString('id=' + FValue))));
  end;

  if FMode = piCreate then
  begin
    GetNextParam(ParamIndex, FValue,junk, True);
    GetNextParam(ParamIndex, ip,junk, True);
    If LowerCase(ip) <> 'default' Then
      writeln(WideStringToUTF8(FAPI.ManageConfig(UTF8ToWideString(sel), 'add', UTF8ToWideString('cert=' + URLEncode(FValue)+cAnd + 'ip=' + ip))))
    Else
      writeln(WideStringToUTF8(FAPI.ManageConfig(UTF8ToWideString(sel), 'add', UTF8ToWideString('cert=' + URLEncode(FValue)+cAnd + 'default=1'))))

  end;
End;

// *** Service Actions ***

Procedure ProcessServiceAction(FMode: TProcessItem; Service: ShortString; ParamIndex: Longint);
Var
  FService: TXServiceObject;
Begin
  If FMode = piDisplay Then
  Begin
    FService := TXServiceObject(TXObject(CreateObject('IceWarpServer.StatisticsObject')));
    FService.TokenHandle := TokenHandle;
    FService.Poll(UTF8ToWideString(Service));

    While GetNextParam(ParamIndex, FParam, FValue, True) Do
    Try
      If Pos(API_ALL_VARS_MASK, FParam) = 0 Then
        FValue := AnsiString(FService.GetProperty(FParam))
      Else
        FValue := cCRLF + AnsiString(FService.GetProperties(UTF8ToWideString(FParam)));

      Writeln(FParam + cDisplaySep + FValue, True);
    Except
      On E: Exception Do
      Begin
        Writeln(AnsiString(E.Message));
        ExitCode := R_ERROR_SERVICE_PROBLEM;
      End;
    End;
  End;
End;

// *** Tables Actions ***

Procedure ProcessTables(FMode: TProcessItem; Service: AnsiString; ConnectionString: ShortString);
var
  table: TTablesService;
Begin
  // Get table
  if IsNumber(Service) then
    table := TTablesService(StrToCardinal(Service))
  else
    table := GetTableStringTableService(Service);

  If FMode = piCreate Then
  Begin
    If FAPI.CreateTables(Longint(table), UTF8ToWideString(ConnectionString)) Then
      Writeln(SettingsUpdated)
    Else
    Begin
      Writeln(SettingsProblem);
      ExitCode := R_ERROR_API_FAILURE;
    End;
  End
  Else

  If FMode = piDelete Then
  Begin
    If FAPI.DropTables(Longint(table)) Then
      Writeln(SettingsUpdated)
    Else
    Begin
      Writeln(SettingsProblem);
      ExitCode := R_ERROR_API_FAILURE;
    End;
  End;
End;

// *** Backup Actions ***

Procedure ProcessBackupAction(FMode: TProcessItem; FileName: ShortString; Param: Longint);
Begin
  If (FMode = piCreate) Or (FMode = piExport) Then
  Begin
    If FAPI.BackupConfig(UTF8ToWideString(FileName)) Then
      Writeln(Format(BackupFinished, [FileName]))
    Else
    Begin
      Writeln(BackupProblem);
      ExitCode := R_ERROR_BACKUP_PROBLEM;
    End;
  End
  Else

  If FMode = piImport Then
  Begin
    If FAPI.RestoreConfig(UTF8ToWideString(FileName), UTF8ToWideString(ParamStr(Param))) Then
      Writeln(Format(RestoreFinished, [FileName]))
    Else
    Begin
      Writeln(RestoreProblem);
      ExitCode := R_ERROR_RESTORE_PROBLEM;
    End;
  End;
End;

// *** Storage Actions ***

function ProcessGWExport(AMask, AFileName: AnsiString): AnsiString;
var
  gwSession: TGWSession;
  res: AnsiString;
begin
  if not CheckGWIsAvailable then
    Exit('[Error] GW is not available');

  gwSession:= TGWSession.Create;
  try
    if not gwSession.LoginSuperuser then
      Exit('[Error] GW cannot get session')
    else
    begin
      res := gwSession.GWScalarBasicFunctionCall('exportdata', AMask);

      if res = '' then
        Exit('[Error] GW exportdata error');

      // save to file
      if SaveStringToFile(res, AFileName) then
        Result := '[OK] GW data saved to "' + AFileName + '".'
      else
        Result := '[Error] GW could not save data to "' + AFileName + '".'
    end;
  finally
    FreeNil(gwSession);
  end;
end;

function ProcessGWImport(AFileName: AnsiString): AnsiString;
var
  gwSession: TGWSession;
  data, res: AnsiString;
begin
  if not CheckGWIsAvailable then
    Exit('[Error] GW is not available');

  gwSession:= TGWSession.Create;
  try
    if not gwSession.LoginSuperuser then
      Exit('[Error] GW cannot get session')
    else
    begin
      data := LoadFileToString(AFileName);
      res := gwSession.GWScalarBasicFunctionCall('importdata', data);

      if res = '' then
        Exit('[Error] GW importdata error');

      Result := '[OK] GW data imported';
    end;
  finally
    FreeNil(gwSession);
  end;
end;

Procedure ToolRun; Forward;

Procedure ProcessOptions; Forward;

// *** File Actions ***

Procedure ProcessBatch(BatchFile, Prefix: Shortstring);
Var
  line, str: AnsiString;
  list: TAnsiStringList;
Begin
  // Open
  list := GetFileParamContent(BatchFile);
  try
    // For each command
    for str in list do
    begin
      line := Trim(str);

      // Blank
      If line = '' Then
        Continue;

      // Section title
      If Pos(cSolidL, line) = 1 Then
        Continue;

      // Comment
      If Pos(cComment, line) = 1 Then
        Continue;

      // Parse and run
      If ParseBatchLine(Prefix + line) Then
      begin
        ProcessOptions;

        //writeln(Prefix + line);
        ToolRun;
      end;
    End;
  finally
    FreeNil(list);
  end;
End;

Procedure ProcessLog();
Begin
  FAPI.DoLog(StrToNum(ParamStr(ParamOptions + 2)), StrToNum(ParamStr(ParamOptions + 3)), UTF8ToWideString(ParamStr(ParamOptions + 4)), UTF8ToWideString(ParamStr(ParamOptions + 5)), UTF8ToWideString(ParamStr(ParamOptions + 6)));
End;

Procedure ProcessMerge;
var
  destFile, sourceFile: ansistring;
  sourceSettings, destSettings: TMailServer;
Begin
  // Only local connections are supported
  if TokenHandle <> 0 then
  begin
    Writeln(MergeFailed);
    ExitCode := R_ERROR_MERGE_PROBLEM;
    exit;
  end;

  // Source
  sourceFile := ParamStr(ParamOptions + 2);

  // Load source
  ClearRecord(sourceSettings, SizeOf(sourceSettings));
  if not LoadConfigInternal(sourceFile, sourceSettings) then
  begin
    Writeln(MergeFailed);
    ExitCode := R_ERROR_MERGE_PROBLEM;
    exit;
  end;

  // Dest
  destFile := WideStringToUTF8(FAPI.GetConfigFile);

  // Load Dest
  ClearRecord(destSettings, SizeOf(destSettings));
  if not LoadConfigInternal(destFile, destSettings) then
  begin
    Writeln(MergeFailed);
    ExitCode := R_ERROR_MERGE_PROBLEM;
    exit;
  end;

  // Merge properties
  destSettings.GWSuperpass := sourceSettings.GWSuperpass;
  destSettings.SuperUserPassword := sourceSettings.SuperUserPassword;
  destSettings.LocalUserPassPrefix := sourceSettings.LocalUserPassPrefix;
  destSettings.SmartAttachKey := sourceSettings.SmartAttachKey;
  destSettings.YodaServerID := sourceSettings.YodaServerID;

  // Save
  SaveConfig({SkupDomains}true, {TouchOnly}false, '', @destSettings);
End;

Procedure ProcessFiles(SubParam: ShortString; Param: ShortString);
Begin
  // Batch
  If SubParam = CBATCH Then ProcessBatch(Param, ParamStr(ParamOptions + 4));
End;

Procedure ProcessCleanAction(SubParam: ShortString);
Var
  res: TCleanResult;
Begin
  res := PerformClean(SubParam);
  If res <> crDone Then
    ExitCode := R_CLEAN_BASE + Ord(res);
End;

function CreateSettingsAction: Boolean;
begin
  // Only local connections are supported
  if TokenHandle <> 0 then
  begin
    Writeln(ErrLocalOnly);
    Exit(False);
  end;

  // Init ConfigPath
  Init;

  // Config must not exist
  if FileExistsUTF8(ConfigPath + stConfigFile) then
  begin
    Writeln(ErrSettingsExists);
    Exit(False);
  end;

  // Create
  CheckDir(ConfigPath);
  LoadConfig;
  Result := True;
end;

Procedure ProcessUpgrades(Previous, NoForce: ShortString);
Begin
  // Call upgrade functions
  FAPI.ProcessUpgradeTasks(UTF8ToWideString(Previous), UTF8ToWideString(NoForce));
End;

Procedure ProcessManageConfig(ASelector, ACommand, AParams: ansistring);
var
  res: WideString;
Begin
  // Call
  res := FAPI.ManageConfig(UTF8ToWideString(LowerCase(ASelector)), UTF8ToWideString(ACommand), UTF8ToWideString(AParams));

  // Result
  writeln(WideStringToUTF8(res));
End;

Procedure ProcessSearch(Str: ShortString);
Begin
  Writeln(AnsiString(FAPI.LocateString(UTF8ToWideString(Str))));
End;

Procedure ToolRun;
Begin
  // Search functions
  If Command = piSearch Then
    ProcessSearch(SubParam)

  // ManageConfig
  else if Command = piManageConfig Then
    ProcessManageConfig(SubParam, ParamStr(ParamOptions + 3), ParamStr(ParamOptions + 4))

  // Upgrade functions
  Else If Command = piUpgrade Then
    ProcessUpgrades(SubParam, ParamStr(ParamOptions + 3))

  // Log functions
  Else If Command = piLOG Then
    ProcessLog

  // Merge
  Else If Command = piMERGE Then
    ProcessMerge

  // Account functions
  Else If SubParam = CACCOUNT Then
    ProcessAccountAction(Command, ParamStr(ParamOptions + 3), 4)

  // Remote account functions
  Else If SubParam = CREMOTEACCOUNT Then
    ProcessRemoteAccountAction(Command, ParamStr(ParamOptions + 3), 4)

  // Domain functions
  Else If SubParam = CDOMAIN Then
    ProcessDomainAction(Command, ParamStr(ParamOptions + 3), 4)

  // Tables functions
  Else If SubParam = CTABLES Then
    ProcessTables(Command, ParamStr(ParamOptions + 3), ParamStr(ParamOptions + 4))

  // System functions
  Else If SubParam = CSYSTEM Then
    ProcessSystemAction(Command, 3)

  // User statistics
  Else If SubParam = CUSERSTATISTICS Then
    Writeln(AnsiString(FAPI.GetUserStatistics(UTF8ToWideString(ParamStr(ParamOptions + 3)), UTF8ToWideString(ParamStr(ParamOptions + 4)), UTF8ToWideString(ParamStr(ParamOptions + 5)))))

  // Certificate functions
  Else If SubParam = CCERTIFICATE Then
    ProcessCertificateAction(Command, ParamStr(ParamOptions + 3), 4)

  // Service functions
  Else If SubParam = CSERVICE Then
    ProcessServiceAction(Command, ParamStr(ParamOptions + 3), 4)

  // Service functions
  Else If SubParam = CBACKUP Then
    ProcessBackupAction(Command, ParamStr(ParamOptions + 3), 4)

  // Groupware
  Else If (Command = piExport) and (SubParam = CGROUPWARE) Then
    Writeln(ProcessGWExport(ParamStr(ParamOptions + 3){mask}, ParamStr(ParamOptions + 4){filename}))
  Else If (Command = piImport) and (SubParam = CGROUPWARE) Then
    Writeln(ProcessGWImport(ParamStr(ParamOptions + 3){filename}))

  // File functions
  else If Command = piFile Then ProcessFiles(SubParam, ParamStr(ParamOptions + 3))

  // Clean functions
  Else If Command = piClean Then
    ProcessCleanAction(SubParam)

  // Create settings check
  Else If (SubParam = CSETTINGS) And (Command = piCreate) Then
    WriteLn(SettingsCreated)

  Else
    // Error
    ParamError;
End;

Procedure ProcessOptions;
Const
  ParamChars = [cSlash, cDash];
Var
  FI: Longint;
  Param: AnsiString;

Procedure ProcessSwitch(Switch: AnsiString; Var Index: Longint);
Var
  param, value: AnsiString;
  equalPos: Longint;
  isEqual: boolean;

Function GetValue: AnsiString;
Begin
  Result := value;

  // Only get the next arg if = is missing
  If (Result = '') and not isEqual Then
  Begin
    Inc(Index);
    Result := ParamStr(Index);
  End;
End;

Begin
  // Parase parm and value
  param := Switch;
  value := '';
  equalPos := Pos(cEqual, Switch);
  isEqual := equalPos <> 0;
  If isEqual Then
  Begin
    value := Copy(Switch, equalPos + 1, Length(Switch) - equalPos);
    param := Copy(Switch, 1, equalPos - 1);
  End;
  param := UpperCase(param);

  // Options
  If (param = CQUIET) Or (param = CQ) Then
    QuietMode := True
  Else If (param = CRPC) Or (param = CREMOTE) Or (param = CRCP) Or (param = CR) Then
    CreateRPC(GetValue)
  Else If (param = CPATH) Or (param = CP) Then
    ToolUnit.InitPath := GetValue
  Else If (param = CFILTER) Or (param = CF)  Then
    Filter := GetValue
  Else If (param = CTEMPLATE) Or (param = CM) Then
    Template := GetValue
  Else If (param = CH) Or (param = CHELP) Or (param = CASK) Then
  Begin
    Help(ParamStr(FI + 1));
    Halt(0);
  End
  Else If (param = CT) Or (param = CTUTORIAL) Then
    Help('tutorial')
  Else If (param = CV) Or (param = CVERSION) Then
    Help('version', False)
  Else If (param = COUTPUTFILE) Or (param = CO) then
  begin
    OutputFileName := GetValue;
    CreateOutputFile;
  end;
End;

Begin
  // Parse options parameters
  FI := 0;
  While True Do
  Begin
    Inc(FI);
    Param := ParamStr(FI);

    // Is option
    If (Length(Param) >= 2) And (Param[1] In ParamChars) Then
    Begin
      // Parse param
      Delete(Param, 1, 1);
      If Param[1] In ParamChars Then
        Delete(Param, 1, 1);

      // Match parameters
      ProcessSwitch(Param, FI);
      ParamOptions := FI;
    End
    Else
      Break;
  End;

  // Help
  If ParamCount = 0 Then
  Begin
    Help;
    Halt(0);
  End;

  If ParamOptions = ParamCount Then Halt(0);

  // Parse command and subparam
  Command := GetProcessMode(UpperCase(ParamStr(ParamOptions + 1)));
  SubParam := UpperCase(ParamStr(ParamOptions + 2));

  // Help
  If ParamCount <= ParamOptions + 1 Then
  Begin
    // Check if not in single param
    If Not (Command In [piUpgrade, piClean]) Then
    Begin
      Help(ParamStr(ParamOptions + 1));
      Halt(0);
    End;
  End;
End;

Var
  LastErr: Integer;
{$IFDEF LINUX}
  pw: Pnxt_passwd;
{$ENDIF}

Begin
  {$IFDEF MSWINDOWS}
  // check, if running from administration console
  If Not IsUserAnAdmin Then
  Begin
    Writeln(ElevationRequired);
    Halt(ERROR_ELEVATION_REQUIRED);
  End;
  {$ENDIF}

  If TRUE <> ReadEnvironmentVariables Then
    Halt(R_ERROR_ENV_VARS);

  {$IFDEF LINUX}
  pw := NXGetpwnam(PAnsiChar(AnsiString(ProcessUser)));
  If pw = Pointer(nxc_NULL) Then
  Begin
    Writeln(SafeFormat(ErrCannotFindUser, [ProcessUser]));
    Halt(R_ERROR_USER_SWITCH);
  End;

  If pw^.pw_uid <> nxf_geteuid() Then
    If TRUE <> SetProcessUserIDGroupID(ProcessUser) Then
    Begin
      Writeln(SafeFormat(ErrCannotSwitchUser, [ProcessUser]));
      Halt(R_ERROR_USER_SWITCH);
    End;
  {$ENDIF}

  ParseParameters;
  ProcessOptions;

  If (SubParam = CSETTINGS) And (Command = piCreate) Then
  Begin
    If Not CreateSettingsAction Then
    Begin
      ExitCode := R_ERROR_SETTINGS_PROBLEM;
      Halt(ExitCode);
    End;
  End;

  APIInit(Command, SubParam);
  ToolRun;

  If IDispatch(FAPI) <> Nil Then
  Begin
    LastErr := FAPI.LastErr;
    If LastErr <> 0 Then
    Begin
      If ((LastErr + R_API_LASTERROR_BASE) > R_API_GENERIC_ERROR)
        Or ((LastErr + R_API_LASTERROR_BASE) <= _R_ERROR_TOP) Then
      Begin
        LastErr := R_API_GENERIC_ERROR
      End;

      ExitCode := LastErr;
    End;
  End;

  if OutputToFile then
    FreeNil(OutputFile);

  PerformCustomFinalizations;
  APIDone;

  globalFastLogger.Terminate;
  TLoggerService.DoneGlobalLogger(Nil);
End.
