/*																	*
 * This file is a part of an example of how to communicate			*
 * with IceWarp Unified Messaging server using XML-RPC				*
 *																	*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.				*
 *																	*
 * IceWarp Ltd 2011													*
 *																	*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.IO;

namespace Icewarp.XmlRpcExample.Utils
{
	public class Logger
	{
		#region Nested types
		public class Helper
		{
			#region Fields
			private string m_publisher;
			private string m_counter;
			#endregion // Fields

			#region Ctor
			/// <summary>
			/// Ctor
			/// </summary>
			/// <param name="publisher">Publisher of the log</param>
			public Helper(string publisher)
			{
				m_publisher = publisher;
				m_counter = Logger.Instance.GetCounter();

				Logger.Registrator.Instance.Register(this);
			}
			#endregion // Ctor

			#region Public methods
			/// <summary>
			/// Writes log
			/// </summary>
			/// <param name="level">Level of priority</param>
			/// <param name="message">Message to be logged</param>
			public void WriteLog(LogLevel level, string message)
			{
				Logger.Instance.WriteLog(level, m_publisher, m_counter, message);
			}

			/// <summary>
			/// Writes binary file
			/// </summary>
			/// <param name="message"></param>
			/// <param name="array"></param>
            public void WriteByteLog(string message, byte[] array)
            {
                string newFileName = Guid.NewGuid().ToString();
                WriteLog(LogLevel.HighPriority, String.Format("{0}, saved to {1}", message, newFileName));
                Logger.Instance.WriteByteLog(newFileName, array);
            }

			/// <summary>
			/// Closes underlying logs
			/// </summary>
			public void Close()
			{
				Logger.Instance.Close();
			}
			#endregion // Public methods
		}

		public class Registrator
		{
			#region Singleton
			static readonly Registrator instance=new Registrator();

			// Explicit static constructor to tell C# compiler
			// not to mark type as beforefieldinit
			static Registrator()
			{
			}

			Registrator()
			{
			}

			public static Registrator Instance
			{
				get
				{
					return instance;
				}
			}
			#endregion // Singleton

			#region Fields
			private List<Logger.Helper> m_openedLogs = new List<Helper>();
			#endregion // Fields

			#region Public methods
			public void Register(Logger.Helper helper)
			{
				m_openedLogs.Add(helper);
			}

			public void Close()
			{
				foreach (Logger.Helper helper in m_openedLogs)
				{
					helper.Close();
				}
			}
			#endregion // Public methods
		}

		public enum LogLevel
		{
			/// <summary>
			/// Prints all log (Debug,Low,Normal,High)
			/// </summary>
			DebugPriority = 0,
			/// <summary>
			/// Prints all logs beginning low and higher priority logs (Low,Normal,High)
			/// </summary>	 
			LowPriority = 1,
			/// <summary>
			/// Prints normal and higher priority logs (Normal,High)
			/// </summary>
			NormalPriority = 2,
			/// <summary>
			/// Prints only higher priority logs (High)
			/// </summary>
			HighPriority = 3,
			/// <summary>
			/// Highest priority
			/// </summary>
			Error = 4,
		}
		#endregion // Nested types

		#region Singleton
		Logger()
		{
		}

		public static Logger Instance
		{
			get
			{
				return Nested.instance;
			}
		}

		class Nested
		{
			// Explicit static constructor to tell C# compiler
			// not to mark type as beforefieldinit
			static Nested()
			{
			}

			internal static readonly Logger instance = new Logger();
		}
		#endregion // Singleton

		#region Fields
		private FileLog m_log;
		private int m_logEntryCounter;
		private string m_fullPath;
		private static bool m_directoryGenerated;
		private static string m_directoryName;

		#region Presets
		private LogLevel m_logLevel = LogLevel.NormalPriority;
		private readonly object m_locker = new object();		
		#endregion // Presets
		#endregion // Fields

		#region Properties
        /// <summary>
        /// Synchronization object
        /// </summary>
		protected object Locker
		{
			get
			{
				return m_locker;
			}
		}
		#endregion // Properties

		#region Public methods
		/// <summary>
		/// Sets log level
		/// </summary>
		/// <param name="level"></param>
		public void SetLogLevel(LogLevel level)
		{
			m_logLevel = level;
		}

		/// <summary>
		/// Creates instance of log mechanism
		/// </summary>
		/// <param name="fullPath"></param>
		public void CreateLog(string fullPath, string fileName)
		{
			lock (Locker)
			{
				m_log = new FileLog(m_fullPath = fullPath, fileName);
			}
		}

		/// <summary>
		/// Returns counter of the assigned log (hex)
		/// </summary>
		/// <returns></returns>
		public string GetCounter()
		{
			lock (Locker)
			{
				return String.Format("{0:X8}", m_logEntryCounter++);
			}
		}

		/// <summary>
		/// Writes log with publisher stamp, set to normal priority
		/// </summary>
		/// <param name="publisher"></param>
		/// <param name="message"></param>
		public void WriteLog(string publisher, string counter, string message)
		{
			WriteLog(LogLevel.NormalPriority, publisher, counter, message);
		}

		/// <summary>
		/// Writes log without publisher, set to normal priority
		/// </summary>
		/// <param name="message"></param>
		public void WriteLog(string message)
		{
			WriteLog(LogLevel.NormalPriority, message);
		}

		/// <summary>
		/// Writes log with publisher
		/// </summary>
		/// <param name="level"></param>
		/// <param name="publisher"></param>
		/// <param name="message"></param>
		public void WriteLog(LogLevel level, string publisher, string counter, string message)
		{
			if (level >= m_logLevel)
			{
				lock (Locker)
				{
					if (m_log != null)
					{
						m_log.WriteLog(publisher, counter, message);
					}
				}
			}
		}

        public void WriteByteLog(string fileName, byte[] array)
        {
            string newName = String.Format("{0}\\{1}", m_fullPath, fileName);

            using (Stream stream = File.Create(newName))
            {
                if (stream != null)
                {
                    stream.Write(array, 0, array.Length);
                    stream.Close();
                }
            }
        }

		/// <summary>
		/// Writes log without publisher
		/// </summary>
		/// <param name="level"></param>
		/// <param name="message"></param>
		public void WriteLog(LogLevel level, string message)
		{
			if (level >= m_logLevel)
			{
				lock (Locker)
				{
					if (m_log != null)
					{
						m_log.WriteLog(message);
					}
				}
			}
		}

		/// <summary>
		/// Closes log
		/// </summary>
		public void Close()
		{
			m_log.Close();
		}

		public static string GetFileName()
		{
			DateTime now = DateTime.Now;
			int counter = 0;
			string fullPath;

			do
			{
				fullPath = String.Format("{0:MM_dd_yyyy_hh_mm_ss}__{1}.txt", now, counter);
				counter++;
			}
			while (File.Exists(fullPath) && (counter < 100));

			return fullPath;
		}

		public static string GetDirectoryName()
		{
			return GetDirectoryName(String.Empty);
		}

		public static string GetDirectoryName(string initialDirectory)
		{
			if (!m_directoryGenerated)
			{
				DateTime now = DateTime.Now;
				int counter = 0;

				do
				{
					m_directoryName = String.Format("{2}\\{0:MMddyyyy@hhmmss}#{1}", now, counter, initialDirectory);
					counter++;
				}
				while (Directory.Exists(m_directoryName) && (counter < 100));

				m_directoryGenerated = !Directory.Exists(m_directoryName);
			}

			return m_directoryName;
		}

		public static string GetTimeStamp()
		{
			DateTime now = DateTime.Now;
			return String.Format("{0:hh:mm:ss.ff-MM/dd/yyyy}", now);
		}
		#endregion // Public methods
	}
}