/*																	*
 * This file is a part of an example of how to communicate			*
 * with IceWarp Unified Messaging server using XML-RPC				*
 *																	*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.				*
 *																	*
 * IceWarp Ltd 2011													*
 *																	*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading;

namespace Icewarp.XmlRpcExample.Utils
{
	public class FileLog : IDisposable
	{
		#region Fields
        private bool m_isDisposed;
		private string m_directory;
		private string m_fileName;
		private string m_fullPath;		
		private StreamWriter m_logFile;		
		#endregion // Fields

		#region Properties
		/// <summary>
		/// Directory where the logs are stored
		/// </summary>
		public string TargetDirectory
		{
			get
			{
				return m_directory;
			}
		}
		#endregion // Properties

		#region Ctor
		/// <summary>
		/// Ctor
		/// </summary>
		/// <param name="directory">Target directory</param>
		public FileLog(string directory, string fileName)
		{
			m_directory = directory;
			Initialize(m_fileName = fileName);            
		}
		#endregion // Ctor

		#region Public methods
        /// <summary>
        /// Writes message to the log
        /// </summary>
        /// <param name="message">Message to be saved</param>
		public void WriteLog(string message)
		{			
			m_logFile.WriteLine(String.Format("[{0}][#{1}] {2}", Logger.GetTimeStamp(), Thread.CurrentThread.Name, message));
			m_logFile.Flush();
		}

        /// <summary>
        /// Writes signed message to the log
        /// </summary>
        /// <param name="publisher">Publisher</param>
        /// <param name="message">Message to be saved</param>
        public void WriteLog(string publisher, string counter, string message)
        {
            WriteLog(String.Format("[{0}] <{1}> {2}", counter, publisher, message));
        }

		/// <summary>
		/// Closes logfile
		/// </summary>
		public void Close()
		{
			m_logFile.Close();
		}
		#endregion // Public methods

		#region Private methods
		private void Initialize(string fileName)
		{
			Directory.CreateDirectory(m_directory);
			m_logFile = File.CreateText(m_fullPath = String.Format("{0}\\{1}", m_directory, fileName));
		}
		#endregion // Private methods

        #region IDisposable Members

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            // If you need thread safety, use a lock around these 
            // operations, as well as in your methods that use the resource.
            if (!m_isDisposed)
            {
                if (disposing)
                {
                    if (m_logFile != null)
                    {
                        m_logFile.Close();
                        m_logFile.Dispose();
                        m_logFile = null;
                    }
                }

                // Indicate that the instance has been disposed.
                m_isDisposed = true;
            }
        }

        #endregion // IDisposable Members
    }
}
