#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: license_operations.sh - wizard menu script
#

source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"

fnc_license_info()
{
    TMP_FILE=$(mktemp)
    run_cmd_print_stdout_log "${IWS_INSTALL_DIR}/tool.sh" "display" "system" "c_license_xml" > ${TMP_FILE}
    # parse product

    TMP_IFS=$IFS
    IFS=$'\n'
    PRODUCTS_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<product/?>" | sed "s|.*<product[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    SUBPRODUCTS_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<subproduct/?>" | sed "s|.*<subproduct[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    DOMAINS_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<domains/?>" | sed "s|.*<domains[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    ACCOUNTS_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<accounts/?>" | sed "s|.*<accounts[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    EXPIRESAFTER_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<expiresafter/?>" | sed "s|.*<expiresafter[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    DEMOEXPIRESAFTER_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<demoexpiresafter/?>" | sed "s|.*<demoexpiresafter[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    CREATIONDATE_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<creationdate/?>" | sed "s|.*<creationdate[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    DEMO_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<demo/?>" | sed "s|.*<demo[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")

    CUSTOMERNAME_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<customername/?>" | sed "s|.*<customername[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    CUSTOMEREMAIL_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<customeremail/?>" | sed "s|.*<customeremail[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    LICENSEHOLDER_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<licenseholder/?>" | sed "s|.*<licenseholder[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    PURCHASEID_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<purchaseid/?>" | sed "s|.*<purchaseid[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    PURCHASEDATE_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<purchasedate/?>" | sed "s|.*<purchasedate[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    LICENSEVERSION_STRING=$(cat ${TMP_FILE} | tr -d "\r" | grep -E "<version/?>" | sed "s|.*<version[^>]*>\([^<]*\).*|\1|g" | sed "s/^$/ /")
    REQUIREDLICENSEVERSION_STRING=$(get_api_variable "system" "C_License_Minimal_Required_Version")
    LICENSESTATUS_STRING=""

    PRODUCTS_ARRAY=($PRODUCTS_STRING)
    SUBPRODUCTS_ARRAY=($SUBPRODUCTS_STRING)
    DOMAINS_ARRAY=($DOMAINS_STRING)
    ACCOUNTS_ARRAY=($ACCOUNTS_STRING)
    EXPIRESAFTER_ARRAY=($EXPIRESAFTER_STRING)
    DEMOEXPIRESAFTER_ARRAY=($DEMOEXPIRESAFTER_STRING)
    CREATIONDATE_ARRAY=($CREATIONDATE_STRING)
    DEMO_ARRAY=($DEMO_STRING)

    NUM_PRODUCTS=${#PRODUCTS_ARRAY[*]}
    
    if [[ -z "$LICENSEVERSION_STRING" ]]; then
          LICENSESTATUS_STRING="Evaluation"
    else
        if [ "$REQUIREDLICENSEVERSION_STRING" -gt "$LICENSEVERSION_STRING" ]; then
            LICENSESTATUS_STRING="Incorrect license version (required license version ${REQUIREDLICENSEVERSION_STRING}). Please contact sales to upgrade the license." 
        else
            LICENSESTATUS_STRING="Ok"
        fi
    fi

    echo "Licenses information:"
    echo ""
    echo "Customer: ${CUSTOMERNAME_STRING}"
    echo "Customer email: ${CUSTOMEREMAIL_STRING}"
    echo "License holder: ${LICENSEHOLDER_STRING}"
    echo "Order ID: ${PURCHASEID_STRING}"
    echo "Purchase date: ${PURCHASEDATE_STRING}"
    echo "License Version: ${LICENSEVERSION_STRING}"
    echo "License Status: ${LICENSESTATUS_STRING}"
    echo ""
    echo "Product                                 Type           Creation date   Renewal period   Domains   Accounts"
    echo "----------------------------------------------------------------------------------------------------------"
    for IDX in $(seq 0 $((${NUM_PRODUCTS} - 1))); do
        VALIDFOR_ARRAY[$IDX]=${EXPIRESAFTER_ARRAY[$IDX]}
        if [ "x${DOMAINS_ARRAY[$IDX]}" == "x0" ]; then
            DOMAINS_ARRAY[$IDX]="Unlimited"
        fi
        if [ "x${ACCOUNTS_ARRAY[$IDX]}" == "x0" ]; then
            ACCOUNTS_ARRAY[$IDX]="Unlimited"
        fi
        if [ "x${EXPIRESAFTER_ARRAY[$IDX]}" == "x0" ]; then
            VALIDFOR_ARRAY[$IDX]="Never"
        fi
        if [ "x${DEMO_ARRAY[$IDX]}" == "x1" ]; then
            DEMO_ARRAY[$IDX]="Evaluation"
            if [ "${PRODUCTS_ARRAY[$IDX]}" == "Mail Server" ]; then
                INSTALL_UNIX_DAY=$(get_api_variable "system" "c_settingstime")
                CREATIONDATE_ARRAY[$IDX]=$((${INSTALL_UNIX_DAY} / (60*60*24) + 25569))
                VALIDFOR_ARRAY[$IDX]=${DEMOEXPIRESAFTER_ARRAY[$IDX]}
            fi
        else
            DEMO_ARRAY[$IDX]="Registered"
        fi

	NOW_UNIX_TIMESTAMP=$(date +"%s")
	NOW_UNIX_DAY=$((${NOW_UNIX_TIMESTAMP} / (60*60*24) + 25569))
	VALIDFOR_ARRAY[$IDX]=$((${VALIDFOR_ARRAY[$IDX]} - (${NOW_UNIX_DAY} - ${CREATIONDATE_ARRAY[$IDX]})))

        # compute creation date
        # Days between TDateTime basis (12/31/1899) and Unix time_t basis (1/1/1970)
        # UnixDateDelta = 25569
        CREATION_DATE=$((${CREATIONDATE_ARRAY[$IDX]} - 25569))
        CREATION_DATE=$(date -d "1970-01-01 ${CREATION_DATE} days" +"%Y-%m-%d")


        printf "%-20s%-20s%-15s%-16s%-17s%-10s%s\n" "${PRODUCTS_ARRAY[$IDX]}" "${SUBPRODUCTS_ARRAY[$IDX]}" "${DEMO_ARRAY[$IDX]}" "$CREATION_DATE" "${VALIDFOR_ARRAY[$IDX]}" "${DOMAINS_ARRAY[$IDX]}" "${ACCOUNTS_ARRAY[$IDX]}"
    done

    IFS=$TMP_IFS
    rm -rf $TMP_FILE

    return 0
}

fnc_license_display()
{
    good "XML formatted output follows:"
    echo
    run_cmd_print_stdout_log "${IWS_INSTALL_DIR}/tool.sh" "display" "system" "c_license"
    EXITCODE=$?
    if [ ${EXITCODE} -ne 0 ]; then
        bad "License display problem, error code ${EXITCODE}."
        return 1
    fi
    return 0
}

fnc_license_export()
{
    getparam "Enter file name to save exported license:"
    run_cmd_print_stdout_log "${IWS_INSTALL_DIR}/tool.sh" "display" "system" "c_license" > $PARAM
    EXITCODE=$?
    if [ ${EXITCODE} -eq 0 ]; then
        good "License saved to file ${PARAM}"
    else
        bad "License export problem, error code ${EXITCODE}."
        return 1
    fi
    return 0
}

fnc_license_online()
{
    hilite "Activate license online"
    good ""
    getparam "Enter order ID:"
    ORDER_ID=$PARAM
    run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "c_onlinelicense" "${ORDER_ID}"
    EXITCODE=$?
    if [ ${EXITCODE} -eq 0 ]; then
        good "License activation successful."
    else
        bad "License activation error, error code ${EXITCODE}. Please check server error log for further information."
        return 1
    fi
    return 0
}

fnc_license_offline()
{
    hilite "Activate license offline"
    good ""
    good "To get the file, please visit http://shop.icewarp.com/license/retrieve_license_customer.php"
    good "You will be asked for Order ID and Reference ID"
    good ""
    getparam "Enter path to the file with license:"
    FILENAME=$PARAM

    # check, if file exists
    if ! [ -f "${FILENAME}" ]; then
        bad "File not found."
        return 1
    fi

    # obtain license.key path
    if [ -f "${IWS_INSTALL_DIR}/path.dat" ]; then
        KEYFILE=$(sed -n '12p' < "${IWS_INSTALL_DIR}/path.dat")
        if [ "x${KEYFILE}" == "x" ]; then
            KEYFILE=$(head -n 1 < "${IWS_INSTALL_DIR}/path.dat")
            if [ "x${KEYFILE}" == "x" ]; then
                KEYFILE="${IWS_INSTALL_DIR}/config/license.key"
            else
                KEYFILE="${KEYFILE}/license.key"
            fi
        fi
    else
        KEYFILE="${IWS_INSTALL_DIR}/config/license.key"
    fi

    # extract contents of data elements
    DATA=false
    DONE=false
    while read LINE; do
        if ! $DATA; then
            if grep "^<data>" <<< ${LINE} >/dev/null; then
                DATA=true
                sed 's/^<data>//' <<< ${LINE} > "${KEYFILE}"
            fi
        else
            if grep "</data>" <<< ${LINE} >/dev/null; then
                DATA=false
                DONE=true
            else
                echo ${LINE} >> "${KEYFILE}"
            fi
        fi
    done < "${FILENAME}"

    if $DONE; then
        good "License stored to ${KEYFILE} successfully."
    else
        bad "License activation error"
    fi
}


fnc_license_refkey_display()
{
    run_cmd_print_stdout_log "${IWS_INSTALL_DIR}/tool.sh" "display" "system" "c_reference"
    return 0
}

fnc_license_refkey_export()
{
    getparam "Enter file name to save exported reference ID:"
    run_cmd_print_stdout_log "${IWS_INSTALL_DIR}/tool.sh" "display" "system" "c_reference" > $PARAM
    EXITCODE=$?
    if [ ${EXITCODE} -eq 0 ]; then
        good "Reference ID saved to file ${PARAM}"
    else
        bad "Reference ID query problem, error code ${EXITCODE}."
        return 1
    fi
    return 0
}

fnc_license_obtain_trial()
{
    hilite "Get trial license"

    # Init variables
    NAME=""
    EMAIL=""
    COUNTRY=""
    ADDRESS=""
    ZIP=""
    COUNTRY_NAME=""
    COUNTRY_CODE=""
    PHONE=""

    # Load list of countries
    MENUGRID_ITEMS=
    COUNTRY_CODES=
    COUNTER=0
    while read LINE; do
        if echo "${LINE}" | grep "COUNTRY" &>/dev/null; then
            MENUGRID_ITEMS[${COUNTER}]=$(echo "${LINE}" | sed 's/.*">//' | sed 's/<\/.*//' | tr '\n' ' ')
            COUNTRY_CODES[${COUNTER}]=$(echo "${LINE}" | sed 's/.*id="//' | sed 's/">.*//')
            COUNTER=$((${COUNTER} + 1))
        fi
    done < "${IWS_INSTALL_DIR}/scripts/wizard/countries.xml"
  
    # Input registration data
    while true; do
        good ""

        # Name
        while true; do
            getparam_with_default "Your name" "${NAME}"
            NAME=$PARAM
            if [ "x${NAME}" != "x" ]; then
                break;
            else
                echo ""
                bad "ERROR: Name cannot be empty"
            fi
        done
        
        # Email
        while true; do
            getparam_with_default "Contact email" "${EMAIL}"
            EMAIL=$PARAM
            if echo "${EMAIL}" | grep '[A-Za-z0-9._%+-]\+@[A-Za-z0-9.-]\+' &>/dev/null; then
                break
            else
                echo ""
                bad "ERROR: You have entered invalid email address"
            fi
        done

        # Company
        getparam_with_default "Company" "${COMPANY}"
        COMPANY=$PARAM
        
        # Address
        while true; do
            getparam_with_default "Address" "${ADDRESS}"
            ADDRESS=$PARAM
            if [ "x${ADDRESS}" != "x" ]; then
                break;
            else
                echo ""
                bad "ERROR: Address cannot be empty"
            fi
        done

        # ZIP code
        while true; do
            getparam_with_default "ZIP code" "${ZIP}"
            ZIP=$PARAM
            if [ "x${ZIP}" != "x" ]; then
                break;
            else
                echo ""
                bad "ERROR: ZIP code cannot be empty"
            fi
        done
 
        # Country
        while true; do
            good "Choose your country:"
            display_menugrid 3 27 "${COUNTRY_CHOICE}"

            # Check, if country is valid
            if [ $PARAM -gt 0 ]; then
                COUNTRY_CHOICE="${PARAM}"
                PARAM=$((${PARAM} - 1))
                COUNTRY_NAME="${MENUGRID_ITEMS[${PARAM}]}"
                COUNTRY_CODE="${COUNTRY_CODES[${PARAM}]}"
                # OK, obtain country code
                break
            else
                echo ""
                bad "ERROR: You have entered invalid country code"
            fi
        done

        # Phone number
        while true; do
            getparam_with_default "Phone number" "${PHONE}"
            PHONE=$PARAM
            if [ "x${PHONE}" != "x" ]; then
                break;
            else
                echo ""
                bad "ERROR: Phone cannot be empty"
            fi
        done

        # Review info
        echo ""
        good "Please review entered information"
        good ""
        good "Name:         ${NAME}"
        good "Email:        ${EMAIL}"
        good "Company:      ${COMPANY}"
        good "Address:      ${ADDRESS}"
        good "ZIP code:     ${ZIP}"
        good "Country:      ${COUNTRY_NAME} [${COUNTRY_CODE}]"
        good "Phone number: ${PHONE}"
        good ""
        ask_with_confirmation "Do you want to edit entered data?" "N" "y"
        
        if [ $? -ne 2 ]; then 
            ask_with_confirmation "Do you want to register trial license now?" "Y" "n"

            if [ $? -eq 1 ]; then
                # perform registration
                REGISTRATION_STRING="$(echo "${NAME}" | tr ';' ' ');${EMAIL};$(echo "${COMPANY}" | tr ';' ' ');$(echo "${ADDRESS}" | tr ';' ' ');$(echo "${ZIP}" | tr ';' ' ');${COUNTRY_CODE};$(echo "${PHONE}" | tr ';' ' ')"
                run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "set" "system" "C_ObtainTrialLicense" "${REGISTRATION_STRING}"
                EXITCODE=$?
                if [ ${EXITCODE} -eq 0 ]; then
                    good "Trial license registered successfully"
                    return 0
                else
                    bad "Trial license registration problem, error code ${EXITCODE}!"
                    bad "Try again later or export reference ID and register at shop.icewarp.com."
                    bad "After registration, activate license on-line using wizard.sh"
                    return 1
                fi
            else
                break
            fi
        fi
    done
}

SCRIPT_NAME=`basename "$0"`
good ""
case $SCRIPT_NAME in
    "license_info")
        fnc_license_info
        ;;
    "license_display")
        fnc_license_display
        ;;
    "license_export")
        fnc_license_export
        ;;
    "license_online")
        fnc_license_online
        ;;
    "license_offline")
        fnc_license_offline
        ;;
    "license_refkey_display")
        fnc_license_refkey_display
        ;;
    "license_refkey_export")
        fnc_license_refkey_export
        ;;
    "license_obtain_trial")
        fnc_license_obtain_trial
        ;;
    *)
        echo "Script called for unknown operation: ${SCRIPT_NAME}"
        exit 1
        ;;
esac

exit 0
