#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#

source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"
source "${IWS_INSTALL_DIR}/scripts/platform"

restart_server()
{
    good "Restarting the server"
    "${IWS_INSTALL_DIR}/icewarpd.sh" --restart
    good "Server restart finished"
}

check_gcore()
{
    RETVAL=0
    
    gcore >/dev/null
    if [ $? -eq 127 ]; then
        RETVAL=1
        warn "Program gcore is probably missing on your computer."
        ask_with_confirmation "Do you want to install gcore?" "Y" "n"
        if [ $? -eq 1 ]; then
            platform_install_packages "1" "gdb"
            
            gcore >/dev/null
            if [ $? -ne 127 ]; then
                RETVAL=0
            fi
        else
            return false
        fi
    fi
    
    return $RETVAL
}

# param1: service name
get_pid()
{
    cat "${IWS_INSTALL_DIR}/var/$1.pid" 2>/dev/null
}

indent() {
    sed 's/^/          /g'
}

# param1: service name
# param2: dumpdir
# param3: dump index
dump_service()
{
    PID=$(get_pid "$1")
    if [ $? -eq 0 ]; then
        gcore -o "$2/core.$1.$PID.$3" $PID | indent
    else
        warn "Service $1 not running"
    fi
}

fnc_diagnostics() 
{
    good "Running server diagnostics."
    
    VAL=$(get_api_variable "system" "C_Accounts_Global_Accounts_DisableCache")
    if [ "${VAL}" == "1" ]; then
        echo ""
        warn "Account cache is disabled."
    fi
    
    echo ""
    good "Diagnostics finished, press ENTER to continue."
    read -s
}

fnc_crashdump_enable()
{
    set_api_variable system C_System_Adv_Exit_On_Signal 1
    ulimit -c unlimited
    
    good "Server restart is needed to start generating crash dumps." 
    ask_with_confirmation "Restart now?" "Y" "n"
    if [ $? -ne 1 ]; then
        return 0
    fi

    restart_server
}

fnc_crashdump_disable()
{
    set_api_variable system C_System_Adv_Exit_On_Signal 0
    ulimit -c 0
    
    good "Server restart is needed to stop generating crash dumps." 
    ask_with_confirmation "Restart now?" "Y" "n"
    if [ $? -ne 1 ]; then
        return 0
    fi

    restart_server
}

fnc_manualdump()
{
    DUMP_CONTROL=false
    DUMP_GW=false
    DUMP_IM=false
    DUMP_POP3=false
    DUMP_SMTP=false
    DUMP_PHP=false
    
    if ! check_gcore; then
        bad "Cannot dump without gcore"
        bad "Press ENTER to continue"
        read -s
        return 0
    fi
    
    DISKFREE=`df -k --output=avail "${IWS_INSTALL_DIR}" | tail -n1` 
    
    if [[ $DISKFREE -lt 1048576 ]]; then          
      good "You seem to have less than 1 GB of free disk space and dump can fail." 
      ask_with_confirmation "Continue?" "Y" "n"
      
      if [ $? -ne 1 ]; then
        return 0
      fi
    fi
    
    MENUGRID_ITEMS=("Control" "GroupWare" "Instant messaging" "POP3/IMAP" "SMTP" "All IceWarp services" "PHP workers" "All IceWarp services and PHP workers")
    good "Choose service to dump:"
    display_menugrid 1 1 0
    case $PARAM in
        1)
            DUMP_CONTROL=true
            ;;
        2)
            DUMP_GW=true
            ;;
        3)
            DUMP_IM=true
            ;;
        4)
            DUMP_POP3=true
            ;;
        5)
            DUMP_SMTP=true
            ;;
        6)
            DUMP_CONTROL=true
            DUMP_GW=true
            DUMP_IM=true
            DUMP_POP3=true
            DUMP_SMTP=true
            ;;
        7)
            DUMP_PHP=true
            ;;
        8)
            DUMP_CONTROL=true
            DUMP_GW=true
            DUMP_IM=true
            DUMP_POP3=true
            DUMP_SMTP=true
            DUMP_PHP=true
            ;;
        *)
            return 0
            ;;
    esac
    
    getparam_with_default "Number of consecutive dumps to take" "1"
    DUMPCOUNT=${PARAM}
    if ! grep -q '[0-9]\+' <<< ${DUMPCOUNT}; then
        bad "Dump count not a number"
        bad "Press ENTER to continue"
        read -s
        return 0
    fi
    
    DUMPDIR="${IWS_INSTALL_DIR}/debug/dumps_$(date "+%Y-%m-%d-%H-%M-%S")"
    getparam_with_default "Directory where to store the dumps" "${DUMPDIR}"
    DUMPDIR="${PARAM}"
    mkdir -p "${DUMPDIR}"
    
    for I in $(seq 1 ${DUMPCOUNT}); do
        good "Dumping iteration #$I"
        if $DUMP_CONTROL; then
            dump_service "control" "${DUMPDIR}" $I
        fi
        if $DUMP_GW; then
            dump_service "cal" "${DUMPDIR}" $I
        fi
        if $DUMP_IM; then
            dump_service "im" "${DUMPDIR}" $I
        fi
        if $DUMP_POP3; then
            dump_service "pop3" "${DUMPDIR}" $I
        fi
        if $DUMP_SMTP; then
            dump_service "smtp" "${DUMPDIR}" $I
        fi
        
        if $DUMP_PHP; then
            for pid in $(pgrep php-fpm); do 
                gcore -o "${DUMPDIR}/core.php.$pid.$I" $pid | indent
            done
        fi
        
        if [ ${I} -lt ${DUMPCOUNT} ]; then        
            sleep 5
        fi
    done     
    
    good "Dumping done"
    good "Press ENTER to continue"
    read -s
}

fnc_pack()
{
    DISKFREE=`df -k --output=avail "${IWS_INSTALL_DIR}" | tail -n1` 
      
    if [[ $DISKFREE -lt 1048576 ]]; then          
      good "You seem to have less than 1 GB of free disk space and dump can fail." 
      ask_with_confirmation "Continue?" "Y" "n"
      
      if [ $? -ne 1 ]; then
        return 0
      fi
    fi

    PACKNAME="iwdebuginfo_$(date "+%Y-%m-%d-%H-%M-%S")"
    PACKDIR="${IWS_INSTALL_DIR}/debug/${PACKNAME}"
    rm -rf "${PACKDIR}"
    mkdir -p "${PACKDIR}"
    good "Creating debuginfo package at ${PACKDIR}"
    sleep 1

    if ls "${IWS_INSTALL_DIR}/core"* >/dev/null 2>&1; then
        good "Some core dumps were found in IceWarp server installation directory."
        ask_with_confirmation "Do you want to collect them?" "Y" "n"
        if [ $? -eq 1 ]; then
            cd ${IWS_INSTALL_DIR} && tar zcfv "${PACKDIR}/crashdumps.tar.gz" core* | indent
            ask_with_confirmation "Delete dumps from the original location?" "Y" "n"
            if [ $? -eq 1 ]; then
                rm -rf "${IWS_INSTALL_DIR}/core"*
            fi
        fi
    fi
    
    
    if ls "${IWS_INSTALL_DIR}/debug/dumps_"* >/dev/null 2>&1; then        
        good "Packing dumps from ${IWS_INSTALL_DIR}/debug/dumps_*" directories
        sleep 1
        cd "${IWS_INSTALL_DIR}"/debug && tar zcfv "${PACKDIR}/dumps.tar.gz" dumps_* | indent
    fi
 
    good "Collecting IceWarp server binaries and libraries"
    sleep 1
    
    if [ -d "${IWS_INSTALL_DIR}/lib64" ];
      then
        cd "${IWS_INSTALL_DIR}" && tar zcfv "${PACKDIR}/iwbin.tar.gz" control cal im pop3 smtp icewarpd lib lib64 php/php-fpm php/ext | indent
      else
        cd "${IWS_INSTALL_DIR}" && tar zcfv "${PACKDIR}/iwbin.tar.gz" control cal im pop3 smtp icewarpd lib php/php-fpm php/ext | indent
    fi
        
    good "Collecting system libraries"
    sleep 1
    ldd "${IWS_INSTALL_DIR}/control" | grep -v "not found" | sed 's/.*=> //' | sed 's/ .*$//' | sed 's/[[:blank:]]//' | grep ^/ | xargs realpath | tar zcfv "${PACKDIR}/syslibs.tar.gz" -T - 2>/dev/null | indent
    
    good "Collecting logs from last three days"
    sleep 1
    LOGPATH=$(get_api_variable system C_System_Storage_Dir_LogPath)
    cd "${LOGPATH}" && find ./ -type f -mtime -3 | tar zcfv "${PACKDIR}/logs.tar.gz" -T - 2>/dev/null | indent
    
    good "Collecting system info"
    sleep 1
    mkdir -p "${PACKDIR}/sysinfo"
    cp -af /etc/icewarp/icewarp.conf "${PACKDIR}/sysinfo"
    cp -af /etc/redhat-release "${PACKDIR}/sysinfo" 2>/dev/null
    cp -af /etc/debian_version "${PACKDIR}/sysinfo" 2>/dev/null
    uname -a > "${PACKDIR}/sysinfo/uname"
    free -mh > "${PACKDIR}/sysinfo/mem"
    df -h > "${PACKDIR}/sysinfo/disk"
    df -ih >> "${PACKDIR}/sysinfo/disk"
    cd "${PACKDIR}/sysinfo" && tar zcfv "${PACKDIR}/sysinfo.tar.gz" * | indent
    rm -rf "${PACKDIR}/sysinfo"
    
    good "Creating one package for all"
    sleep 1
    cd "${IWS_INSTALL_DIR}/debug" && tar cfv "${PACKNAME}.tar" "${PACKNAME}" | indent
    rm -rf "${PACKDIR}"
    
    good "Package created at ${IWS_INSTALL_DIR}/debug/${PACKNAME}.tar"
    ask_with_confirmation "Move package to ${IWS_INSTALL_DIR}/html to allow easy download?" "Y" "n"
    if [ $? -eq 1 ]; then
        mv "${IWS_INSTALL_DIR}/debug/${PACKNAME}.tar" "${IWS_INSTALL_DIR}/html/${PACKNAME}.tar"
        good "Package moved"
    fi
}

fnc_pack_libs()
{
    DISKFREE=`df -k --output=avail "${IWS_INSTALL_DIR}" | tail -n1` 
      
    if [[ $DISKFREE -lt 1048576 ]]; then          
      good "You seem to have less than 1 GB of free disk space and dump can fail." 
      ask_with_confirmation "Continue?" "Y" "n"
      
      if [ $? -ne 1 ]; then
        return 0
      fi
    fi

    PACKNAME="iwdebuginfo_$(date "+%Y-%m-%d-%H-%M-%S")"
    PACKDIR="${IWS_INSTALL_DIR}/debug/${PACKNAME}"
    rm -rf "${PACKDIR}"
    mkdir -p "${PACKDIR}"
    good "Creating debuginfo package at ${PACKDIR}"
    sleep 1
        
    good "Collecting system libraries"
    sleep 1
    ldd "${IWS_INSTALL_DIR}/control" | grep -v "not found" | sed 's/.*=> //' | sed 's/ .*$//' | sed 's/[[:blank:]]//' | grep ^/ | xargs realpath | tar zcfv "${PACKDIR}/syslibs.tar.gz" -T - 2>/dev/null | indent
    
    cd "${IWS_INSTALL_DIR}/debug" && tar cfv "${PACKNAME}.tar" "${PACKNAME}" | indent
    rm -rf "${PACKDIR}"
    
    good "Package created at ${IWS_INSTALL_DIR}/debug/${PACKNAME}.tar"
    ask_with_confirmation "Move package to ${IWS_INSTALL_DIR}/html to allow easy download?" "Y" "n"
    if [ $? -eq 1 ]; then
        mv "${IWS_INSTALL_DIR}/debug/${PACKNAME}.tar" "${IWS_INSTALL_DIR}/html/${PACKNAME}.tar"
        good "Package moved"
    fi
}

fnc_cleanup()
{
    ask_with_confirmation "Delete ${IWS_INSTALL_DIR}/debug/dumps_* and ${IWS_INSTALL_DIR}/debug/iwdebuginfo_*?" "Y" "n"
    if [ $? -eq 1 ]; then
        good "Deleting..."
        rm -rf "${IWS_INSTALL_DIR}"/debug/dumps_*
        rm -rf "${IWS_INSTALL_DIR}"/debug/iwdebuginfo_*
        good "Deleted"
    fi
}

SCRIPT_NAME=`basename $0`
RES=0
case $SCRIPT_NAME in
    "debugging_tools_diagnostics")
        fnc_diagnostics
        RES=$?
        ;;
    "debugging_tools_crashdump_enable")
        fnc_crashdump_enable
        RES=$?
        ;;
    "debugging_tools_crashdump_disable")
        fnc_crashdump_disable
        RES=$?
        ;;
    "debugging_tools_manualdump")
        fnc_manualdump
        RES=$?
        ;;
    "debugging_tools_pack")
        fnc_pack
        RES=$?
        ;;
    "debugging_tools_pack_libs")
        fnc_pack_libs
        RES=$?
        ;;
    "debugging_tools_cleanup")
        fnc_cleanup
        RES=$?
        ;;
    *)
        echo "Script called for unknown operation: ${SCRIPT_NAME}"
        RES=1
        ;;
esac

exit $RES
