#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: db_mysql.sh - wizard menu script
#

source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"

# Init variables
DB_HOST=""
DB_USER=""
DB_PASSWD=""
DB_NAME=""
ODBC_CONN_STRING=""

# Funtion asks user for ODBC connection parameters
# Sets: DB_HOST, DB_USER, DB_PASSWD, DB_NAME with user entered informations
#       ODBC_CONN_STRING to constructed ODBC connection string
# Returns: != 0 in case of error
#          0 if OK
#
# ODBC connection string
# format is:
# DB_NAME;DB_USER;DB_PASSWD;DB_HOST;syntax;driver
# - DB_NAME, DB_USER, DB_PASSWD, DB_HOST are strings
# - syntax is:
#     0 - default
#     1 - MS Access
#     2 - MS SQL
#     3 - MySQL
#     4 - InterBase
#     5 - Oracle
#     6 - FireBird
#     7 - SQLite
#  - driver is:
#     0 - default
#     1 - ODBC
#     2 - MySQL
#     3 - SQLite
#     4 - FireBird
fnc_get_odbc_conn_string()
{
    # try to get local MySQL connection socket
    CFG_FILES="/etc/mysql/my.cnf /etc/my.cnf ~/my.cnf"
    for CFG_FILE in $CFG_FILES
    do
        if [ -f "$CFG_FILE" ]; then
            DB_HOST=$(grep "^socket" $CFG_FILE 2> /dev/null | cut -d'=' -f2 | cut -d' ' -f2 | tail -n 1)
            if [ -n "$DB_HOST" ]; then
            break
            fi
        fi
    done

    good ""
    good "Please enter MySQL connection parameters:"

    if [ "x${DB_HOST}" != "x" ]; then
        getparam "Database server host (or path to socket) [${DB_HOST}]:"
    else
        getparam "Database server host (or path to socket):"
    fi
    if [ "x${PARAM}" != "x" ]; then
        DB_HOST=$PARAM
    fi

    getparam    "    Login name:"
    DB_USER=$PARAM
    getpassword "Login password:"
    DB_PASSWD=$PASSWORD_PARAM
    getparam    " Database name:"
    DB_NAME=$PARAM

    if [ "x${DB_HOST}" == "x" ]; then
        warn "Missing server host name"
        return 1
    fi
    if [ "x${DB_USER}" == "x" ]; then
        warn "Missing login name"
        return 1
    fi
    if [ "x${DB_PASSWD}" == "x" ]; then
        warn "Missing login password"
        return 1
    fi
    if [ "x${DB_NAME}" == "x" ]; then
        warn "Missing database password"
        return 1
    fi

    # convert host to lowercase
    DB_HOST=$(echo "$DB_HOST" | tr "A-Z" "a-z")

    # replace "localhost" string
    if [ "x${DB_HOST}" == "xlocalhost" ]; then
        DB_HOST="127.0.0.1"
    fi

    # construct ODBC string, use MySQL syntax and driver
    ODBC_CONN_STRING="${DB_NAME};${DB_USER};${DB_PASSWD};${DB_HOST};3;2"
    return 0
}

# Common function does:
# - asks for ODBC connection string
# - switches storage mode for selected set: accounts, groupware, antispam
# - creates tables in database
# Parameters:
# $1 - number for tool command "create tables":
#      0 - accounts
#      2 - groupware
#      3 - antispam
# $2 - string with storage type: "accounts", "groupware", "antispam"
#      used in informative prints
#
# Storage type
# 0 - Professional file system
# 1 - Standard file system
# 2 - ODBC
fnc_db_mysql_common()
{
    SERVICE_TYPE="$1"
    SERVICE_NAME="$2"

    good ""
    ask_with_confirmation "Setup ${SERVICE_NAME} storage to MySQL?" "Y" "n"
    if [ $? -ne 1 ]; then
        warn "Setup ${SERVICE_NAME} storage cancelled."
        return 0
    fi

    fnc_get_odbc_conn_string
    if [ $? -ne 0 ]; then
        return 1
    fi

    good "Setting database connection..."
    # Accounts    
    if [ "$SERVICE_TYPE" -eq "0" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_System_Storage_Accounts_ODBCConnString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi
    # GroupWare
    if [ "$SERVICE_TYPE" -eq "2" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_GW_ConnectionString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi
    # Antispam
    if [ "$SERVICE_TYPE" -eq "3" ]; then
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_AS_Challenge_ConnectionString" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
    fi

    good ""
    ask_with_confirmation "Create tables for ${SERVICE_NAME}?" "Y" "n"
    if [ $? -eq 1 ]; then
        good "Creating tables..."
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "create" "tables" "${SERVICE_TYPE}" "${ODBC_CONN_STRING}"
        EXITCODE=$?
        if [ $EXITCODE -ne 0 ]; then
            if [ "${SERVICE_NAME}" == "GroupWare" -a $EXITCODE -eq 3 ]; then
                bad "Error calling tool - GW service not running or other problem occured."
            else
                bad "Error calling tool, error code ${EXITCODE}"
            fi
            return 1
        fi
    else
        good "Tables were not created."
        # continue, because user can have tables already created
    fi

    # Switch storage mode for Accounts
    # GW and Antispam has only DB storage mode
    if [ "$SERVICE_TYPE" -eq "0" ]; then
        good "Switching ${SERVICE_NAME} storage mode..."
        run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "modify" "system" "C_System_Storage_Accounts_StorageMode" "2"
        EXITCODE=$?
        if [ ${EXITCODE} -ne 0 ]; then
            bad "Error calling tool, error code ${EXITCODE}"
            return 1
        fi
        good "Storage switched to MySQL."
    fi

    return 0
}

fnc_db_mysql_accounts()
{
    fnc_db_mysql_common "0" "Accounts"
    if [ $? -ne 0 ]; then
        return 1
    fi

    good ""
    ask_with_confirmation "Convert data from file system to database?" "Y" "n"
    if [ $? -ne 1 ]; then
        return 0
    fi

    good "Converting data from file system to database..."
    run_cmd_log "${IWS_INSTALL_DIR}/tool.sh" "set" "storage" "1"
    EXITCODE=$?
    if [ ${EXITCODE} -ne 0 ]; then
        bad "Error calling tool, error code ${EXITCODE}"
        return 1
    fi
    good "Data converted."

    return 0
}

fnc_db_mysql_groupware()
{
    fnc_db_mysql_common "2" "GroupWare"
    if [ $? -ne 0 ]; then
        return 1
    fi
    return 0
}

fnc_db_mysql_antispam()
{
    fnc_db_mysql_common "3" "Antispam"
    if [ $? -ne 0 ]; then
        return 1
    fi
    return 0
}

SCRIPT_NAME=`basename "$0"`
RES=0

good ""
good "This action requires to have running MySQL server with"
good "created database with required privileges."
ask_with_confirmation "Continue?" "Y" "n"
if [ $? -ne 1 ]; then
    exit 0
fi

case $SCRIPT_NAME in
    "db_mysql_accounts")
        fnc_db_mysql_accounts
        RES=$?
        ;;
    "db_mysql_groupware")
        fnc_db_mysql_groupware
        RES=$?
        ;;
    "db_mysql_antispam")
        fnc_db_mysql_antispam
        RES=$?
        ;;
    *)
        echo "Script called for unknown operation: ${SCRIPT_NAME}"
        RES=1
        ;;
esac

exit $RES
