#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: icewarpd.sh - icewarpd script
#

# Configurable timeouts, in seconds
SERVICE_START_TIMEOUT=10 # Timeout for service start
SERVICE_STOP_TIMEOUT=10  # Timeout between SIGTERM and SIGKILL when stopping services
CHILD_STOP_TIMEOUT=3     # Timeout between SIGTERM and SIGKILL when stopping unterminated childs of services

SERVER_VERSION="IceWarp Server %%SERVER_VERSION_VALUE%%"

# include configuration file
source /etc/icewarp/icewarp.conf
if [ $? -ne 0 ]; then
    echo "Can not find configuration file icewarp.conf"
    exit 255
fi

# Set to icewarpd logfile basename to enable icewarpd logging
#export IWD_LOGFILE="${IWS_INSTALL_DIR}/logs/icewarpd.log"

# Ensure some locale is set
if [ "x${LANG}" == "x" ]; then
    export LANG="${IWS_DEFAULT_LOCALE}"
fi

# export environment variables for IceWarp Server
export IWS_INSTALL_DIR
export IWS_PROCESS_USER
export KL_PLUGINS_PATH="${IWS_INSTALL_DIR}/kaspersky/ppl"
export KL_UPD_PLUGINS_PATH="${IWS_INSTALL_DIR}/kaspersky/updater"
export LD_LIBRARY_PATH="${IWS_INSTALL_DIR}/${IWS_LIB_DIR}:${IWS_INSTALL_DIR}/lib:${IWS_INSTALL_DIR}/php/lib:${IWS_INSTALL_DIR}/kaspersky:${IWS_INSTALL_DIR}/kaspersky/${IWS_LIB_DIR}:${IWS_INSTALL_DIR}/kaspersky/lib:${IWS_INSTALL_DIR}/kaspersky/updater:${IWS_INSTALL_DIR}/kaspersky/ppl:${IWS_INSTALL_DIR}/avast:${IWS_INSTALL_DIR}/purple/lib:${IWS_INSTALL_DIR}/eset/lib:${LD_LIBRARY_PATH}"
export LDAPCONF="${IWS_INSTALL_DIR}/ldap/etc/ldap.conf"

# include functions
source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"
if [ $? -ne 0 ]; then
    echo "Can not find functions include file"
    exit 255
fi

usage()
{
    echo ""
    echo "$SERVER_VERSION"
    echo ""
    echo "Usage:"
    echo "    icewarpd.sh Command [Service]"
    echo ""
    echo "Command:"
    echo "    --start        Start specified Service or all services."
    echo "    --stop         Stop specified Service or all services."
    echo "    --restart      Restart specified Service or all services."
    echo "    --check        Test if specified Service is running."
    echo "                   Returned status code is 0 if Service is running,"
    echo "                   non-zero if service is not running."
    echo "    --command      Issue IceWarp daemon API command"
    echo ""
    echo "Service can be one of: control, gw, im, pop3, smtp, all"
    echo "Command --check does not accept 'all' as a service name."
    echo ""
}

get_api_variable()
{
    CMD_STDOUT=$("${IWS_INSTALL_DIR}/tool.sh" get "$1" "$2" | sed 's/^[^:]*: //')
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

get_logfile_path()
{
    IW_LOGPATH="$(get_api_variable system C_System_Storage_Dir_LogPath)/icewarpd/"
    LOG_FN="$(date +"%Y%m%d").log"
    mkdir -p "$IW_LOGPATH"
    LOGFILE="${IW_LOGPATH}${LOG_FN}"
    echo "$LOGFILE"
}

# Params
# 1 - action (start, stop)
# 2 - name of service
log()
{
    LOGFILE="$(get_logfile_path)"
    
    date >> "$LOGFILE"
    echo "$1 of $2" >> "$LOGFILE"
    echo "Current user: $(whoami)" >> "$LOGFILE"
    pstree -s $$ >> "$LOGFILE"
    echo "" >> "$LOGFILE"
}

ICEWARPD_PID=0
ICEWARPD_PID_DIR=${IWS_INSTALL_DIR}/var

SERVICE=$2
if [ "x$SERVICE" == "x" ]; then
    SERVICE=all
fi

IWDAPI_command()
{
    echo -n "Issuing icewarpd API command..."
    TEXTRES=$("${IWS_INSTALL_DIR}/icewarpd" --command $*)
    RET=$?
    if [ $RET -eq 0 ]; then
        echo "OK"
        echo "Returned value: ${TEXTRES}"
    else
        echo "ERROR"
    fi
    return $RET;
}

wait_running()
{
    COUNTER=${SERVICE_START_TIMEOUT}
    while [ $COUNTER -gt 0 ]; do
        "${IWS_INSTALL_DIR}/icewarpd" --check "$1" &> /dev/null
        if [ $? -eq 0 ]; then
            return 0
        fi
        sleep 1
        let COUNTER=COUNTER-1
    done
    return 1
}

wait_not_running()
{
    # add 5 seconds to be sure
    let COUNTER=${SERVICE_STOP_TIMEOUT}+${CHILD_STOP_TIMEOUT}+5
    while [ $COUNTER -gt 0 ]; do
        "${IWS_INSTALL_DIR}/icewarpd" --check "$1" &> /dev/null
        if [ $? -ne 0 ]; then
            return 0
        fi
        sleep 1
        let COUNTER=COUNTER-1
    done
    return 1
}

check_service()
{
    "${IWS_INSTALL_DIR}/icewarpd" --check "$1" &> /dev/null
    if [ $? -eq 0 ]; then
      echo "Service $1 is running"
    else
      echo "Service $1 is not running"
      return 255
    fi
    return 0
}

start_service()
{
    log "start" "$1"
    case $1 in
        control)
            echo -n "Starting control ... "
            touch "${ICEWARPD_PID_DIR}/control.start" &> /dev/null
            ;;
        gw)
            echo -n "Starting groupware ... "
            touch "${ICEWARPD_PID_DIR}/cal.start" &> /dev/null
            ;;
        im)
            echo -n "Starting im ... "
            touch "${ICEWARPD_PID_DIR}/im.start" &> /dev/null
            ;;
        pop3)
            echo -n "Starting pop3 ... "
            touch "${ICEWARPD_PID_DIR}/pop3.start" &> /dev/null
            ;;
        smtp)
            echo -n "Starting smtp ... "
            touch "${ICEWARPD_PID_DIR}/smtp.start" &> /dev/null
            ;;
        hg)
            echo -n "Starting Holy Grail ... "
            touch "${ICEWARPD_PID_DIR}/icewarp.start" &> /dev/null
            ;;
        all)
            start_service control
            start_service gw
            start_service im
            start_service pop3
            start_service smtp
            return 0
            ;;
        *)
            echo "Unknown service: $1"
            return 1
            ;;
    esac

    LOGDIR=""
    if ! [ -f "${IWS_INSTALL_DIR}/var/icewarpd.pid" ] || ! kill -0 "$(cat "${IWS_INSTALL_DIR}/var/icewarpd.pid")" &>/dev/null; then
        LOGDIR=$(get_api_variable system c_system_storage_dir_logpath)
        LOGDIR="--logdir ${LOGDIR}"
    fi
    
    ICEWARPD_PID=$("${IWS_INSTALL_DIR}/icewarpd" ${LOGDIR})
    kill -SIGUSR1 "$ICEWARPD_PID" &> /dev/null
    wait_running $1
    if [ $? -eq 0 ]; then
        echo "OK"
    else
        "${IWS_INSTALL_DIR}/icewarpd" "--command" "logstderr" "$(date): Warning - waiting for start of $1 service timed out"
        echo "ERROR"
    fi
}

stop_icewarpd()
{
    echo -n "Notifying icewarpd ... "
    COUNTER=${SERVICE_STOP_TIMEOUT}
    PID=$("${IWS_INSTALL_DIR}/icewarpd")
    kill -SIGINT $PID 2> /dev/null
    while [ $COUNTER -gt 0 ]; do
        kill -0 $PID &> /dev/null
        if [ $? -ne 0 ]; then
            break
        fi
        let COUNTER=COUNTER-1
        sleep 1
    done
    echo "OK"
}

stop_service()
{
    log "stop" "$1"
    
    if [ "$1" != "all" ] && [ "$2" != "all" ]; then
        if ! IS_LAST_SERVICE=$("${IWS_INSTALL_DIR}/icewarpd" --command IsLastRunningService "$1"); then
            echo "Cannot stat service count"
            return 1
        fi 
    fi

    if [ "$2" != "all" ]; then
        "${IWS_INSTALL_DIR}/icewarpd" --command SetServicesStopTimeout ${SERVICE_STOP_TIMEOUT}>/dev/null
        "${IWS_INSTALL_DIR}/icewarpd" --command SetChildsStopTimeout ${CHILD_STOP_TIMEOUT}>/dev/null
    fi
    
    if [ "$1" == "gw" ]; then
        CTLFILE="cal"
    elif [ "$1" == "hg" ]; then
        CTLFILE="icewarp"
    else
        CTLFILE="$1"
    fi

    case $1 in
        control)
            echo -n "Stopping control ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
            ;;
        gw)
            echo -n "Stopping groupware ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
            ;;
        im)
            echo -n "Stopping im ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
            ;;
        pop3)
            echo -n "Stopping pop3 ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
            ;;
        smtp)
            echo -n "Stopping smtp ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
            ;;
        hg)
            echo -n "Stoppping Holy Grail ... "
            touch "${ICEWARPD_PID_DIR}/${CTLFILE}.stop"
	    ;;
        all)
            stop_service control all
            stop_service gw all
            stop_service im all
            stop_service pop3 all
            stop_service smtp all
            stop_icewarpd
            
            kill_proc_by_name "${IWS_INSTALL_DIR}/kaspersky/kavehost" "${IWS_INSTALL_DIR}/kaspersky"
            kill_proc_by_name "kavscanner" "${IWS_INSTALL_DIR}/kaspersky"
            
            return 0
            ;;
        *)
            echo "Unknown service: $1"
            return 1
            ;;
    esac

    ICEWARPD_PID=$("${IWS_INSTALL_DIR}/icewarpd")
    kill -SIGUSR2 "$ICEWARPD_PID" &> /dev/null
    wait_not_running $1
    if [ $? -eq 0 ]; then
        if [ -f "${ICEWARPD_PID_DIR}/${CTLFILE}.kill" ]; then
            echo "KILLED"
            rm -f "${ICEWARPD_PID_DIR}/${CTLFILE}.kill"
        else
            echo "OK"
        fi
    else
        echo "ERROR"
    fi

    if [ "${IS_LAST_SERVICE}" == "1" ]; then
        stop_icewarpd
    fi
}

RET=0
case "$1" in
    --start)
        start_service $SERVICE
        ;;
    --stop)
        stop_service $SERVICE
        ;;
    --restart)
        stop_service $SERVICE
        start_service $SERVICE
        ;;
    --check)
        check_service $SERVICE
        RET=$?
        ;;
    --command)
        shift
        IWDAPI_command $*
        RET=$?
        ;;
    *)
        usage
        ;;
esac

exit $RET
