#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: phpd.sh - server internal script
#

# timeout between sigterm and sigkill when terminating previous php instance in seconds
PHP_KILL_TIMEOUT=1

# maximum time needed for completing this script in seconds
PROCESSING_TIME=20

# function for parsing command line arguments
parse_cmdline()
{
    export PHP_FCGI_THREADPOOL=15
    export PHP_FCGI_BINDIP=""
    export PHP_FCGI_MAXROUNDS=0
    export PHP_FCGI_RESTART500=0

    for i in $*; do
        case `echo ${i} | tr a-z A-Z` in
            WEBAPPMAXTHREADS=*)
                export PHP_FCGI_THREADPOOL=`echo ${i} | sed 's/[-a-zA-Z0-9]*=//'`
                ;;
            BINDIP=*)
                export PHP_FCGI_BINDIP=`echo ${i} | sed 's/[-a-zA-Z0-9]*=//'`
                ;;
            MAXROUNDS=*)
                export PHP_FCGI_MAXROUNDS=`echo ${i} | sed 's/[-a-zA-Z0-9]*=//'`
                ;;
            RESTART500=*)
                export PHP_FCGI_RESTART500=`echo ${i} | sed 's/[-a-zA-Z0-9]*=//'`
                ;;
        esac
    done

    # Check threadpool low limit
    if [ $PHP_FCGI_THREADPOOL -le 0 ]; then
        PHP_FCGI_THREADPOOL=15
    fi
}

# function for terminating process with given PID
# $1: pid
# $2: timeout (s)
terminate_process()
{
    PID=$1
    TIMEOUT=$2
    
    kill ${PID} &> /dev/null
    sleep ${TIMEOUT}
    kill -0 ${PID} &> /dev/null
    if [ $? -eq 0 ]; then
        kill -SIGKILL ${PID} &> /dev/null
        sleep ${TIMEOUT}
    fi
}

# function terminating php, if exists
terminate_php()
{
    if [ -f "${PHPD_PID_FILE}" ]; then
        terminate_process $(cat "${PHPD_PID_FILE}" 2>/dev/null) ${PHP_KILL_TIMEOUT}
        rm -f "${PHPD_PID_FILE}"
    fi
    rm -f "${PHP_SOCKET_FILE}"
    
    # kill our remaining php-fpm master processes
    for PID in $(pidof "php-fpm: master process (${IWS_INSTALL_DIR}/php/php-fpm.conf)" | sed 's/ /\n/g')
    do
        terminate_process ${PID} ${PHP_KILL_TIMEOUT}
    done
}

# function called to ensure, that only single instance of this script is run
function ensure_single_instance()
{
    set -o noclobber
    if ! echo $$>"${PHPD_LOCK_FILE}"; then
        age=$(($(date +%s)-$(stat -c %Y ${PHPD_LOCK_FILE})))
        if [ ${age} -ge ${PROCESSING_TIME} ]; then
            if rm "${PHPD_LOCK_FILE}" 2>/dev/null; then
                if ! echo $$>"${PHPD_LOCK_FILE}"; then
                    exit 3
                fi
            else
                exit 2
            fi
        else
            exit 1
        fi
    fi
    set +o noclobber
}

# function have to be called at the end of the script
function done_single_instance()
{
  rm -f "${PHPD_LOCK_FILE}"
}

# include configuration file
source /etc/icewarp/icewarp.conf
if [ $? -ne 0 ]; then
    echo "Can not find configuration file icewarp.conf"
    exit 255
fi

# export environment variables for IceWarp Server
export IWS_INSTALL_DIR
export IWS_PROCESS_USER
export KL_PLUGINS_PATH="${IWS_INSTALL_DIR}/kaspersky/ppl"
export KL_UPD_PLUGINS_PATH="${IWS_INSTALL_DIR}/kaspersky/updater"
export LD_LIBRARY_PATH="${IWS_INSTALL_DIR}/php/lib:${IWS_INSTALL_DIR}/${IWS_LIB_DIR}:${IWS_INSTALL_DIR}/lib:${IWS_INSTALL_DIR}/kaspersky:${IWS_INSTALL_DIR}/kaspersky/${IWS_LIB_DIR}:${IWS_INSTALL_DIR}/kaspersky/lib:${IWS_INSTALL_DIR}/kaspersky/updater:${IWS_INSTALL_DIR}/kaspersky/ppl:${IWS_INSTALL_DIR}/avast:${IWS_INSTALL_DIR}/purple/lib:${IWS_INSTALL_DIR}/eset/lib:${LD_LIBRARY_PATH}"

# include functions
source "${IWS_INSTALL_DIR}/scripts/inc/functions.sh"
if [ $? -ne 0 ]; then
    echo "Can not find functions include file"
    exit 255
fi

PHPD_PID_FILE="${IWS_INSTALL_DIR}/var/phpd.pid"
PHP_SOCKET_FILE="${IWS_INSTALL_DIR}/var/php.socket"
PHPD_LOCK_FILE="${IWS_INSTALL_DIR}/var/phpd.lock"

# ensure this script is running only once
ensure_single_instance

# parse command line arguments
parse_cmdline $*

# terminate previous php-fpm
terminate_php

# obtain log directory
IWS_LOGDIR=$(get_api_variable system c_system_storage_dir_logpath)
export IWS_LOGDIR

# ensure, that log directory exists
mkdir -p "${IWS_LOGDIR}/php-fpm"

# start PHP
"${IWS_INSTALL_DIR}/php/php-fpm" -y "${IWS_INSTALL_DIR}/php/php-fpm.conf" -R -c "${IWS_INSTALL_DIR}/php/php.ini" -p "${IWS_INSTALL_DIR}/php" 2> "${IWS_LOGDIR}/phpstartup.log"


# script done
done_single_instance

exit 0
