#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: install_hunspell_dict.sh - installer for hunspell dictionary
#

# include configuration file
source /etc/icewarp/icewarp.conf
if [ $? -ne 0 ]; then
    echo "Can not find configuration file icewarp.conf"
    exit 255
fi

echo "This script installs OpenOffice.org hunspell dictionary to IceWarp"

# command line parameters test
if [ $# -ne 2 ]; then
    echo "Incorrect count of command line parameters"
    echo ""
    echo "Usage:"
    echo "${0} oxt_file language_code"
    echo "oxt_file      - OpenOffice.org extension file with dictionary"
    echo "language_code - identifier of language and dialect, like en_US, case sensitive"
    echo ""
    echo "Dictionaries can be currently downloaded from http://extensions.services.openoffice.org/dictionary"
    echo ""
    exit 1
fi

HUNSPELL_DICT_FILE_NAME=${1}
HUNSPELL_LANGUAGE_CODE=${2}
HUNSPELL_DICT_BASE_FILE_NAME=$(basename ${1})

FILE_EXT_REGEX='\.oxt$'

# test if $1 has extension .oxt
if [[ ! "$HUNSPELL_DICT_BASE_FILE_NAME" =~ $FILE_EXT_REGEX ]]; then
    echo "File name is not OpenOffice.org extension (oxt)"
    exit 1
fi

# Check if unzip is available
if ! unzip >/dev/null 2>&1; then
    echo "Program unzip is needed, but not found"
    exit 1
fi

# create temp directory
TEMP_DIR=$(mktemp -dt)
if [ $? -ne 0 ]; then
    echo "Cannot create temporary directory"
    exit 1
fi

# extract archive into temporary directory
unzip "$HUNSPELL_DICT_FILE_NAME" -d "$TEMP_DIR"
if [ $? -ne 0 ]; then
    echo "Cannot extract dictionary package"
    rm -rf "$TEMP_DIR"
    exit 1
fi

# change to extracted dictionary directory
cd "$TEMP_DIR"
if [ $? -ne 0 ]; then
    echo "Can not change to directory: ${TEMP_DIR}"
    rm -rf "$TEMP_DIR"
    exit 1
fi

# Copy dictionary
mkdir -p "${IWS_INSTALL_DIR}/enchant/myspell"
cp "${TEMP_DIR}/${HUNSPELL_LANGUAGE_CODE}.aff" "${IWS_INSTALL_DIR}/enchant/myspell"
if [ $? -ne 0 ]; then
    echo "Cannot find file ${HUNSPELL_LANGUAGE_CODE}.aff in dictionary"
    rm -rf "$TEMP_DIR"
    exit 2
fi

cp "${TEMP_DIR}/${HUNSPELL_LANGUAGE_CODE}.dic" "${IWS_INSTALL_DIR}/enchant/myspell"
if [ $? -ne 0 ]; then
    rm -f "${IWS_INSTALL_DIR}/enchant/myspell/${HUNSPELL_LANGUAGE_CODE}.aff"
    echo "Cannot find file ${HUNSPELL_LANGUAGE_CODE}.aff in dictionary"
    rm -rf "$TEMP_DIR"
    exit 2
fi

# delete temporary directory
rm -rf "$TEMP_DIR"

# print info message
echo "----------"
echo "Hunspell dictionary: ${HUNSPELL_DICT_FILE_NAME} installed successfully."

exit 0
