#!/bin/bash
#
# IceWarp Server
# Copyright (c) 2008-2022 IceWarp Ltd. All rights reserved.
#
# http://www.icewarp.com
#
# file: functions.sh - include file with basic functions
#

colorize()
{
    case "$TERM" in
    xterm-256color|xterm|xterm-color)
      GOOD=$'\e[38;5;2m'
      BAD=$'\e[38;5;1m'
      NORMAL=$'\e[0m'
      WARN=$'\e[38;5;11m'
      HILITE=$'\e[40;38;5;15m'
      BRACKET=$'\e[40;38;5;8m'
      ;;
      linux)
      GOOD=$'\e[0;32m'
      BAD=$'\e[0;31m'
      NORMAL=$'\e[0m'
      WARN=$'\e[1;33m'
      HILITE=$'\e[1;37m'
      BRACKET=$'\e[1;30m'
        ;;
      *)
      # https://en.wikipedia.org/wiki/ANSI_escape_code
      GOOD=$'\e[0;32m'
      BAD=$'\e[0;31m'
      NORMAL=$'\e[0m'
      WARN=$'\e[0;33m'
      HILITE=${NORMAL}
      BRACKET=${NORMAL}
    esac
}

good()
{
    echo -e "${GOOD}**${NORMAL}\t$*"
}

bad()
{
    echo -e "${BAD}**${NORMAL}\t$*"
}

verybad()
{
    echo "${HILITE}${BAD}************${HILITE}************${BAD}************${NORMAL}"
    echo "${HILITE}$*"
    echo "${HILITE}${BAD}************${HILITE}************${BAD}************${NORMAL}"
    exit 1
}

warn()
{
    echo -e "${WARN}**${NORMAL}\t$*"
}

hilite()
{
    echo -e "${GOOD}**\t${HILITE}$*${NORMAL}"
}

clear_stdin()
{
    while read -t 0.1; do : ; done
}

getparam()
{
    echo -e -n "${GOOD}**${NORMAL}\t${1} "
    clear_stdin
    read PARAM
}

getparam_with_default()
{
    echo -e -n "${GOOD}**${NORMAL}\t${1}"
    if [ "x${2}" != "x" ]; then
        echo -n " [${2}]"
    fi

    echo -n ": "
    clear_stdin
    read PARAMDATA

    if [ "x${PARAMDATA}" != "x" ]; then
        PARAM="${PARAMDATA}"
    else if [ "x${2}" != "x" ]; then
        PARAM="${2}"
    else
        PARAM=""
    fi fi
}

getpassword()
{
    DIFFER="1"
    while [ "$DIFFER" == "1" ]
    do
        echo -e -n "${GOOD}**${NORMAL}\t${1} "
        clear_stdin
        read -s TOKEN
        good ""
        echo -e -n "${GOOD}**${NORMAL}\tOnce again to confirm: "
        clear_stdin
        read -s TOKEN_CONFIRM
        good ""

        if [ "x$TOKEN" == "x$TOKEN_CONFIRM" ]; then
            PASSWORD_PARAM=$TOKEN
            DIFFER="0"
        else
            bad ""
            bad "Passwords don't match."
            bad "Try again"
            bad ""
        fi
    done
}

# Ask question with confirmation
# Parameters:
# $1 - question
# $2 - first choice, it is choosed if user enters only empty string
# $3 - second choice, $4 third choice...
# Return:
# 0 on error - unknown string entered
# 1 if first choice was selected
# 2 if second choice was selected
# 3 if third choice...
ask_with_confirmation()
{
    if [ -z "$1" ] || [ -z "$2" ] || [ -z "$3" ]; then
        return 0
    fi
    LOWER_CHOICE_1=$(echo "$2" | tr "A-Z" "a-z")
    LOWER_CHOICE_2=$(echo "$3" | tr "A-Z" "a-z")

    # make options string
    local OPTIONS="${GOOD}**${NORMAL}\t${1} ${BRACKET}[${HILITE}${2}${NORMAL}"
    for ARG in "${@:3}"; do 
        OPTIONS+="/${ARG}"
    done
 
    echo -e -n "${OPTIONS}${BRACKET}]${NORMAL}: "
    clear_stdin
    read PARAM
    if [ -z "$PARAM" ]; then 
        return 1
    fi

    PARAM=$(echo "$PARAM" | tr "A-Z" "a-z")
    declare -i I=1
    for ARG in "${@:2}"; do  # arguments 2 through n (i.e. 3 args starting at number 2)
        local LOWER_CHOICE=$(echo "$ARG" | tr "A-Z" "a-z")
        if [ "$PARAM" == "$LOWER_CHOICE" ]; then
            return $I
        fi
        ((I++))
    done

    return 0
}

display_menu()
{
    STR_LENGTH=${#1}
    STR_LINE=$(while [ "$STR_LENGTH" -ne "0" ]; do printf "%s" "-"; let "STR_LENGTH-=1"; done; printf "\n")
    good ""
    hilite "$1"
    hilite "$STR_LINE"
    good ""
    good "You have the following options:"
    good ""
    # LINE and $2 can contain spaces
    for LINE in `cat ${3} | grep "^${2}:"`
    do
        NAME=$(echo "$LINE" | cut -d':' -f2)
        LINECHOICE=$(echo "$LINE" |cut -d':' -f3)
        display_menuitem "$NAME" "$LINECHOICE"
    done
    good ""
    display_menuitem "0" "Return"
    display_menuitem "Q" "Exit"
    good ""
    getparam "Enter your choice:"
}

# This function expects exported variable MENU_SCRIPTS_DIR with menu scripts stored
menu()
{
    IFS_PREV=$IFS
    IFS=$'\n'
    while true
    do
        display_menu "$1" "$2" "$3"
        # quit from menu
        if [ "x$PARAM" == "x0" ]; then
            return
        fi
        # quit from script which uses menu
        if [ "x$PARAM" == "xQ" ] || [ "x$PARAM" == "xq" ]; then
            exit 0
        fi
        for LINE in `cat ${3} | grep "^${2}:"`; do
            LINECHOICE=$(echo "$LINE" | cut -d':' -f2)
            NAME=$(echo "$LINE" | cut -d':' -f3)
            TYPE=$(echo "$LINE" | cut -d':' -f4)
            if [ "x$LINECHOICE" == "x$PARAM" ]; then
                unset PARAM
                case "$TYPE" in
                    "script"|"SCRIPT")
                        good ""
                        SCRIPT=$(echo "$LINE" | cut -d':' -f5)
                        pushd "$IWS_INSTALL_DIR" &>/dev/null
                        export MENU_SCRIPTS_DIR
                        "${MENU_SCRIPTS_DIR}/${SCRIPT}"
                        if [ $? -ne 0 ]; then
                            bad "Error running script!"
                        fi
                        popd &>/dev/null
                        ;;
                    "menu"|"MENU")
                        MENUROOT=$(echo "$LINE" | cut -d':' -f5)
                        if [ "$MENUROOT" != "QUIT" ]; then
                            clear
                            menu "$NAME" "$MENUROOT" "$3"
                            clear
                        fi
                        ;;
                    *)
                        MATCHED="0"
                        ;;
                esac
            fi
        done # for LINE ...
    done # while true
    IFS=$IFS_PREV
}

display_menuitem()
{
    CHOICE=$1
    NAME=$2
    echo -e "${GOOD}**\t${HILITE}${GOOD}[${HILITE}${CHOICE}${GOOD}]${HILITE} ${NAME}${NORMAL}"
}

# params
# 1: column count
# 2: padding
# 3: default choice
# Items in array called MENUGRID_ITEMS
display_menugrid()
{
    COLUMN_COUNT=$1
    PADDING=$2
    
    # for each item
    COLUMN=0
    CHOICE=0
    for IDX in $(seq 0 $((${#MENUGRID_ITEMS[*]}-1))); do
        ITEM="${MENUGRID_ITEMS[${IDX}]}"
        CHOICE=$((${CHOICE} + 1))
        CHOICESTR=$(printf '%2d' ${CHOICE})
        MSG="${HILITE}${GOOD}[${HILITE}${CHOICESTR}${GOOD}]${HILITE} ${ITEM}${NORMAL}"
        SPACELEFT=$((${PADDING} + 26- ${#MSG}))

        if [ ${COLUMN} -eq 0 ]; then
            echo -en "${GOOD}**\t"
        fi
        echo -en "${MSG}"
        for X in $(seq 1 ${SPACELEFT}); do
            echo -n " "
        done
        COLUMN=$((${COLUMN} + 1))
        if [ ${COLUMN} -eq ${COLUMN_COUNT} ]; then
            COLUMN=0
            echo ""
        fi
    done
    if [ ${COLUMN} -ne 0 ]; then
        echo ""
    fi

    good ""
    getparam_with_default "Enter your choice" "${3}"

    if ! $(echo "${PARAM}" | grep '^[0-9]\+$' &>/dev/null); then
        # not a natural number
        PARAM="-2"
    else if [ ${PARAM} -lt 1 -o ${PARAM} -gt ${#MENUGRID_ITEMS[*]} ]; then
        # out of range
        PARAM="-1"
    fi fi
}

run_cmd_log()
{
    # log command
    DATE_TIME=$(date "+%Y-%m-%d %H:%M:%S.%N")
    echo "${DATE_TIME}: \"$1\" \"$2\" \"$3\" \"$4\" \"$5\" \"$6\" \"$7\" \"$8\" \"$9\" \"${10}\" \"${11}\" \"${12}\" \"${13}\" \"${14}\" \"${15}\"" >> "$WIZARD_CMD_LOG"

    export LD_LIBRARY_PATH="${SERVER_LD_LIBRARY_PATH}:${LD_LIBRARY_PATH}"
    CMD_STDOUT=$("$1" "$2" "$3" "$4" "$5" "$6" "$7" "$8" "$9" "${10}" "${11}" "${12}" "${13}" "${14}" "${15}")
    CMD_RET=$?

    # log command output
    DATE_TIME=$(date "+%Y-%m-%d %H:%M:%S.%N")
    echo "${DATE_TIME}: Command output:" >> "$WIZARD_CMD_LOG"

    DATE_TIME=$(date "+%Y-%m-%d %H:%M:%S.%N")
    echo "${DATE_TIME}: ${CMD_STDOUT}" >> "$WIZARD_CMD_LOG"

    # log command result
    DATE_TIME=$(date "+%Y-%m-%d %H:%M:%S.%N")
    echo "${DATE_TIME}: Result: ${CMD_RET}" >> "$WIZARD_CMD_LOG"

    return $CMD_RET
}

run_cmd_print_stdout_log()
{
    CMD_STDOUT=""
    CMD_RET="0"
    run_cmd_log $@
    echo "$CMD_STDOUT"
    return $CMD_RET
}

# calls tool and returns single value of single api variable
# IWS_INSTALL_DIR have to be set
# params: object variable
get_api_variable()
{
    CMD_STDOUT=$("${IWS_INSTALL_DIR}/tool.sh" get "$1" "$2" | sed 's/^[^:]*: //')
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

# calls tool and sets single value of single api variable
# IWS_INSTALL_DIR have to be set
# params: object variable value
set_api_variable()
{
    CMD_STDOUT=$("${IWS_INSTALL_DIR}/tool.sh" set "$1" "$2" "$3")
    CMD_RET=$?
    echo "${CMD_STDOUT}"
    return ${CMD_RET}
}

# kills process with CWD in specified folder
# $1 process name
# $2 folder where must point process CWD to proceed with kill

kill_proc_by_name()
{
    PROCNAME_TO_KILL=$1
    PROCNAME_TO_KILL_PATH=$2

    PIDS=$(ps -ef | grep $PROCNAME_TO_KILL | awk '{print $2}')
    for PID in $PIDS
    do
            CWD=$(readlink -e /proc/$PID/cwd)
            
            # Check if proces CWD is in desired folder
            if [ "$CWD" == "$PROCNAME_TO_KILL_PATH" ]; then
                kill -9 $PID
            fi
    done
}

colorize
