#!/bin/bash
#
# kavehost        Startup script for the Kavehost
#
# processname: kavehost
# pidfile: ${IWS_INSTALL_DIR}/var/kavehost.pid

# include configuration file
source /etc/icewarp/icewarp.conf
if [ $? -ne 0 ]; then
    echo "Can not find configuration file icewarp.conf"
    exit 255
fi

RETVAL=0

CONFIG="${IWS_INSTALL_DIR}/kaspersky/kavehost.xml"

# Check config file
if [ ! -f "${CONFIG}" ]
then
    echo -n "Config file [${CONFIG}] does not exist."
    echo
    exit 1
fi

# Kavehost binary name
prog="kavehost"

# Home directory of the kavehost binary.
kavehost_dir="${IWS_INSTALL_DIR}/kaspersky"

# Full path to the kavehost binary
kavehost="${kavehost_dir}/${prog}"

# Extract pid file
pidfile=`grep PidPath "${CONFIG}" | sed 's/^.*<PidPath>//' | sed 's/<\/PidPath>//'`

if [ "x${pidfile}" = "x" ]
then
    echo -n "Config file [${CONFIG}] does not contain or contains empty PidPath tag."
    echo
    exit 1
fi

process_exist()
{
    kill -0 $1 2>/dev/null
    return $?
}

MOVE_TO_COL="echo -n               "
echo_success() { ${MOVE_TO_COL} ; echo -n ' [  OK  ] ' ; }
echo_failure() { ${MOVE_TO_COL} ; echo -n ' [FAILED] ' ; }
echo_warning() { ${MOVE_TO_COL} ; echo -n ' [WARNING] ' ; }

if [ "x${KL_PLUGINS_PATH}" = "x" ]
then
    KL_PLUGINS_PATH="${HOME}"/builds/`uname -s`-x86/bin
    export KL_PLUGINS_PATH
    LD_LIBRARY_PATH="${KL_PLUGINS_PATH}:${LD_LIBRARY_PATH}"
    export LD_LIBRARY_PATH
fi

start() {
    echo -n "Starting ${prog}: "
    if [ -e "${pidfile}" ]
    then
        PID=`cat ${pidfile}`
        if process_exist "${PID}"
        then
            echo -n "kavehost is already running."
            echo_failure
            echo
            return 0
        fi
    fi

    cd "${kavehost_dir}"
    "${kavehost}" -c "${CONFIG}"
    RC=$?

    if [ "${RC}" -eq 0 ]
    then
        # modify rights of pidfile
        chmod a+rw "${pidfile}"
        echo_success
    else
        echo_failure
    fi

    echo
    return ${RC}
}

stop() {
    echo -n "Stopping ${prog}: "
    if [ ! -e "${pidfile}" ]
    then
        echo -n "kavehost is not running."
        echo_failure
        echo
        return 0
    fi

    PID=`cat ${pidfile}`
    if ! process_exist "${PID}"
    then
        echo -n "kavehost is not running."
        echo_failure
        echo
        return 0
    fi

    seconds_to_wait=10

    kill "${PID}"
    RC=$?
    
    if [ "${RC}" -eq 0 ]
    then
        while process_exist "${PID}"
        do
            if [ "${seconds_to_wait}" -eq 0 ]
            then
                break
            fi

            seconds_to_wait=$((seconds_to_wait-1))
            
            sleep 1
        done
        
        if  [ "${seconds_to_wait}" -eq 0 ]
        then
            echo -n "stopping kavehost timed out, terminating by force"
            kill -9 "${PID}"
            killall -9 kavscanner
        else
            echo_success
        fi
    else
        echo_failure
    fi

    echo
    return ${RC}
}

reloaddb() {
    echo -n "Reload databases ${prog}: "
    if [ ! -e "${pidfile}" ]
    then
        echo -n "kavehost is not running."
        echo_failure
        echo
        return 1
    fi

    PID=`cat ${pidfile}`
    if ! process_exist "${PID}"
    then
        echo -n "kavehost is not running."
        echo_failure
        echo
        return 1
    fi

    kill -USR1 "${PID}"
    RC=$?

    if [ "${RC}" -eq 0 ]
    then
        echo_success
    else
        echo_failure
    fi

    echo
    return ${RC}
}

status() {
    if [ ! -e "${pidfile}" ]
    then
        echo -n "kavehost is not running."
        echo
        return 1
    fi

    PID=`cat ${pidfile}`
    if ! process_exist "${PID}"
    then
        echo -n "kavehost dead but pid file exists."
        echo
        return 1
    fi

    echo "kavehost (pid ${PID}) is running..."
    return 0
}

# See how we were called.
case "$1" in
  start)
    start
    RETVAL=$?
    ;;
  stop)
    stop
    RETVAL=$?
    ;;
  reloaddb)
    reloaddb
    RETVAL=$?
    ;;
  status)
    status
    RETVAL=$?
    ;;
  restart)
    stop 
    start
    RETVAL=$?
    ;;
  *)
    echo "Usage: ${prog} {start|stop|restart|reloaddb|status}"
    RETVAL=1
    ;;
esac

exit $RETVAL
