<?php
	 

namespace tools;

use error\InvalidSearchExpressionException;

class SearchParserYoda
{
	static public $_allowedKeywords = [
		'from',
		'to',
		'subject',
		'after',
		'before',
		'has',
		'fulltext',
		 		'tag',
		'taglist',
		'with',
		'yoda',
	];

	static public $_keywordTranslations = [
		'after' => 'SINCE',
		'has' => 'HEADER',
		'fulltext' => 'TEXT',
		'tag' => 'KEYWORD',
		'taglist' => 'KEYWORD',
		'with' => 'X-WITH',
		'yoda' => 'X-YODA',
	];

	static public $_keywordValuesTranslations = [
		'is' => ['unread' => 'UNSEEN'],
	];


	 
	static public function parseExpression($searchExpression)
	{
		$result = [];

		$length = strlen($searchExpression);
		$first = 0;
		$offset = 0;

		$escape = false;
		$quotes = false;
		$term = false;

		while ($offset < $length)
		{
			$ch = $searchExpression[$offset];
			if ($ch <= ' ')
			{
				if (!$quotes && $term)
				{
					$result[] = self::parseTerm(substr($searchExpression, $first, $offset - $first));
					$term = false;
				}

				$escape = false;
			}
			else
			{
				if ($ch == '\\')
				{
					$escape = !$escape;
				}
				else
				{
					if ($ch == '"')
					{
						if (!$escape)
							$quotes = !$quotes;
					}

					$escape = false;
				}

				if (!$term)
				{
					$first = $offset;
					$term = true;
				}
			}

			$offset++;
		}

		if ($term)
			$result[] = self::parseTerm(substr($searchExpression, $first, $offset - $first));

		return $result;
}


	 
	static private function parseTerm($term)
	{
		if (!preg_match_all('/^(?P<operator>[+\-?])?(?P<keyword>[a-zA-Z]+:)?(?P<value>(".+")|([^\s+\-?:"{}]+))(?P<metadata>{[^\s"]+})?$/u', $term, $matches, PREG_SET_ORDER))
			throw new InvalidSearchExpressionException("Invalid term found: '".$term."'");

		$match = $matches[0];

		$value = $match['value'];
		if ($value[0] == '"')
		{
			$escape = false;
			$length = strlen($value) - 1;
			for ($i = 1; $i < $length; $i++)
			{
				if (!$escape && $value[$i] == '"')
				throw new InvalidSearchExpressionException("Invalid value found: '".$value."' in term '".$term."'");

				if ($value[$i] == '\\')
					$escape = !$escape;
				else
					$escape = false;
			}

			if ($escape)
				throw new InvalidSearchExpressionException("Invalid value found: '".$value."' in term '".$term."'");
		}

		$result = [];
		$result['operator'] = $match['operator'];
		$result['keyword'] = !empty($match['keyword']) ? substr($match['keyword'], 0, strlen($match['keyword']) - 1) : '';
		$result['value'] = $match['value'];
		$result['metadata'] = $match['metadata'] ?? '';

		return $result;
	}


	 
	static public function getImapCriteria($searchExpression, $defaultKeyword = 'fulltext')
	{
		return self::getImapCriteriaFromParsedData(self::parseExpression($searchExpression), $defaultKeyword);
	}


	 
	static public function getImapCriteriaFromParsedData($parsedData, $defaultKeyword = 'fulltext')
	{
		$result = '';

		if (!count($parsedData))
			return $result;

		foreach ($parsedData as $term)
		{
			$result .= $result ? ' ' : '';
			$result .= ($term['operator'] == '-' ? 'NOT ' : '').($term['operator'] == '?' ? 'X-OPT ' : '');

			$keyword = strtolower($term['keyword']);
			$value = strtolower($term['value']);
			if (!$keyword || (!in_array($keyword, self::$_allowedKeywords) && !self::$_keywordValuesTranslations[$keyword][$value]))
				$keyword = $defaultKeyword;

			$keywordValue = self::$_keywordValuesTranslations[$keyword][$value];
			$keyword = self::$_keywordTranslations[$keyword] ?? $keywordValue ?? strtoupper($keyword);
			$result .= $keyword.(!$keywordValue ? (' '.$term['value']) : '');
		}

		return $result;
	}
}
?>