/*																	*
 * This file is a part of an example of how to communicate			*
 * with IceWarp Unified Messaging server using XML-RPC				*
 *																	*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.				*
 *																	*
 * IceWarp Ltd 2011													*
 *																	*/

using System;
using System.Collections.Generic;
using System.Reflection;
using System.Linq;
using System.Text;

// Icewarp
using Icewarp.XmlRpcExample.XmlRpc.Base;
using Icewarp.XmlRpcExample.Utils;

namespace Icewarp.XmlRpcExample.XmlRpc
{
	/// <summary>
	/// Class for intercepting Merak's Groupware
	/// </summary>
	public class IceWarpGroupware : IceWarpXmlRpc, IDisposable
	{
		#region Nested types
		public enum FolderType
		{
			Contact,
			Calendar,
			Task,
			Mail,
		}

		public enum FormatType
		{
			Eas,
			Versit,
		}

		public enum CreateFolderResult
		{
			Failed,
			AlreadyExists,
			Created,
		}
		#endregion // Nested types

		#region Fields
		private string m_superUser;
		private string m_superPassword;
		private bool m_connected;
		private bool m_isDisposed;

		#region Presets
		private Logger.Helper m_log = new Logger.Helper("IceWarpGroupware");
		#endregion // Presets
		#endregion // Fields

		#region Ctor
		/// <summary>
		/// Ctor
		/// </summary>
		/// <param name="credentials">Login credentials with administrator account on Icewarp</param>
		public IceWarpGroupware(LoginCredentials credentials)
			: base(credentials)
		{
			CreateGroupwareComConnection();
		}
		#endregion // Ctor

		#region Public methods
		/// <summary>
		/// Invokes login user
		/// </summary>
		/// <param name="user">Username of the administrator account</param>
		/// <param name="password">Passwotd of the administrator account</param>
		/// <returns>True if login was successful, false otherwise</returns>
		public bool LoginUser(string user, string password)
		{			
			if ((m_sid = MethodInvoker("FunctionCall", new string[] { "loginuser", user, password, "", "" }) as string) != null)
			{
				m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Logged as {0}", user));

				m_connected = true;
			}
			else
			{
				m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Unable to login as {0}", user));
			}

			return m_connected;
		}

		/// <summary>
		/// Substitutes for the specified user (masked connection), connection will
		/// behave exactly as logged as a substituted user.
		/// </summary>
		/// <param name="user">Email address of the substituted user</param>
		public void SubstituteUser(string user)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Substituting as user {0}", user));

			MethodInvoker("FunctionCall", new string[] { "substituteuser", m_sid, user, "", "" });
		}

		/// <summary>
		/// Obtains superuser credentials to access groupware.
		/// No user is able to obtain superuser credentials except 
		/// the administrator of the server.
		/// </summary>
		/// <returns>True if the superuser credentials were obtained</returns>
		public bool GetSuperuser()
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, "Getting superuser credentials");

			IceWarpApi api = new IceWarpApi(m_credentials);
			m_superUser = api.GetProperty("C_GW_SuperUser");
			m_superPassword = api.GetProperty("C_GW_SuperPass");

			return ((!String.IsNullOrEmpty(m_superUser)) && (!String.IsNullOrEmpty(m_superPassword)));
		}		

		/// <summary>
		/// Opens all groups, uses wildcard "*"
		/// </summary>
		/// <returns>Group identification</returns>
		public string OpenGroup()
		{
			return OpenGroup("*");
		} 

		/// <summary>
		/// Opens specified group
		/// </summary>
		/// <param name="wildcard">Group to open</param>
		/// <returns>Group identification</returns>
		public string OpenGroup(string wildcard)
		{
			return MethodInvoker("FunctionCall", new string[] { "opengroup", m_sid, wildcard, "", "" }) as string;
		}

		/// <summary>
		/// Adds attachment to the groupware item
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Groupware item identification</param>
		/// <param name="parameters">AttName=name.txt&AttType=F&AttDesc=description&AttSize=size</param>
		/// <param name="value">Empty string</param>
		/// <returns>Returns unique name of the attachment</returns>
		public string AddAttachment(string gid, string oid, string parameters, string value)
		{
			return MethodInvokerBase64(new string[] { "AddAttachment", gid, oid, parameters, value });
		}

		/// <summary>
		/// Locks attachment, must be called before the operation with the attachment
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Groupware item identification</param>
		/// <param name="attName">Unique name of the attachment</param>
		/// <returns>Path to the file where the attachment shall be stored</returns>
		public string LockAttachment(string gid, string oid, string attName)
		{			
			return MethodInvokerBase64(new string[] { "GetAttachmentPath", gid, oid, attName, "LOCK" });
		}

		/// <summary>
		/// Unlocks attachment
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Groupware item identification</param>
		/// <param name="attName">Unique name of the attachment</param>
		/// <returns>Returns true if attachment is successfully unlocked, false otherwise</returns>
		public bool UnlockAttachment(string gid, string oid, string attName)
		{
			string result = MethodInvokerBase64(new string[] { "GetAttachmentPath", gid, oid, attName, "UNLOCK" });

			return (!String.IsNullOrEmpty(result) && (result.Equals("1")));
		}

		/// <summary>
		/// Opens specified folder
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="name">Name of the folder</param>
		/// <returns>Unique folder identification</returns>
		public string OpenFolder(string gid, string name)
		{
			return MethodInvokerBase64(new string[] { "openfolder", gid, name, "", "" });
		}

		/// <summary>
		/// Adds vcard to the calendar (appointment)
		/// </summary>
		/// <param name="fid">Folder identification</param>
		/// <param name="vcard">EAS format</param>
		/// <param name="format">EAS for EAS format</param>
		/// <returns>Unique item identification</returns>
		public string AddVCalendar(string fid, string vcard, string format)
		{
			return MethodInvokerBase64(new string[] { "addvcalendar", fid, vcard, "", format });
		}

		/// <summary>
		/// Adds vcard to the calendar (appointment)
		/// </summary>
		/// <param name="fid">Folder identification</param>
		/// <param name="vcard">EAS format</param>
		/// <returns>Unique item identification</returns>
		public string AddVCalendar(string fid, string vcard)
		{
			return AddVCalendar(fid, vcard, String.Empty);
		}

		/// <summary>
		/// Deletes item from groupware database
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Item identification</param>
		/// <returns>True if item was successfully delete, false otherwise</returns>
		public bool DeleteItem(string gid, string oid)
		{
			string retVal = MethodInvoker("FunctionCall", new string[] { "DeleteEvent", gid, oid, "", "" });

			return (!String.IsNullOrEmpty(retVal) && (retVal.Equals("1")));
		}

		/// <summary>
		/// GWApi.FunctionCall('AddFolderRight',gid,FolderName,FolderRights.FindKey,IntToStr(FolderRight-1));
		/// </summary>
		/// <returns>True if setting was successful, false otherwise</returns>
		public bool AddFolderRights(string gid, string folderName, string mailbox, string rights)
		{			
			string retVal = MethodInvokerBase64(new string[] { "AddFolderRight", gid, folderName, mailbox, rights });

			return (!String.IsNullOrEmpty(retVal));
		}

		/// <summary>
		/// Adds vcard (contact)
		/// </summary>
		/// <param name="fid">Folder identification</param>
		/// <param name="vcard">Vcard in EAS</param>
		/// <returns>Unique item identification</returns>
		public string AddVCard(string fid, string vcard)
		{
			return MethodInvokerBase64(new string[] { "addvcard", fid, vcard, "", "" });			
		}

		/// <summary>
		/// Creates contact folder
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="fullName">Name of the folder</param>
		/// <param name="folderType">Determines type of the folder</param>
		/// <returns>If empty then call failed, if 1 folder was created, if 0 folder already exists</returns>
		public CreateFolderResult CreateFolder(string gid, string fullName, FolderType folderType)
		{			
			string retVal = "";
			CreateFolderResult result;

			switch (folderType)
			{
				case FolderType.Contact:
					retVal = CreateFolder(gid, fullName, "C");
					break;
				case FolderType.Calendar:
					retVal = CreateFolder(gid, fullName, "E");
					break;
				case FolderType.Mail:
					retVal = CreateFolder(gid, fullName, "M");
					break;
				case FolderType.Task:
					retVal = CreateFolder(gid, fullName, "T");
					break;
			}

			switch (retVal)
			{
				case "0":
					result = CreateFolderResult.AlreadyExists;
					break;
				case "1":
					result = CreateFolderResult.Created;
					break;
				case "":
				default:
					result = CreateFolderResult.Failed;
					break;
			}

			return result;
		}		

		/// <summary>
		/// Deletes groupware folder
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="fullName">Name of the folder</param>
		/// <returns>True if delete was successful, false otherwise</returns>
		public bool DeleteFolder(string gid, string fullName)
		{
			string retVal = MethodInvokerBase64(new string[] { "DeleteFolder", gid, fullName, "", "" });

			return (!String.IsNullOrEmpty(retVal) && (retVal.Equals("True")));
		}

		/// <summary>
		/// Renames groupware folder
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oldName">Current folder name</param>
		/// <param name="newName">New folder name</param>
		/// <returns></returns>
		public string RenameFolder(string gid, string oldName, string newName)
		{
			return MethodInvokerBase64(new string[] { "RenameFolder", gid, oldName, newName, "" }); 
		}

		/// <summary>
		/// Adds certificate to the existing contact
		/// </summary>
		/// <param name="fid">Folder Id</param>
		/// <param name="contactId">Item id</param>
		/// <param name="certificate">Certificate in PEM format</param>
		/// <returns></returns>
		public string AddCertificate(string fid, string contactId, string certificate)
		{
			return MethodInvokerBase64(new string[] { "setcontactcertificate", fid, contactId, certificate, "" });
		}

		/// <summary>
		/// Logging as superuser, does not substitute to specific user
		/// </summary>
		/// <returns>True if successfully logged, false otherwise</returns>
		public bool LoginAsSuperuser()
		{
			return LoginAsSuperuser(null);
		}

		/// <summary>
		/// Logout
		/// </summary>
		public void Logout()
		{
			m_connected = false;

			MethodInvoker("FunctionCall", new string[] { "logout", "", "", "", "" });

			m_sid = null;
		}

		/// <summary>
		/// Login as superuser
		/// </summary>
		/// <param name="substitutedUser"></param>
		/// <returns></returns>
		public bool LoginAsSuperuser(string substitutedUser)
		{
			bool result = false;

			if (GetSuperuser())
			{
				if (result = LoginUser(m_superUser, m_superPassword))
				{
					if (!String.IsNullOrEmpty(substitutedUser))
					{
						SubstituteUser(substitutedUser);
					}
				}
			}

			return result;
		}

		/// <summary>
		/// Folder will be selected as default on the server.
		/// Requires call for OpenGroup with specified name.
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <returns>True if folder was set, false otherwise</returns>
        public bool SetDefaultFolder(string gid)
        {
            bool result = false;

            string retVal = MethodInvoker("FunctionCall", new string[] { "SetDefaultFolder", gid, "", "", "" });
            if (result = !String.IsNullOrEmpty(retVal))
            {
                result = retVal.Equals("True");
            }

            return result;
        }

		/// <summary>
		/// Returns appointment in specified format
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Item identification</param>
		/// <param name="formatType">Determines output format</param>
		/// <returns>VCalendar item in specified format</returns>		 
		public string GetVCalendar(string gid, string oid, FormatType formatType)
		{
			string retVal = null;

			switch (formatType)
			{
				case FormatType.Eas:
					retVal = MethodInvoker("FunctionCall", new string[] { "getvcalendar", gid, oid, "EAS", "" });
					break;
				case FormatType.Versit:
					retVal = MethodInvoker("FunctionCall", new string[] { "getvcalendar", gid, oid, "", "" });
					break;
			}

			return retVal;
		}

		/// <summary>
		/// Returnd contact in specified format
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="oid">Item identification</param>
		/// <param name="formatType">Determines output format</param>
		/// <returns>VCard item in specified format</returns>	
		public string GetVCard(string gid, string oid, FormatType formatType)
		{
			string retVal = null;

			switch (formatType)
			{
				case FormatType.Eas:
					retVal = MethodInvoker("FunctionCall", new string[] { "getvcard", gid, oid, "EAS", "" });
					break;
				case FormatType.Versit:
					retVal = MethodInvoker("FunctionCall", new string[] { "getvcard", gid, oid, "", "" });
					break;
			}

			return retVal;
		}

		#endregion // Public methods

		#region Private methods
		/// <summary>
		/// Creates object interface connection
		/// </summary>
		/// <returns>True if connection has been established, false otherwise</returns>
		private bool CreateGroupwareComConnection()
		{
			return CreateComConnection("IceWarpServer.Groupware");
		}

		/// <summary>
		/// Creates calendar folder
		/// </summary>
		/// <param name="gid">Group identification</param>
		/// <param name="fullName">Folder name</param>
		/// <param name="folderType">Folder type</param>
		/// <returns>Folder identification</returns>
		/// <seealso cref="CreateFolder"/>
		private string CreateFolder(string gid, string fullName, string folderType)
		{
			return MethodInvokerBase64(new string[] { "AddFolder", gid, fullName, folderType, "" });
		}
		#endregion // Private methods

		#region IDisposable Members

		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		protected virtual void Dispose(bool disposing)
		{
			// If you need thread safety, use a lock around these 
			// operations, as well as in your methods that use the resource.
			if (!m_isDisposed)
			{
				if (disposing)
				{
					if (m_connected)
					{
						Logout();
					}
				}

				// Indicate that the instance has been disposed.
				m_isDisposed = true;
			}
		}

		#endregion // IDisposable Members
	}
}
