/*																	*
 * This file is a part of an example of how to communicate			*
 * with IceWarp Unified Messaging server using XML-RPC				*
 *																	*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.				*
 *																	*
 * IceWarp Ltd 2011													*
 *																	*/

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Runtime.Remoting;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Linq;
using System.Text;
using System.Net;
using System.IO;
using System.Xml;

// Icewarp
using Icewarp.XmlRpcExample.Utils;

namespace Icewarp.XmlRpcExample.XmlRpc.Base
{
	/// <summary>
	/// Base class for all COM interceptors
	/// </summary>
	public abstract class IceWarpXmlRpc
	{		
		#region Fields
		protected string m_sid;
		protected object m_comObjHandle;
		protected Type m_comAppType;
        protected LoginCredentials m_credentials;	

		// remote connection
		protected string m_connection;
		protected string m_comToken;
		protected string m_comTokenHandle;
		protected bool m_fullAddress;

		#region Presets
		private static Regex RgxValueParser = new Regex(@"^.+?<value>(?<VALUE>.*?)</value>.+?$", RegexOptions.IgnoreCase | RegexOptions.Singleline);
		private Logger.Helper m_log = new Logger.Helper("IceWarpCom");
		#endregion // Presets
		#endregion // Fields

		#region Properties
		/// <summary>
		/// Returns true if COM interface is connected to session,
		/// false otherwise.
		/// </summary>
		public bool IsConnected
		{
			get
			{
				return (m_sid != null);
			}
		}

		/// <summary>
		/// Returns true if COM object is created, false
		/// otherwise.
		/// </summary>
		public bool IsCreated
		{
			get
			{
				return (m_connection != null);
			}
		}
		#endregion // Properties

		#region Ctor
		/// <summary>
		/// Ctor
		/// </summary>
		/// <param name="credentials">Login credentials</param>
		public IceWarpXmlRpc(LoginCredentials credentials)
		{
			m_credentials = credentials;
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Creating COM {0}", credentials.Username));
		}
		#endregion // Ctor

		#region Protected methods
		/// <summary>
		/// Creates communication object interface connection
		/// </summary>
		/// <param name="comObject">Name of the object</param>
		/// <returns>True if connection has been established, false otherwise</returns>
		protected bool CreateComConnection(string comObject)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Creating direct object {0}", comObject));

			m_connection = SendRpc("0->Create", comObject);

			return (m_connection != null);
		}

        /// <summary>
        /// Invokes method using XML-RPC call.
        /// </summary>
        /// <param name="functionName">Method name for invocation</param>
        /// <param name="parameters">Method parameters</param>
        /// <returns>Server response</returns>
		protected string MethodInvoker(string functionName, string[] parameters)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Invoking method {0}", functionName));

			string retVal = null;

			if (!String.IsNullOrEmpty(m_connection))
			{
				retVal = SendRpc(String.Format("{0}->{1}", m_connection, functionName), parameters);
			}
			else
			{
				m_log.WriteLog(Logger.LogLevel.Error, "Connection is NULL!");
			}

			return retVal;
		}

		protected string SimpleMethodInvoker(string functionName, string[] parameters)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Invoking method {0}", functionName));

			return SendRpc(functionName, parameters);
		}

		/// <summary>
		/// Invokes method using XML-RPC call.
		/// </summary>
		/// <param name="functionName">Method name for invocation</param>
		/// <param name="parameters">Method parameters</param>
		/// <returns>Server response</returns>
		protected string MultiMethodInvoker(List<MultiXmlQuery> queries)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Invoking multicall"));

			return SendRpc(queries);
		}

        /// <summary>
        /// Invokes method using XML-RPC, but all input arguments
        /// are encoded in base64
        /// </summary>
        /// <param name="parameters">Input parameters</param>
        /// <returns>Server response</returns>
		protected string MethodInvokerBase64(string[] parameters)
		{
            string retVal = null;

            if ((parameters != null) && (parameters.Length > 0))
            {
				m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Invoking method {0}", parameters[0]));

                string[] base64params = new string[parameters.Length];

                base64params[0] = parameters[0];
                for (int i = 1; i < parameters.Length; i++)
                {
                    base64params[i] = Convert.ToBase64String(Encoding.UTF8.GetBytes(parameters[i]));
                }

                retVal = MethodInvoker("FunctionCallBase64", base64params);

				Logger.Instance.WriteLog(String.Format("RAW result: {0}", retVal));

                if (!String.IsNullOrEmpty(retVal) && retVal != "0")
                {
                    retVal = Encoding.UTF8.GetString(Convert.FromBase64String(retVal));
                }
            }
            else
            {
				m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Empty invokes"));
            }

            return retVal;
		}
		#endregion // Protected methods

		#region Private methods
		/// <summary>
		/// Parses XML RPC response
		/// </summary>
		/// <param name="value">Incoming response</param>
		/// <returns>Parsed response </returns>
		private string GetRpcValue(string value)
		{
			string retVal = null;

			Match match = RgxValueParser.Match(value);
			if (match.Success)
			{
				retVal = match.Groups["VALUE"].Value;
			}

			return retVal;
		}

		//<methodResponse>
		//<params>
		//<param>
		//<value>
		//<array>
		//<data>
		
		//<value>
		//<array>
		//<data>
		//<value>a18a6f0c57ac361a78f9704d182fe67c</value>
		//</data>
		//</array>
		//</value>

		//<value>
		//<array>
		//<data>
		//<value>gwsuperuser</value>
		//</data>
		//</array>
		//</value>

		//</data>
		//</array>
		//</value>
		//</param>
		//</params>
		//</methodResponse>
		private void ParseMultiResponse(string response, ref List<MultiXmlQuery> queries)
		{
			XmlDocument doc = new XmlDocument();
			doc.Load(new StringReader(response));
			
			XmlNode root = doc.DocumentElement;
			XmlNodeList nodeList = root.SelectNodes("params/param/value/array/data/value");
		
			int i = 0;
			foreach (XmlNode node in nodeList)
			{
				string xml = node.InnerXml;
				XmlNodeList innerNodes = node.SelectNodes("array/data/value");
				foreach (XmlNode innerNode in innerNodes)
				{
					queries[i].Responses.Add(innerNode.InnerXml);
				}

				i++;
			}
		}

		/// <summary>
		/// Creates simple XML RPC request
		/// </summary>
		/// <param name="methodName">Name of the method that will be invoked (case insensitive)</param>
		/// <param name="parameters">Parameters of the method (usually five)</param>
		/// <returns>XML RPC</returns>
		private string PrepareXmlRpc(string methodName, params string[] parameters)
		{
            SimpleXmlWriter xmlWriter = new SimpleXmlWriter();

			xmlWriter.OpenElement("methodCall");
			xmlWriter.AddElement("methodName", methodName);
			xmlWriter.OpenElement("params");

			if (parameters != null)
			{
				foreach (string parameter in parameters)
				{
					xmlWriter.OpenElement("param");
					xmlWriter.AddElement("value", parameter);
					xmlWriter.CloseElement();
				}
			}

			xmlWriter.CloseElement();
			xmlWriter.CloseElement();

			return xmlWriter.GetText();
		}

		public class MultiXmlQuery
		{
			#region Fields
			public string m_methodName;
			public string m_parentObject;
			public List<string> m_parameters = new List<string>();
			public List<string> m_responses = new List<string>();
			#endregion // Fields

			#region Properties
			public string MethodName 
			{ 
				get 
				{ 
					return m_methodName;
				}
			}

			public List<string> Parameters
			{
				get
				{
					return m_parameters;
				}
			}

			public List<string> Responses
			{
				get
				{
					return m_responses;
				}
			}

			public string ParentObject
			{
				get
				{
					return m_parentObject;
				}
			}
			#endregion // Properties

			#region Ctor
			/// <summary>
			/// Ctor
			/// </summary>
			/// <param name="methodName"></param>
			/// <param name="parentObject"></param>
			/// <param name="parameters"></param>
			public MultiXmlQuery(string methodName, string parentObject, List<string> parameters)
			{
				m_methodName = methodName;
				m_parentObject = parentObject;
				m_parameters = parameters;
			}
			#endregion // Ctor
		}

		/// <summary>
		/// Creates simple XML RCP request
		/// </summary>
		/// <param name="methodName">Name of the method that will be invoked (case insensitive)</param>
		/// <param name="parameters">Parameters of the method (usually five)</param>
		/// <returns>XML RPC</returns>
		private string PrepareMultiXmlRpc(List<MultiXmlQuery> queries)
		{
			SimpleXmlWriter xmlWriter = new SimpleXmlWriter();

			xmlWriter.OpenElement("methodCall");
			{
				xmlWriter.AddElement("methodName", "system.multicall");
				xmlWriter.OpenElement("params");				
				xmlWriter.OpenElement("param");
				xmlWriter.OpenElement("value");
				xmlWriter.OpenElement("array");
				xmlWriter.OpenElement("data");
				{
					foreach (MultiXmlQuery query in queries)
					{
						xmlWriter.OpenElement("value");
						xmlWriter.OpenElement("struct");
						{
							xmlWriter.OpenElement("member");
							{
								xmlWriter.AddElement("name", "methodName");
								xmlWriter.OpenElement("value");
								{
									xmlWriter.AddElement("string", String.Format("{0}->{1}", query.ParentObject, query.MethodName));
								}
								xmlWriter.CloseElement(); // value
							}
							xmlWriter.CloseElement(); // member

							xmlWriter.OpenElement("member");
							{
								xmlWriter.AddElement("name", "params");

								xmlWriter.OpenElement("value");								
								xmlWriter.OpenElement("array");									
								xmlWriter.OpenElement("data");
								{
									foreach (string param in query.Parameters)
									{
										xmlWriter.AddElement("value", param);
									}
								}
								xmlWriter.CloseElement(); // data
								xmlWriter.CloseElement(); // array
								xmlWriter.CloseElement(); // value
							}
							xmlWriter.CloseElement(); // member
						}
						xmlWriter.CloseElement(); // value
						xmlWriter.CloseElement(); // struct
					}
				}
				xmlWriter.CloseElement();
				xmlWriter.CloseElement();
				xmlWriter.CloseElement();
				xmlWriter.CloseElement(); // param
				xmlWriter.CloseElement(); // params
			}
			xmlWriter.CloseElement(); // methodCall

			return xmlWriter.GetText();
		}	
		#endregion // Private methods

		#region Public methods
		/// <summary>
		/// Sends RPC to IveWarp Unified server, results in WebRequest
		/// </summary>
		/// <param name="methodName">Name of the method that will be invoked (case insensitive)</param>
		/// <param name="parameters">Parameters of the method (usually five)</param>
		/// <returns>Response of the server</returns>
		public string SendRpc(string methodName, params string[] parameters)
		{
			m_log.WriteLog(Logger.LogLevel.HighPriority, String.Format("Sending RPC {0}", methodName));

			return SendRpc(PrepareXmlRpc(methodName, parameters), true, null);
		}

		public string SendRpc(List<MultiXmlQuery> queries)
		{
			return SendRpc(PrepareMultiXmlRpc(queries), false, queries);
		}

		/// <summary>
		/// Sends RPC to IveWarp Unified server, results in WebRequest
		/// </summary>
		/// <param name="methodName">Name of the method that will be invoked (case insensitive)</param>
		/// <param name="parameters">Parameters of the method (usually five)</param>
		/// <returns>Response of the server</returns>
		public string SendRpc(string rpcQuery, bool singleCall, List<MultiXmlQuery> queries)
		{
			string retVal;			

			try
			{
				Logger.Instance.WriteLog(rpcQuery);

				string address = String.Format("http://{0}:{1}/rpc/", m_credentials.Hostname, m_credentials.ControlPort);
				WebRequest request = HttpWebRequest.Create(address);				
				request.Method = "POST";
				request.CachePolicy = new System.Net.Cache.RequestCachePolicy(System.Net.Cache.RequestCacheLevel.BypassCache);

				byte[] byteArray;
				if ((byteArray = Encoding.UTF8.GetBytes(rpcQuery)) != null)
				{
					request.ContentLength = byteArray.Length;
					Stream dataStream;
					if ((dataStream = request.GetRequestStream()) != null)
					{
						dataStream.Write(byteArray, 0, byteArray.Length);
						dataStream.Close(); 
					}
				}

				request.ContentType = "text/xml";				
                request.Credentials = new NetworkCredential(m_credentials.Username, m_credentials.Password);

	 			WebResponse response = request.GetResponse();
				Stream respStream;
				if ((respStream = response.GetResponseStream()) != null)
				{
					StreamReader reader = new StreamReader(respStream);
					if (singleCall)
					{
						retVal = GetRpcValue(reader.ReadToEnd());
					}
					else
					{
						ParseMultiResponse(reader.ReadToEnd(), ref queries);
						retVal = null;
					}
					respStream.Close();
				}
				else
				{
					retVal = null;
				}
			}
			catch (Exception ex)
			{
				m_log.WriteLog(Logger.LogLevel.Error, ex.Message);
				retVal = null;
			}

			return retVal;
		}
		#endregion // Public methods
	}
}
 